/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Collections 
{
  using Nemerle.Assertions;
  
  using SC = System.Collections;

  /**
   * A functional type-safe wrapper for the System.Collections.Hashtable class.   
   */
  [System.Serializable]
  public class Hashtable ['a,'b] : IDictionary ['a,'b]
  {
    /* -- PUBLIC CONSTRUCTORS ---------------------------------------------- */

    /**
     * Creates an empty hashtable
     */
    public this ()
    {
      m_hashtable = SC.Hashtable ();
    }

    
    /**
     * Creates an empty hashtable and sets its default capacity
     */
//    [Requires (capacity > 0)]
    public this (capacity : int)
    {
      m_hashtable = SC.Hashtable (capacity);
    }

    
    /**
     * Creates an empty hashtable and sets its default capacity
     * and the load factor.
     */
    [Requires (capacity > 0 && loadFactor >= 0.1f && loadFactor <= 1.0f)]
    public this (capacity : int, loadFactor : float)
    {
      m_hashtable = SC.Hashtable (capacity, loadFactor);
    }

    
    /* -- COPYING CONSTRUCTOR ---------------------------------------------- */

    /**
     * Creates a shallow copy of this hashtable
     */
    [Requires (ht != null)]
    public this (ht : Hashtable ['a, 'b])
    {
      m_hashtable = ht.m_hashtable.Clone () :> SC.Hashtable
    }
    
    

    /* -- PUBLIC PROPERTIES ------------------------------------------------ */

    /**
     * Unsafe access to the underlying hashtable
     */
    public Item [key : 'a] : 'b
    {
      get
      {
        m_hashtable [key] :> 'b
      }
      
      set
      {
        m_hashtable [key] = value
      }
    }


    /**
     * Returns a collection of the keys from this hashtable
     */
    public Keys : ICollection ['a]
    {
      get
      {
        def keys = m_hashtable.Keys;
        def result = LinkedList ();

        foreach (key in keys)
          result.Add (key :> 'a);

        result.Reverse ();
        result
      }
    }


    /**
     * Returns a collection of the values from this hashtable
     */
    public Values : ICollection ['b]
    {
      get
      {
        def values = m_hashtable.Values;
        def result = LinkedList ();

        foreach (value in values)
          result.Add (value :> 'b);

        result.Reverse ();
        result
      }
    }

    
    /**
     * Returns a collection of the key/value pairs from this hashtable
     */
    public KeyValuePairs : ICollection ['a * 'b]
    {
      get
      {
        def values_enumerator = m_hashtable.Values.GetEnumerator ();
        def keys_enumerator = m_hashtable.Keys.GetEnumerator ();
        def result = LinkedList ();

        while (keys_enumerator.MoveNext () && values_enumerator.MoveNext ())
          result.Add ((keys_enumerator.Current :> 'a, values_enumerator.Current :> 'b));

        result.Reverse ();
        result
      }
    }

    
    /**
     * Returns the number of key/value pairs in the hashtable
     */
    public Count : int
    {
      get { m_hashtable.Count }
    }

    
    /* -- PUBLIC METHODS --------------------------------------------------- */
        
    /**
     * Returns an optional value associated with the specified key.
     */
    public Get (key : 'a) : option ['b]
    {
      def value = m_hashtable [key];
      
      if (value != null)
        Some (value :> 'b)
      else
        None ()
    }

    
    /**
     * This is different from add, which can fail if the key is
     * already in the underlying Framework hashtable...
     */
    public Set (key : 'a, val : 'b) : void
    {
      m_hashtable [key] = val
    }

    
    /**
     * Adds a key/value pair to this hashtable. This method will
     * raise ArgumentException if the given key is already in
     * in the underlying Framework hashtable.
     */
    public Add (key : 'a, val : 'b) : void
    {
      m_hashtable.Add (key, val)
    }

    
    /**
     * Clears the contents of this hashtable.
     */
    public Clear () : void
    {
      m_hashtable.Clear ()
    }

    
    /**
     * Clones this hashtable.
     */
    public Clone () : Hashtable ['a,'b]
    {
      Hashtable (this)
    }


    
    /**
     * Returns `true' if the hashtable contains the specified key.
     *
     * NOTE: this is the same as ContainsKey.
     */
    public Contains (key : 'a) : bool
    {
      m_hashtable.Contains (key)
    }

    
    /**
     * Returns `true' if the hashtable contains the specified key.
     */
    public ContainsKey (key : 'a) : bool
    {
      m_hashtable.ContainsKey (key)
    }

    
    /**
     * Returns `true' if the hashtable contains the specified value.
     */
    public ContainsValue (val : 'b) : bool
    {
      m_hashtable.ContainsValue (val)
    }

    
    /**
     * Removes a key from the hashtable.
     */
    public Remove (key : 'a) : void
    {
      m_hashtable.Remove (key)
    }
     
    
    /**
     * Folds a function over the key/value pairs.
     */
    public Fold ['c] (s : 'c, f : ('a * 'b * 'c) -> 'c) : 'c 
    {
      mutable acc = s;
      
      foreach (x :> SC.DictionaryEntry in m_hashtable)
        acc = f (x.Key :> 'a, x.Value :> 'b, acc);
        
      acc
    }

    
    /**
     * Iterates a function over the key/value pairs in the hashtable.
     */
    public Iter (f : 'a * 'b -> void) : void
    {
      foreach (x :> SC.DictionaryEntry in m_hashtable)
        f (x.Key :> 'a, x.Value :> 'b)
    }

    
    /**
     * Maps a given function defined of key-value pairs to the contents
     * of this hashtable. A new hashtable object is created, containing
     * the results of the application.
     */
    public Map ['c, 'd] (f : 'a * 'b -> 'c * 'd) : Hashtable ['c,'d]
    {
      def ht = Hashtable ();

      foreach (x :> SC.DictionaryEntry in m_hashtable)
      {
        def (k, v) = f ((x.Key :> 'a), (x.Value :> 'b));
        
        ht.Add (k, v)
      }
      
      ht
    }

    
    /** 
     * Creates an enumerator
     */
    public GetEnumerator () : HashtableEnumerator ['a, 'b]
    {
      HashtableEnumerator (m_hashtable)
    }

    
    /* -- PRIVATE FIELDS --------------------------------------------------- */
    
    private m_hashtable : SC.Hashtable;
    
  } /* end of class Hashtable ('a,'b) */


  /**
   * Enumeration of a hashtable's key/value pairs
   */
  public class HashtableEnumerator ['a,'b] : IDictionaryEnumerator ['a,'b]
  {
    /* -- PUBLIC CONSTRUCTORS ---------------------------------------------- */
    
    /**
     * Creates a fresh enumerator for a given hashtable
     */
    [Requires (hashtable != null)]
    public this (hashtable : SC.Hashtable)
    {
      m_enumerator = hashtable.GetEnumerator ()
    }

    
    /* -- PUBLIC PROPERTIES ------------------------------------------------ */
    
    /**
     * Returns the current key
     */
    public Key : 'a
    {
      get { m_enumerator.Key :> 'a }
    }

    
    /**
     * Returns the current value
     */
    public Value : 'b
    { 
      get { m_enumerator.Value :> 'b }
    }

    
    /**
     * Returns the current key/value pair
     */
    public Entry : DictionaryEntry ['a, 'b]
    {
      get
      {
        def de = m_enumerator.Entry;
        
        DictionaryEntry (de.Key :> 'a, de.Value :> 'b)
      }
    }


    /**
     * Returns the current key/value pair
     */
    public Current : DictionaryEntry ['a, 'b]
    {
      get { Entry }
    }


    /* -- PUBLIC METHODS --------------------------------------------------- */
    
    /**
     * Resets the enumerator to its initial position
     */
    public Reset () : void
    {
      (m_enumerator : SC.IEnumerator).Reset ()
    }


    /**
     * Moves the enumerator to the next key/value pair
     */
    public MoveNext () : bool
    {
      (m_enumerator : SC.IEnumerator).MoveNext ()
    }

    
    /* -- PRIVATE FIELDS --------------------------------------------------- */

    /**
     * The underlying Framework enumerator
     */
    private mutable m_enumerator : SC.IDictionaryEnumerator;
  }
  
} /* end of namespace */

