/*
 *	Nautilus Wallpaper extension
 *
 *	Copyright (C) 2005 Adam Israel
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Adam Israel <adam@battleaxe.net> 
 */
 
#include <config.h>
#include <string.h>
#include <gconf/gconf-client.h>
#include <glib/gi18n-lib.h>
#include <libnautilus-extension/nautilus-extension-types.h>
#include <libnautilus-extension/nautilus-file-info.h>
#include <libnautilus-extension/nautilus-menu-provider.h>
#include "nautilus-nwe.h"

static GObjectClass *parent_class;

static void
set_wallpaper_callback (NautilusMenuItem *item,
              gpointer          user_data)
{
	GList            *files;
	GConfClient *client;
	GError *err;
	NautilusFileInfo *file;
	gchar            *uri, *filename;
	
	files = g_object_get_data (G_OBJECT (item), "files");
	file = files->data;
	
	uri = nautilus_file_info_get_uri (file);
	filename = g_filename_from_uri(uri, NULL, NULL);
	
	
	client = gconf_client_get_default ();
	gconf_client_set_string (client, "/desktop/gnome/background/picture_filename", 
							 filename, NULL);

	g_object_unref (client);	
	g_free (filename);
	g_free (uri);
	
}

static gboolean
is_image (NautilusFileInfo *file)
{
	gchar *mimeType;
	gboolean isImage;
	
	mimeType = nautilus_file_info_get_mime_type (file);
	
	isImage = g_str_has_prefix (nautilus_file_info_get_mime_type (file), "image/");
	
	g_free (mimeType);
	
	return isImage;	
}


static GList *
nautilus_nwe_get_file_items (NautilusMenuProvider *provider,
			      GtkWidget            *window,
			      GList                *files)
{
	GList    *items = NULL;
	GList    *scan;
	gboolean  one_item;
	NautilusMenuItem *item;

	
	for (scan = files; scan; scan = scan->next) {
		NautilusFileInfo *file = scan->data;
		gchar            *scheme;
		gboolean          local;

		scheme = nautilus_file_info_get_uri_scheme (file);
		local = strncmp (scheme, "file", 4) == 0;
		g_free (scheme);

		if (!local)
			return NULL;
	}
	
	one_item = (files != NULL) && (files->next == NULL);
	if (one_item && is_image ((NautilusFileInfo *)files->data) &&
	    !nautilus_file_info_is_directory ((NautilusFileInfo *)files->data)) {
		item = nautilus_menu_item_new ("NautilusNwe::sendto",
					       _("Set as Wallpaper..."),
					       _("Set image as the current wallpaper..."),
					       NULL);
		g_signal_connect (item, 
				  "activate",
				  G_CALLBACK (set_wallpaper_callback),
			  	provider);
		g_object_set_data_full (G_OBJECT (item), 
					"files",
					nautilus_file_info_list_copy (files),
					(GDestroyNotify) nautilus_file_info_list_free);
		items = g_list_append (items, item);
	}
	return items;
}


static void 
nautilus_nwe_menu_provider_iface_init (NautilusMenuProviderIface *iface)
{
	iface->get_file_items = nautilus_nwe_get_file_items;
}


static void 
nautilus_nwe_instance_init (NautilusNwe *nwe)
{
}


static void
nautilus_nwe_class_init (NautilusNweClass *class)
{
	parent_class = g_type_class_peek_parent (class);
}


static GType nwe_type = 0;


GType
nautilus_nwe_get_type (void) 
{
	return nwe_type;
}


void
nautilus_nwe_register_type (GTypeModule *module)
{
	static const GTypeInfo info = {
		sizeof (NautilusNweClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) NULL,
		(GClassInitFunc) nautilus_nwe_class_init,
		NULL, 
		NULL,
		sizeof (NautilusNwe),
		0,
		(GInstanceInitFunc) nautilus_nwe_instance_init,
	};

	static const GInterfaceInfo menu_provider_iface_info = {
		(GInterfaceInitFunc) nautilus_nwe_menu_provider_iface_init,
		NULL,
		NULL
	};

	nwe_type = g_type_module_register_type (module,
					         G_TYPE_OBJECT,
					         "NautilusNwe",
					         &info, 0);

	g_type_module_add_interface (module,
				     nwe_type,
				     NAUTILUS_TYPE_MENU_PROVIDER,
				     &menu_provider_iface_info);
}
