/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <locale.h>

#include <NASPRO/core/lib.h>

#include "config.h"

void
nacore_manifest_print_subject_prefixes(FILE *fp)
{
	fprintf(fp, "@prefix lv2: <http://lv2plug.in/ns/lv2core#> .\n");
}

void
nacore_manifest_print_subject_triple(struct nacore_descriptor *desc, FILE *fp)
{
	fprintf(fp, "<%s> a lv2:Plugin .\n", desc->uri);
}

static void
print_escaped_string(FILE *fp, const char *string)
{
	size_t i, len;

	len = strlen(string);
	for (i = 0; i < len; i++)
	  {
		switch (string[i])
		  {
			case '\t':
				fputs("\\t", fp);
				break;
			case '\r':
				fputs("\\r", fp);
				break;
			case '\n':
				fputs("\\n", fp);
				break;
			case '"':
				fputs("\\\"", fp);
				break;
			case '\\':
				fputs("\\\\", fp);
				break;
			/* TODO: unicode escapes, see
			 * http://www.dajobe.org/2004/01/turtle/#sec-strings */
			default:
				fputc(string[i], fp);
				break;
		  }
	  }
}

#define PRINT_C1(na_class, string) \
	if (desc->classes_1 & na_class) \
		fprintf(fp, "\ta " string " ;\n");

#define PRINT_C2(na_class, string) \
	if (desc->classes_2 & na_class) \
		fprintf(fp, "\ta " string " ;\n");

static void
print_classes(FILE *fp, struct nacore_descriptor *desc)
{
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_GENERATOR,	"lv2:GeneratorPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_INSTRUMENT,	"lv2:InstrumentPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_OSCILLATOR,	"lv2:OscillatorPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_UTILITY,	"lv2:UtilityPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_CONVERTER,	"lv2:ConverterPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_ANALYSER,	"lv2:AnalyserPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_MIXER,		"lv2:MixerPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_SIMULATOR,	"lv2:SimulatorPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_DELAY,		"lv2:DelayPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_MODULATOR,	"lv2:ModulatorPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_REVERB,	"lv2:ReverbPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_PHASER,	"lv2:PhaserPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_FLANGER,	"lv2:FlangerPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_CHORUS,	"lv2:ChorusPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_FILTER,	"lv2:FilterPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_LOWPASS,	"lv2:LowpassPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_BANDPASS,	"lv2:BandpassPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_HIGHPASS,	"lv2:HighpassPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_COMB,		"lv2:CombPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_ALLPASS,	"lv2:AllpassPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_EQ,		"lv2:EQPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_PARAEQ,	"lv2:ParaEQPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_MULTIEQ,	"lv2:MultiEQPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_SPECTRAL,	"lv2:SpectralPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_PITCH,		"lv2:PitchPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_AMPLIFIER,	"lv2:AmplifierPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_DISTORTION,	"lv2:DistortionPlugin");
	PRINT_C1(NACORE_DESCRIPTOR_CLASS_WAVESHAPER,	"lv2:WaveshaperPlugin");
	PRINT_C2(NACORE_DESCRIPTOR_CLASS_DYNAMICS,	"lv2:DynamicsPlugin");
	PRINT_C2(NACORE_DESCRIPTOR_CLASS_COMPRESSOR,	"lv2:CompressorPlugin");
	PRINT_C2(NACORE_DESCRIPTOR_CLASS_EXPANDER,	"lv2:ExpanderPlugin");
	PRINT_C2(NACORE_DESCRIPTOR_CLASS_LIMITER,	"lv2:LimiterPlugin");
	PRINT_C2(NACORE_DESCRIPTOR_CLASS_GATE,		"lv2:GatePlugin");
}

#define PRINT_UNIT(na_unit, lv2_unit) \
	case na_unit: \
		fprintf(fp, "\t\tlv2units:unit " lv2_unit " ;\n"); \
		break;

static void
print_scale_units(FILE *fp, struct nacore_descriptor *desc, size_t index)
{
	switch (desc->port_descs[index].scale.unit)
	  {
		PRINT_UNIT(nacore_scale_unit_bar,	"lv2units:bar");
		PRINT_UNIT(nacore_scale_unit_beat,	"lv2units:beat");
		PRINT_UNIT(nacore_scale_unit_bpm,	"lv2units:bpm");
		PRINT_UNIT(nacore_scale_unit_cent,	"lv2units:cent");
		PRINT_UNIT(nacore_scale_unit_cm,	"lv2units:cm");
		PRINT_UNIT(nacore_scale_unit_coeff,	"lv2units:coef");
		PRINT_UNIT(nacore_scale_unit_db,	"lv2units:db");
		PRINT_UNIT(nacore_scale_unit_deg,	"lv2units:degree");
		PRINT_UNIT(nacore_scale_unit_hz,	"lv2units:hz");
		PRINT_UNIT(nacore_scale_unit_inch,	"lv2units:inch");
		PRINT_UNIT(nacore_scale_unit_khz,	"lv2units:khz");
		PRINT_UNIT(nacore_scale_unit_km,	"lv2units:km");
		PRINT_UNIT(nacore_scale_unit_m,		"lv2units:m");
		PRINT_UNIT(nacore_scale_unit_mhz,	"lv2units:mhz");
		PRINT_UNIT(nacore_scale_unit_midi_note, "lv2units:midiNote");
		PRINT_UNIT(nacore_scale_unit_mi,	"lv2units:mile");
		PRINT_UNIT(nacore_scale_unit_min,	"lv2units:min");
		PRINT_UNIT(nacore_scale_unit_mm,	"lv2units:mm");
		PRINT_UNIT(nacore_scale_unit_ms,	"lv2units:ms");
		PRINT_UNIT(nacore_scale_unit_oct,	"lv2units:oct");
		PRINT_UNIT(nacore_scale_unit_pc,	"lv2units:pc");
		PRINT_UNIT(nacore_scale_unit_s,		"lv2units:s");
		PRINT_UNIT(nacore_scale_unit_semitone,
			   "lv2units:semitone12TET");
		default:
			break;
	  }
}

static void
print_scale_points(FILE *fp, struct nacore_descriptor *desc, size_t index)
{
	size_t i;
	
	if (desc->port_descs[index].scale.points_count == 0)
		return;

	for (i = 0; i < desc->port_descs[index].scale.points_count; i++)
	  {
		fprintf(fp, "\t\tlv2:scalePoint [ rdfs:label \"");
		print_escaped_string(fp,
			desc->port_descs[index].scale.points[i].label);
		fprintf(fp, "\"; rdf:value %f ] ;\n",
			desc->port_descs[index].scale.points[i].value);
	  }
}

void
nacore_manifest_print_data(struct nacore_descriptor *desc, FILE *fp,
			   const char *binary)
{
	struct nacore_port_descriptor *port_desc;
	char *prev_locale;
	size_t i;

	/* FIXME: we need a locale-independent way of printing float values */

	prev_locale = setlocale(LC_NUMERIC, NULL);
	setlocale(LC_NUMERIC, "POSIX");
	
	fprintf(fp, "@prefix rdf: <http://www.w3.org/1999/02/"
		    "22-rdf-syntax-ns#> .\n");
	fprintf(fp, "@prefix rdfs: <http://www.w3.org/2000/01/"
		     "rdf-schema#> .\n");
	fprintf(fp, "@prefix lv2: <http://lv2plug.in/ns/lv2core#> .\n");
	fprintf(fp, "@prefix doap: <http://usefulinc.com/ns/doap#> .\n");
	fprintf(fp, "@prefix dc: <http://purl.org/dc/elements/1.1/> .\n");
	fprintf(fp, "@prefix epp: <http://lv2plug.in/ns/dev/extportinfo#> .\n");
	fprintf(fp, "@prefix lv2units: <http://lv2plug.in/ns/extensions/"
		    "units#> .\n");
	fprintf(fp, "@prefix lv2ev: <http://lv2plug.in/ns/ext/event#> .\n");
	fprintf(fp, "@prefix lv2midi: <http://lv2plug.in/ns/ext/midi#> .\n");

	fprintf(fp, "<%s>\n", desc->uri);
	fprintf(fp, "\ta lv2:Plugin ;\n");
	fprintf(fp, "\tlv2:binary <");
	print_escaped_string(fp, binary);
	fprintf(fp, SO_FILE_EXT "> ;\n");
	fprintf(fp, "\tdoap:name \"");
	print_escaped_string(fp, desc->name);
	fprintf(fp, "\" ;\n");
	fprintf(fp, "\tdoap:license "
		"<http://usefulinc.com/doap/licenses/unknown> ;\n");
	if (desc->creator != NULL)
	  {
		fprintf(fp, "\tdc:creator \"");
		print_escaped_string(fp, desc->creator);
		fprintf(fp, "\" ;\n");
	  }
	if (desc->rights != NULL)
	  {
		fprintf(fp, "\tdc:rights \"");
		print_escaped_string(fp, desc->rights);
		fprintf(fp, "\" ;\n");
	  }

	if (desc->properties & NACORE_DESCRIPTOR_HAS_LIVE_DEP)
		fprintf(fp, "\tlv2:requiredFeature lv2:isLive ;\n");
	if (desc->properties & NACORE_DESCRIPTOR_INPLACE_BROKEN)
		fprintf(fp, "\tlv2:requiredFeature lv2:inPlaceBroken ;\n");
	if (desc->properties & NACORE_DESCRIPTOR_HARD_RT_CAPABLE)
		fprintf(fp, "\tlv2:optionalFeature lv2:hardRtCapable ;\n");

	print_classes(fp, desc);

	for (i = 0; i < desc->port_descs_count; i++)
	  {
		if (desc->port_descs[i].properties & NACORE_PORT_IS_MIDI)
		  {
			fprintf(fp, "\tlv2:requiredFeature "
				"<http://lv2plug.in/ns/ext/event> ;\n");
			fprintf(fp, "\tlv2:requiredFeature "
				"<http://lv2plug.in/ns/ext/uri-map> ;\n");
			break;
		  }
	  }

	for (i = 0; i < desc->port_descs_count; i++)
	  {
		port_desc = desc->port_descs + i;

		if (i)
			fprintf(fp, " ,\n\t");
		else
			fprintf(fp, "\tlv2:port ");

		fprintf(fp, "[\n");
		fprintf(fp, "\t\ta %s, lv2:%s ;\n",
			(port_desc->properties & NACORE_PORT_IS_AUDIO)
			? "lv2:AudioPort" :
			((port_desc->properties & NACORE_PORT_IS_MIDI)
			 ? "lv2ev:EventPort" : "lv2:ControlPort"),
			(port_desc->properties & NACORE_PORT_IS_OUTPUT)
			? "OutputPort" : "InputPort");

		fprintf(fp, "\t\tlv2:index %lu ;\n", (unsigned long)i);
		fprintf(fp, "\t\tlv2:symbol \"port%lu\" ;\n", (unsigned long)i);
		fprintf(fp, "\t\tlv2:name \"");
		print_escaped_string(fp, port_desc->name);
		fprintf(fp, "\" ;\n");

		if (port_desc->properties & NACORE_PORT_CONNECTION_OPTIONAL)
			fprintf(fp, "\t\tlv2:portProperty "
				"lv2:connectionOptional ;\n");

		if (!(port_desc->properties & NACORE_PORT_IS_AUDIO)
		    && !(port_desc->properties & NACORE_PORT_IS_MIDI))
		  {
			if (port_desc->properties & NACORE_PORT_REPORTS_LATENCY)
				fprintf(fp, "\t\tlv2:portProperty "
					"lv2:reportsLatency ;\n");
			if (port_desc->properties & NACORE_PORT_TOGGLED)
				fprintf(fp, "\t\tlv2:portProperty "
					"lv2:toggled ;\n");
			if (port_desc->properties & NACORE_PORT_SAMPLE_RATE)
				fprintf(fp, "\t\tlv2:portProperty "
					"lv2:sampleRate ;\n");
			if (port_desc->properties & NACORE_PORT_INTEGER)
				fprintf(fp, "\t\tlv2:portProperty "
					"lv2:integer ;\n");
			if (port_desc->properties & NACORE_PORT_LOGARITHMIC)
				fprintf(fp, "\t\tlv2:portProperty "
					"epp:logarithmic ;\n");
			if (port_desc->scale.properties & NACORE_SCALE_HAS_MIN)
				fprintf(fp, "\t\tlv2:minimum %f ;\n",
					port_desc->scale.min);
			if (port_desc->scale.properties & NACORE_SCALE_HAS_MAX)
				fprintf(fp, "\t\tlv2:maximum %f ;\n",
					port_desc->scale.max);
			if (port_desc->scale.properties
			    & NACORE_SCALE_HAS_DEFAULT)
				fprintf(fp, "\t\tlv2:default %f ;\n",
					port_desc->scale.defaultv);

			print_scale_units(fp, desc, i);
			print_scale_points(fp, desc, i);
		  }
		else if (port_desc->properties & NACORE_PORT_IS_MIDI)
		  {
			fprintf(fp, "\t\tlv2ev:supportsEvent "
				"lv2midi:MidiEvent ;\n");
		  }

		fprintf(fp, "\t]");
	  }

	fprintf(fp, " .\n");

	setlocale(LC_NUMERIC, prev_locale);
}
