/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure_parser.cpp
 * @brief Parser for figure elements, which are converted from XML to our internal model.
 * 
 */

#include "myx_gc_figure_parser.h"
#include "myx_gc_gl_helper.h"

static map<string, TFigureElementResize> resizeLookup;
static map<string, TFigureElementLayout> layoutLookup;
static map<string, TAlignment> alignmentLookup;

#ifdef __GNUC__
  static map<wstring, TActionType> actionLookup;
#else
  static hash_map<wstring, TActionType> actionLookup;
#endif

//----------------------------------------------------------------------------------------------------------------------

extern const string defaultFontFamily;
extern const string defaultFontStyle;
extern const string defaultFontWeight;
extern const int defaultFontSize;

extern const string DefaultLayout;
extern const string DefaultResize;

//----------------- CFigureParser --------------------------------------------------------------------------------------

CFigureParser::CFigureParser(CGenericCanvas* canvas): CGCBase(canvas)
{
}

//----------------------------------------------------------------------------------------------------------------------

CFigureParser::~CFigureParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if the static lookup tables are set up already. If not then it is done.
 */
void CFigureParser::checkLookupTables(void)
{

  if (layoutLookup.size() == 0)
  {
    layoutLookup["row"] = GC_LAYOUT_ROW;
    layoutLookup["column"] = GC_LAYOUT_COLUMN;
  };

  if (alignmentLookup.size() == 0)
  {
    alignmentLookup["left"] = GC_ALIGN_LEFT_TOP;
    alignmentLookup["center"] = GC_ALIGN_CENTER;
    alignmentLookup["centered"] = GC_ALIGN_CENTER;
    alignmentLookup["middle"] = GC_ALIGN_CENTER;
    alignmentLookup["right"] = GC_ALIGN_RIGHT_BOTTOM;
    alignmentLookup["top"] = GC_ALIGN_LEFT_TOP; 
    alignmentLookup["bottom"] = GC_ALIGN_RIGHT_BOTTOM; 
  };

  if (actionLookup.size() == 0)
  {
    actionLookup[L"none"] = GC_ACTION_NONE;
    actionLookup[L"toggle"] = GC_ACTION_TOGGLE;
    actionLookup[L"collapse"] = GC_ACTION_COLLAPSE;
    actionLookup[L"expand"] = GC_ACTION_EXPAND;
    actionLookup[L"changeStyle"] = GC_ACTION_CHANGE_STYLE;
    actionLookup[L"dragFigure"] = GC_ACTION_DRAG_FIGURE;
    actionLookup[L"do"] = GC_ACTION_APPLICATION;
  };

  if (resizeLookup.size() == 0)
  {
    resizeLookup["none"] = GC_RESIZE_NONE;
    resizeLookup["horizontal"] = GC_RESIZE_HORIZONTAL_ONLY;
    resizeLookup["vertical"] = GC_RESIZE_VERTICAL_ONLY;
    resizeLookup["all"] = GC_RESIZE_ALL;
  };

}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given XML node and interprets it as a caption definition.
 *
 * @param node The XML node to parse.
 * @return A new caption element.
 */
CCaptionElementTemplate* CFigureParser::parseCaption(xmlNodePtr node)
{
  wstring key = utf8ToUtf16(getStringAttributeDef(node, "key", ""));
  wstring text = utf8ToUtf16(getStringAttributeDef(node, "default", ""));
  float x = getFloatAttributeDef(node, "x", 0);
  float y = getFloatAttributeDef(node, "y", 0);
  string fontFamily = getStringAttributeDef(node, "font-family", defaultFontFamily);
  int fontSize = getIntAttributeDef(node, "font-size", defaultFontSize);
  string fontStyle = getStringAttributeDef(node, "font-style", defaultFontStyle);
  string fontWeightString = getStringAttributeDef(node, "font-weight", defaultFontWeight);
  int weight = convertFontWeight(fontWeightString);
  string attribute = getStringAttributeDef(node, "horizontal-alignment", "left");
  TAlignment horizontalAlignment = alignmentLookup[attribute];
  attribute = getStringAttributeDef(node, "vertical-alignment", "top");
  TAlignment verticalAlignment = alignmentLookup[attribute];
  GLubyte color[4];
  color[3] = 255;
  GLubyte* colorPtr = color;
  if (convertColor(node, "fill", color) != 0)
    colorPtr = NULL;
  string wrap= getStringAttributeDef(node, "wrap", "false");
  bool wrapText = strcmp(wrap.c_str(), "true")==0;

  TConstraints constraints;
  constraints.maxHeight = getFloatAttributeDef(node, "max-height", -1);
  constraints.maxWidth = getFloatAttributeDef(node, "max-width", -1);
  constraints.minHeight = getFloatAttributeDef(node, "min-height", -1);
  constraints.minWidth = getFloatAttributeDef(node, "min-width", -1);

  CCaptionElementTemplate* result = new CCaptionElementTemplate(key);
  result->initialize(text, x, y, fontFamily, fontSize, weight, fontStyle, horizontalAlignment, verticalAlignment, 
    colorPtr, constraints, wrapText);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses a single element and returns a new figure element instance. Can be called recursively.
 *
 * @param node The XML node to parse.
 * @return The new figure element instance created out of the element description.
 */
CFigureElementTemplate* CFigureParser::parseElement(xmlNodePtr node, CGCModel* model)
{
  wstring id = utf8ToUtf16(getStringAttributeDef(node, "id", ""));
  wstring key = utf8ToUtf16(getStringAttributeDef(node, "key", ""));
  CFigureElementTemplate* result = new CFigureElementTemplate(id, key);

  string attribute = getStringAttributeDef(node, "layout", DefaultLayout);
  TFigureElementLayout layout = layoutLookup[attribute];

  attribute = getStringAttributeDef(node, "resize", DefaultResize);
  TFigureElementResize resizeMode = resizeLookup[attribute];

  wstring styleName;
  if (getStringAttribute(node, "template", attribute))
    styleName = utf8ToUtf16(attribute);

  TConstraints constraints;
  constraints.maxHeight = getFloatAttributeDef(node, "max-height", -1);
  constraints.maxWidth = getFloatAttributeDef(node, "max-width", -1);
  constraints.minHeight = getFloatAttributeDef(node, "min-height", -1);
  constraints.minWidth = getFloatAttributeDef(node, "min-width", -1);
  TOccurence occurence = GC_OCC_ONCE;
  if (getStringAttribute(node, "occurrence", attribute))
  {
    if (attribute == "+")
      occurence = GC_OCC_ONE_OR_MORE;
    if (attribute == "*")
      occurence = GC_OCC_ZERO_OR_MORE;
  };
  result->initialize(layout, resizeMode, model->style(styleName), constraints, occurence);

  // Parse action entries.
  if (getStringAttribute(node, "action", attribute))
    parseActions(attribute, result, node->line);

  // Now go through the child nodes. There can either be child elements or a caption node.
  xmlNodePtr run = node->children;
  while (run != NULL)
  {
    if (XML_IS(run, "caption"))
    {
      CCaptionElementTemplate* childElement = parseCaption(run);
      result->setCaption(childElement);
    }
    else
      if (XML_IS(run, "element"))
      {
        CFigureElementTemplate* childElement = parseElement(run, model);
        result->addChild(childElement);
      };  

    run = run->next;
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses an "action" entry in the layout definition.
 *
 * @param source The source string to parse.
 * @param template_ The target template that gets the actions.
 */
void CFigureParser::parseActions(string source, CFigureElementTemplate* template_, unsigned short lineNumber)
{
  bool foundError = false;

#define PARSE_ERROR(message) { \
  char buffer[100]; \
  sprintf(buffer, message, lineNumber); \
  error(this, buffer); \
  foundError = true; \
  break; }

  // Split entire actions separated by semicolon.
  StringTokenizer actionTokenizer(source, ";");
  while (!foundError && actionTokenizer.hasMoreTokens())
  {
    // A new action start.
    string token = actionTokenizer.nextToken();

    // Split action definition into "action" and "parameters".
    StringTokenizer partTokenizer(token, "()");
    while (partTokenizer.hasMoreTokens())
    {
      TAction action;
      action.type = actionLookup[utf8ToUtf16(partTokenizer.nextToken())];

      if (action.type != GC_ACTION_NONE)
      {
        if ((partTokenizer.lastDelimiter() != '(') && !partTokenizer.hasMoreTokens())
          PARSE_ERROR("Figure action parser error (line %d): parentheses expected.")

        // Collect parameters in the second part, separated by comma, if there are any.
        if (partTokenizer.hasMoreTokens())
        {
          StringTokenizer parameterTokenizer(partTokenizer.nextToken(), ",");
          while (parameterTokenizer.hasMoreTokens())
          {
            string parameter = parameterTokenizer.nextToken();

            // Trim the parameter. 
            // TODO: use a library function for trimming. Better, switch to a good tokenizer.
            string::size_type head = 0;
            string::size_type tail = parameter.size();
            while (head < tail && (parameter[head] == ' ' || parameter[head] == '\t'))
              ++head;
            while (tail > head && (parameter[tail] == ' ' || parameter[tail] == '\t'))
              --tail;
            
            // Remove quotes if there are any.
            if (parameter[0] == '"' || parameter[0] == '\'')
              parameter = parameter.substr(1, parameter.length() - 2);

            action.parameters.push_back(utf8ToUtf16(parameter.substr(head, tail)));
          };
        };
        template_->addAction(action);
      };
    };
  };

#undef PARSE_ERROR
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property to return.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CFigureParser::property(const char* name, unsigned int index)
{
  TGCVariant variant;

  return variant;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses a single layout definition and creates a figure template from it.
 *
 * @param definition The definition to parse.
 * @param The model class that gets the new template.
 */
void CFigureParser::parseLayoutDefinition(xmlNodePtr definition, CGCModel* model)
{
  checkLookupTables();

  wstring type = utf8ToUtf16(getStringAttributeDef(definition, "type", ""));
  wstring layoutClass = utf8ToUtf16(getStringAttributeDef(definition, "layout-class", ""));
  if (type.size() > 0)
  {
    CFigureTemplate* figureTemplate = model->createLayout(type, layoutClass);
    
    // Find first <element> node. This is our root.
    xmlNodePtr run = definition->children;
    while (run != NULL)
    {
      if (XML_IS(run, "element"))
        break;
      run = run->next;
    };

    if (run != NULL)
      figureTemplate->FContent = parseElement(run, model);
  };
}

//----------------------------------------------------------------------------------------------------------------------

