/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdlib.h>

#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include <myx_grt_public_interface.h>
#include <myx_grt_lua.h>
#include <myx_grt_java.h>

#include <myx_grt_lua_shell.h>


static char *read_command(MYX_GRT *env)
{
  char *line= NULL;
  //int status;
#ifndef USE_READLINE
  char linebuf[1024];
#endif
  char *prompt= myx_grt_lua_shell_get_prompt(env);

#ifdef USE_READLINE
  line= readline(prompt);
  if (!line)
    return NULL;
  add_history(line);
  line= str_g_append(line, "\n");
#else
  printf(prompt);
  fflush(stdout);
  if (fgets(linebuf, sizeof(linebuf), stdin) <= 0)
  {
    return NULL;
  }
  line= g_strdup(linebuf);
#endif

  return line;
}


MYX_GRT *initialize_grt(const char *classpath)
{
  MYX_GRT *grt= myx_grt_initialize();
  MYX_GRT_MODULE_LOADER *loader;
  MYX_GRT_ERROR error;

  // initialize the loaders
  g_message("Initializing Java loader...");
  loader= myx_java_init_loader(grt, classpath, &error);
  if (!loader)
  {
    g_warning("Error initializing Java module loader (%i)", error);
  }
  else
  {
    if (myx_grt_register_module_loader(grt, loader) < 0)
    {
      g_warning("Could not register Java module loader.");
    }
  }

  g_message("Initializing Lua loader...");
  loader= myx_lua_init_loader(grt, &error);
  if (!loader)
  {
    g_warning("Error initializing Lua module loader (%i)", error);
  }
  else
  {
    if (myx_grt_register_module_loader(grt, loader) < 0)
    {
      g_warning("Could not register Java module loader.");
    }
  }
  myx_grt_init_lua_shell(grt);

  return grt;
}


int main(int argc, char **argv)
{
  MYX_GRT_ERROR status;
  MYX_GRT *grt;
  char *classpath;
  char *linebuf= NULL;
  int c;

  if(argc==1)
  {
    fprintf(stdout, "Usage: %s (-l | -w | classpath) [-x] [luafile]\n\n"
      "  -l ......... specifies '../java' as java classpath (typical on linux).\n"
      "  -w ......... specifies 'java' as java classpath (typical on windows).\n"
      "  classpath .. set the classpath to the given value.\n"
      "  -x ......... exits the shell after running the specified file\n"
      "  luafile .... file that is run at startup.\n"
      "\n", argv[0]);
    exit(0);
  }

  //Initialize jni_sys
  if((argv[1][0]=='-')&&(argv[1][1]=='w'))
    classpath= "../../source/java";
  else if((argv[1][0]=='-')&&(argv[1][1]=='l'))
    classpath= "../../source/java";
  else
    classpath= argv[1];

  //Initialize grt
  grt= initialize_grt(classpath);
  if (!grt)
  {
    g_message("error initializing GRT");
    exit(1);
  }

  g_assert(lua_gettop(myx_grt_lua_shell_get_lua(grt))==0);

  g_message("Scanning for plugins...");
  c= myx_grt_scan_for_modules(grt, ".", &status);

  g_message("Initialized %i modules", c);

  g_assert(lua_gettop(myx_grt_lua_shell_get_lua(grt))==0);

  if ((argc == 4) && (strcmp(argv[2], "-x") == 0))
  {
    exit(myx_grt_lua_shell_run_file(grt, argv[3]));
  } 
  else if ((argc == 3) && (argv[2][0]!='-'))
  {
    myx_grt_lua_shell_run_file(grt, argv[2]);
  }

  for (;;)
  {
    linebuf= read_command(grt);
    if (!linebuf)
    {
      break;
    }

    if(myx_grt_lua_shell_execute(grt, linebuf) == MYX_GRT_SHELL_COMMAND_EXIT)
      break;

    g_free(linebuf);
  }

  puts("\nExiting...");
  myx_grt_finalize(grt);
  
  return 0;
}
