/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_view.cpp 
 * @brief Implementation of the view class.
 * 
 */

#include "myx_gc.h"

#include "myx_gc_view.h"

//----------------- CHitResults ----------------------------------------------------------------------------------------

CHitResults::CHitResults(void)
{
  FCurrentEntry = FEntries.end();
}

//----------------------------------------------------------------------------------------------------------------------

CHitResults::~CHitResults(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::addHit(CFigureInstance* Instance)
{
  FEntries.push_back(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

int CHitResults::count(void)
{
  return FEntries.size();
}

//----------------------------------------------------------------------------------------------------------------------

bool CHitResults::hasNext(void)
{
  return FCurrentEntry != FEntries.end();
}

//----------------------------------------------------------------------------------------------------------------------

CFigureInstance* CHitResults::next(void)
{
  if (FCurrentEntry == FEntries.end())
    return NULL;
  else
    return *FCurrentEntry++;
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::reset(void)
{
  FCurrentEntry = FEntries.begin();
}

//----------------- CLayerListener -----------------------------------------------------------------------------

void CLayerListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (view != NULL)
    view->change(origin, reason);
};

//----------------------------------------------------------------------------------------------------------------------

void CLayerListener::onDestroy(CGCBase* object)
{
  if (view != NULL)
    view->removeLayer((CLayer*) object);
}

//----------------------------------------------------------------------------------------------------------------------

void CLayerListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (view != NULL)
    view->error(origin, message);
}

//----------------- CGCView --------------------------------------------------------------------------------------------

CGCView::CGCView(CGenericCanvas* Owner, string Name): CGCBase(Owner)
{
  _className = "CGCView";
  FCanvas = Owner;
  FName = Name;
  FColor[0] = 0;
  FColor[1] = 0;
  FColor[2] = 0;
  FColor[3] = 1;
  FZoomX = 1;
  FZoomY = 1;
  FOffsetX = 0;
  FOffsetY = 0;
  FNearPlane = -100;
  FFarPlane = 100;
  FJitter = 0;
  FGrid = new CGridLayer(this, Owner);
  FListener.view = this;
}

//----------------------------------------------------------------------------------------------------------------------

CGCView::~CGCView()
{
  delete FGrid;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Used to set up things that need only to be done once if a view becomes active.
 */
void CGCView::activate(void)
{
  glClearColor(FColor[0], FColor[1], FColor[2], FColor[3]);

  // TODO: make this dynamically configurable (where applicable).
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glFrontFace(GL_CW);
  glDisable(GL_AUTO_NORMAL);
  glDisable(GL_CULL_FACE);
  glDisable(GL_DITHER);
  glDisable(GL_DEPTH_TEST);

  // glEnable(GL_POINT_SMOOTH);
  // glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_LINE_SMOOTH);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_POLYGON_SMOOTH);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

  glDisable(GL_TEXTURE_2D);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets up the current projection and modelview matrices.
 *
 * @param DoProjection A flag telling if also the project matrix should be set. Useful to avoid unnecessary applications.
 */
void CGCView::applyTransformations(bool DoProjection)
{
  if (DoProjection)
  {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-FJitter, FViewport.width - FJitter, FViewport.height - FJitter, -FJitter, FNearPlane, FFarPlane);
  };

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns a zoom value suitable for feedback hit tests.
 *
 * @return The largest of all scale factors.
 */
float CGCView::feedbackZoom(void)
{
  return (FZoomX >= FZoomY) ? FZoomX : FZoomY;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This is the main paint routine. It is called by the canvas if this view is the current view.
 */
void CGCView::render(void)
{
  applyTransformations(true);

  // Transform viewport coordinates into view space. They are used for occlusion culling.
  TBoundingBox visibleBounds;
  visibleBounds.upper.x = (FViewport.left - FOffsetX) / FZoomX;
  visibleBounds.upper.y = (FViewport.top - FOffsetY) / FZoomY;
  visibleBounds.lower.x = (FViewport.left + FViewport.width - FOffsetX) / FZoomX;
  visibleBounds.lower.y = (FViewport.top + FViewport.height - FOffsetY) / FZoomY;

  for (CLayers::iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    (*iterator)->render(FMinimalZoom, visibleBounds);

  FGrid->render(FMinimalZoom, visibleBounds);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a layer to the internal list of layers that belong to this view (only if not yet there).
 *
 * @param Layer The layer to add.
 */
void CGCView::addLayer(CLayer* Layer)
{
  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    if (*iterator == Layer)
      return;

  Layer->addListener(&FListener);
  FLayers.push_back(Layer);
  
  change(this, GC_CHANGE_VIEW_ADD_LAYER);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new background color of this view.
 *
 * @param Red The red color component.
 * @param Green The green color component.
 * @param Blue The blue color component.
 * @param Alpha The transparency component.
 */
void CGCView::color(float Red, float Green, float Blue, float Alpha)
{
  FColor[0] = Red;
  FColor[1] = Green;
  FColor[2] = Blue;
  FColor[3] = Alpha;

  change(this, GC_CHANGE_VIEW_PROPERTY);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new background color of this view
 *
 * @param NewColor The new color to use.
 */
void CGCView::color(GLfloat* NewColor)
{
  FColor[0] = NewColor[0];
  FColor[1] = NewColor[1];
  FColor[2] = NewColor[2];
  FColor[3] = NewColor[3];

  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether this view contains a reference to the given layer.
 *
 * @param Layer The layer to look for.
 * @param True if the layer is referenced in this view otherwise false.
 */
bool CGCView::contains(CLayer* Layer)
{
  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    if (*iterator == Layer)
      return true;

  return false;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Jittering the viewport a little bit sometimes improves display quality (e.g. for thin lines). This function
 * sets this value for this view.
 *
 * @param Value The new jitter value.
 */
void CGCView::jitter(float Value)
{
  FJitter = Value;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current jitter value.
 *
 * @return The current jitter value.
 */
float CGCView::jitter(void)
{
  return FJitter;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given coordinates and tries to find figure instances that were rendered at this position.
 * Positions are given in the coordinate system of the viewer window (subpixel accuracy is supported).
 *
 * @param x The horizontal window coordinate.
 * @param y The vertical window coordinate. 
 * @param SingleHit If true then search for hits is stopped after the first one was found.
 * @return A hit result class is returned regardless of the actual number of hits. It must be freed by the caller.
 */
CHitResults* CGCView::getHitTestInfoAt(const float x, const float y, bool SingleHit)
{
  CHitResults* Result = new CHitResults();

  applyTransformations(true);

  GLint viewport[4];
  glGetIntegerv(GL_VIEWPORT, viewport);

  GLdouble ModelviewMatrix[16];
  glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

  GLdouble ProjectionMatrix[16];
  glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

  // Convert window coordinates into object (scene) coordinates.
  double LocalX, LocalY, LocalZ;
  gluUnProject(x, viewport[3] - y, 0, ModelviewMatrix, ProjectionMatrix, viewport, &LocalX, &LocalY, &LocalZ);

  for (CLayers::reverse_iterator iterator = FLayers.rbegin(); iterator != FLayers.rend(); ++iterator)
  {
    CLayer* Layer = *iterator;
    if (Layer->enabled())
      Layer->getHitTestInfoAt(Result, (float) LocalX, (float) LocalY, SingleHit);

    // Stop here if we have a hit and only one is requested.
    if (SingleHit && Result->count() > 0)
      break;
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the horizontal offset of the view. The offset is value about which the content of the view is moved.
 *
 * @param Value The new horizontal offset.
 */
void CGCView::offsetX(float Value)
{
  FOffsetX = Value;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current horizontal offset.
 *
 * @return The current horizontal offset.
 */
float CGCView::offsetX(void)
{
  return FOffsetX;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new vertical offset.
 *
 * @param Value The new vertical offset.
 */
void CGCView::offsetY(float Value)
{
  FOffsetY = Value;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current vertical offset.
 *
 * @return The curretn vertical offset.
 */
float CGCView::offsetY(void)
{
  return FOffsetY;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this parameter gives the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CGCView::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              result.s = FName;
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "A view comprising a set of layers.";
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FCanvas;
              break;
            };
          case GC_PROPERTY_COLOR:
            {
              result.type = GC_VAR_STRING;
              result.s = colorToString(FColor);
              break;
            };
          case GC_PROPERTY_ZOOMX:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FZoomX;
              break;
            };
          case GC_PROPERTY_ZOOMY:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FZoomY;
              break;
            };
          case GC_PROPERTY_X:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FOffsetX;
              break;
            };
          case GC_PROPERTY_Y:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FOffsetY;
              break;
            };
          case GC_PROPERTY_JITTER:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FJitter;
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_LAYERS:
      {
        result.type = GC_VAR_LIST;
        if (index < FLayers.size())
          result.reference = (CGCBase*) FLayers[index];
        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this parameter gives the index into that list.
 * @param value The new value of the property. Automatic conversion is performed where possible.
 */
void CGCView::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getPropertyID(name))
  {
    case GC_PROPERTY_COLOR:
      {
        stringToColor(variantToString(value), FColor);
        FColor[3] = 1;
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_ZOOMX:
      {
        FZoomX = variantToFloat(value);
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_ZOOMY:
      {
        FZoomY = variantToFloat(value);
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_X:
      {
        FOffsetX = variantToFloat(value);
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_Y:
      {
        FOffsetY = variantToFloat(value);
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_JITTER:
      {
        FJitter = variantToFloat(value);
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
  };
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 *  Removes the given layer from the list of layers that comprise this view.
 *
 * @param Layer The layer to remove.
 */
void CGCView::removeLayer(CLayer* Layer)
{
  for (CLayers::iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    if (*iterator == Layer)
    {
      if (!Layer->destroying())
        Layer->removeListener(&FListener);
      FLayers.erase(iterator);
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new viewport for this view.
 *
 * @param NewViewport The new viewport to be used.
 */
void CGCView::viewport(const TGCViewport& NewViewport)
{
  FViewport = NewViewport;
  glViewport(FViewport.left, FViewport.top, FViewport.width, FViewport.height);
  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets a new horizontal zoom factor.
 *
 * @param Value The new zoom factor.
 */
void CGCView::zoomX(float Value)
{
  FZoomX = Value;
  FMinimalZoom = FZoomX < FZoomY ? FZoomX : FZoomY;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current horizontal zoom factor.
 *
 * @return The current horizontal zoom factor.
 */
float CGCView::zoomX(void)
{
  return FZoomX;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new vertical zoom factor.
 *
 * @param Value The new zoom factor.
 */
void CGCView::zoomY(float Value)
{
  FZoomY = Value;
  FMinimalZoom = FZoomX < FZoomY ? FZoomX : FZoomY;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Return the current vertical zomm factor.
 *
 * @return The current vertical zoom factor.
 */
float CGCView::zoomY(void)
{
  return FZoomY;
}

//----------------------------------------------------------------------------------------------------------------------

