/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_model.cpp 
 * @brief Implementation of the model that manages the visual representation in the generic canvas.
 * 
 */

#include "myx_gc.h"

#include "myx_gc_figure.h"
#include "myx_gc_model.h"
#include "myx_gc_canvas.h"
#include "myx_gc_utilities.h"

//----------------------------------------------------------------------------------------------------------------------

void CModelFigureListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (model != NULL)
    model->change(origin, reason);
}

//----------------------------------------------------------------------------------------------------------------------

void CModelFigureListener::onDestroy(CGCBase* sender)
{
  if (model != NULL)
  {
    CFigure* figure = (CFigure*) sender;
    if (figure != NULL)
      model->removeFigure(figure);
  };
}
  
//----------------------------------------------------------------------------------------------------------------------

void CModelFigureListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (model != NULL)
    model->error(origin, message);
}

//----------------- CGCModel -------------------------------------------------------------------------------------------

CGCModel::CGCModel(CGenericCanvas* Controller): CGCBase(Controller)
{
  FCanvas = Controller;
  _className = "CGCModel";
  FListener.model = this;
}

//----------------------------------------------------------------------------------------------------------------------

CGCModel::~CGCModel(void)
{
  setDestroying();

  clearFigures();
  clearLayouts();
  clearStyles();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure to the internal list and registers a listener with it.
 */
void CGCModel::addFigure(CFigure* figure)
{
  figure->addListener(&FListener);
  FFigures.push_back(figure);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if there is any pending OpenGL error (e.g. from style creation). Call is forwared to the canvas and handeled there.
 */
void CGCModel::checkError(void)
{
  FCanvas->checkError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure from the internal figures list.
 *
 * @param figure The figure to be removed.
 */
void CGCModel::removeFigure(CFigure* figure)
{
  if (!updating())
  {
    for (CFigureList::iterator iterator = FFigures.begin(); iterator != FFigures.end(); ++iterator)
    {
      if (*iterator == figure)
      {
        (*iterator)->removeListener(&FListener);
        FFigures.erase(iterator);
        break;
      };
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the model, that is, the figures defined in this model.
 */
void CGCModel::clearFigures(void)
{
  beginUpdate();
  for (CFigureList::iterator iterator = FFigures.begin(); iterator != FFigures.end(); ++iterator)
    delete *iterator;
  FFigures.clear();
  endUpdate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the figure template list (layouts).
 *
 * @note: Existing figures are not concerned by deleting the template list.
 */
void CGCModel::clearLayouts(void)
{
  beginUpdate();
  for (CLayoutList::iterator iterator = FLayouts.begin(); iterator != FLayouts.end(); ++iterator)
    delete iterator->second;
  FLayouts.clear();
  endUpdate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears all defined styles.
 *
 * @note: Deleting all styles means to leave all figures without visual representation.
 */
void CGCModel::clearStyles(void)
{
  beginUpdate();
  for (CStyleList::iterator iterator = FStyles.begin(); iterator != FStyles.end(); ++iterator)
    delete iterator->second;
  FStyles.clear();
  endUpdate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new figure and puts in into the internal list.
 *
 * @param type The name of the application defined type for which to create the figure.
 * @param layoutClass The layout class to be used (e.g. icon, detail). Also application defined.
 */
CFigure* CGCModel::createFigure(wstring type, wstring layoutClass)
{
  // createLayout only creates a new layout entry if it does not yet exist.
  CFigureTemplate* figureTemplate = createLayout(type, layoutClass);
  CFigure* figure = new CFigure(this, figureTemplate);

  return figure;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates and returns the layout entry for the given type and class.
 * If the layout entry already exists then no new instance is created, but the existing one is returned.
 *
 * @param type The type to which the layout is associated.
 * @param layoutClass An additional criterion specifying a certain layout arrangement. This value is app. specific.
 * @return The corresponding layout entry.
 */
CFigureTemplate* CGCModel::createLayout(wstring type, wstring layoutClass)
{
  CFigureTemplate* result = getLayout(type, layoutClass);

  if (result == NULL)
  {
    result = new CFigureTemplate(this, type, layoutClass);
    FLayouts.insert(CLayoutPair(type, result));
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Same as the getLayout(wstring, wstring) but for UTF-8 encoded strings.
 */
CFigureTemplate* CGCModel::getLayout(const char* type, const char* layoutClass)
{
  return getLayout(utf8ToUtf16(type), utf8ToUtf16(layoutClass));
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the layout entry for the given type if it exists.
 *
 * @param type The type to which the style is associated.
 * @param layoutClass An additional criterion specifying a certain layout arrangement. This value is app. specific.
 * @return The corresponding layout entry if it exists or NULL if not.
 */
CFigureTemplate* CGCModel::getLayout(wstring type, wstring layoutClass)
{
  CFigureTemplate* result = NULL;
  pair<CLayoutList::iterator, CLayoutList::iterator> range;

  range = FLayouts.equal_range(type);
  CLayoutList::iterator start = range.first;
  CLayoutList::iterator stop = range.second;
  while (start != stop)
  {
    if (start->second->layoutClass() == layoutClass)
    {
      result = start->second;
      break;
    };
    ++start;
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property to return.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CGCModel::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              result.s = "canvas model";
              
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "The central model of the canvas containing all styles, layouts and figures.";
              
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FCanvas;
              break;
            };
        };

        break;
      };
    case GC_CONTAINER_FIGURES:
      {
        result.type = GC_VAR_LIST;
        if (index < FFigures.size())
          result.reference = (CGCBase*) FFigures[index];

        break;
      };
    case GC_CONTAINER_STYLES:
      {
        result.type = GC_VAR_LIST;
        if (index < FStyles.size())
        {
          CStyleList::iterator iterator = FStyles.begin();
          while (index > 0)
          {
            ++iterator;
            --index;
          };
          result.reference = (CGCBase*) iterator->second;
        };

        break;
      };
    case GC_CONTAINER_LAYOUTS:
      {
        result.type = GC_VAR_LIST;
        if (index < FLayouts.size())
        {
          CLayoutList::iterator iterator = FLayouts.begin();
          while (index > 0)
          {
            ++iterator;
            --index;
          };
          result.reference = (CGCBase*) iterator->second;
        };

        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param Value The new value of the property. Automatic conversion is performed where possible.
 */
void CGCModel::property(const char* name, unsigned int index, const TGCVariant& value)
{
  // No properties to set at this point.
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the style entry for the given identifier. If there is no style with that name one is created.
 *
 * @param ID The identification of the style.
 * @return The corresponding style entry, can be NULL if ID is empty.
 */
CGCStyle* CGCModel::style(wstring ID)
{
  CGCStyle* Result = NULL;

  if (ID.size() > 0)
  {
    CStyleList::iterator iterator = FStyles.find(ID);
    if (iterator == FStyles.end())
    {
      Result = new CGCStyle(this, ID);
      FStyles[ID] = Result;
    }
    else
      Result = iterator->second;
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

