/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure_parser.cpp
 * @brief Parser for figure elements, which are converted from XML to our internal model.
 * 
 */

#include "myx_gc.h"

#include <libxml/xmlmemory.h>
#include <libxml/tree.h>

#include "myx_gc_figure_parser.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_model.h"
#include "myx_gc_const.h"
#include "myx_gc_font_manager.h"

static map<string, TFigureElementLayout> layoutLookup;
static map<string, TAlignment> alignmentLookup;

//----------------- CFigureParser --------------------------------------------------------------------------------------

CFigureParser::CFigureParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

CFigureParser::~CFigureParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if the static lookup tables are set up already. If not then it is done.
 */
void CFigureParser::checkLookupTables(void)
{

  if (layoutLookup.size() == 0)
  {
    layoutLookup["row"] = GC_LAYOUT_ROW;
    layoutLookup["column"] = GC_LAYOUT_COLUMN;
  };
  if (alignmentLookup.size() == 0)
  {
    alignmentLookup["left"] = GC_ALIGN_LEFT_TOP;
    alignmentLookup["center"] = GC_ALIGN_CENTER;
    alignmentLookup["centered"] = GC_ALIGN_CENTER;
    alignmentLookup["middle"] = GC_ALIGN_CENTER;
    alignmentLookup["right"] = GC_ALIGN_RIGHT_BOTTOM;
    alignmentLookup["top"] = GC_ALIGN_LEFT_TOP; 
    alignmentLookup["bottom"] = GC_ALIGN_RIGHT_BOTTOM; 
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given XML node and interprets it as a caption definition.
 *
 * @param node The XML node to parse.
 * @return A new caption element.
 */
CCaptionElementTemplate* CFigureParser::parseCaption(xmlNodePtr node)
{
  wstring key = utf8ToUtf16(getStringAttributeDef(node, "key", ""));
  wstring text = utf8ToUtf16(getStringAttributeDef(node, "default", ""));
  float x = getFloatAttributeDef(node, "x", 0);
  float y = getFloatAttributeDef(node, "y", 0);
  string fontFamily = getStringAttributeDef(node, "font-family", DefaultFontFamily);
  int fontSize = getIntAttributeDef(node, "font-size", DefaultFontSize);
  string fontStyle = getStringAttributeDef(node, "font-style", DefaultFontStyle);
  string fontWeightString = getStringAttributeDef(node, "font-weight", DefaultFontWeight);
  int weight = convertFontWeight(fontWeightString);
  string attribute = getStringAttributeDef(node, "horizontal-alignment", "left");
  TAlignment horizontalAlignment = alignmentLookup[attribute];
  attribute = getStringAttributeDef(node, "vertical-alignment", "top");
  TAlignment verticalAlignment = alignmentLookup[attribute];
  GLubyte color[4];
  color[3] = 255;
  GLubyte* colorPtr = color;
  if (convertColor(node, "fill", color) != 0)
    colorPtr = NULL;

  TConstraints constraints;
  constraints.maxHeight = getFloatAttributeDef(node, "max-height", -1);
  constraints.maxWidth = getFloatAttributeDef(node, "max-width", -1);
  constraints.minHeight = getFloatAttributeDef(node, "min-height", -1);
  constraints.minWidth = getFloatAttributeDef(node, "min-width", -1);

  CCaptionElementTemplate* result = new CCaptionElementTemplate(key);
  result->initialize(text, x, y, fontFamily, fontSize, weight, fontStyle, horizontalAlignment, verticalAlignment, 
    colorPtr, constraints);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses a single element and returns a new figure element instance. Can be called recursively.
 *
 * @param node The XML node to parse.
 * @return The new figure element instance created out of the element description.
 */
CFigureElementTemplate* CFigureParser::parseElement(xmlNodePtr node, CGCModel* model)
{
  wstring key = utf8ToUtf16(getStringAttributeDef(node, "key", ""));
  CFigureElementTemplate* result = new CFigureElementTemplate(key);

  string attribute = getStringAttributeDef(node, "layout", DefaultLayout);
  TFigureElementLayout layout = layoutLookup[attribute];

  string canResize = getStringAttributeDef(node, "resize", DefaultResize);
  wstring styleName;
  if (getStringAttribute(node, "template", attribute))
    styleName = utf8ToUtf16(attribute);

  TConstraints constraints;
  constraints.maxHeight = getFloatAttributeDef(node, "max-height", -1);
  constraints.maxWidth = getFloatAttributeDef(node, "max-width", -1);
  constraints.minHeight = getFloatAttributeDef(node, "min-height", -1);
  constraints.minWidth = getFloatAttributeDef(node, "min-width", -1);
  TOccurence occurence = GC_OCC_ONCE;
  if (getStringAttribute(node, "occurence", attribute))
  {
    if (attribute == "+")
      occurence = GC_OCC_ONE_OR_MORE;
    if (attribute == "*")
      occurence = GC_OCC_ZERO_OR_MORE;
  };
  result->initialize(layout, canResize == "true", model->style(styleName), constraints, occurence);

  // Now go through the child nodes. There can either be child elements or a caption node.
  xmlNodePtr run = node->children;
  while (run != NULL)
  {
    if (XML_IS(run, "caption"))
    {
      CCaptionElementTemplate* childElement = parseCaption(run);
      result->setCaption(childElement);
    }
    else
      if (XML_IS(run, "element"))
      {
        CFigureElementTemplate* childElement = parseElement(run, model);
        result->addChild(childElement);
      };  

    run = run->next;
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses a single layout definition and creates a figure template from it.
 *
 * @param definition The definition to parse.
 * @param The model class that gets the new template.
 */
void CFigureParser::parseLayoutDefinition(xmlNodePtr definition, CGCModel* model)
{
  checkLookupTables();

  wstring type = utf8ToUtf16(getStringAttributeDef(definition, "type", ""));
  wstring layoutClass = utf8ToUtf16(getStringAttributeDef(definition, "layout-class", ""));
  if (type.size() > 0)
  {
    CFigureTemplate* figureTemplate = model->createLayout(type, layoutClass);
    
    // Find first <element> node. This is our root.
    xmlNodePtr run = definition->children;
    while (run != NULL)
    {
      if (XML_IS(run, "element"))
        break;
      run = run->next;
    };

    if (run != NULL)
      figureTemplate->FContent = parseElement(run, model);
  };
}

//----------------------------------------------------------------------------------------------------------------------

