/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_texture.h 
 * @brief Implementation of a texture class.
 * 
 */

#ifndef __GC_TEXTURE_H__
#define __GC_TEXTURE_H__

#include <string>
#include <map>
#include <vector>
#ifdef __APPLE__
#include <OpenGL/gl.h>
#else
#include <GL/gl.h>
#endif
#include "myx_gc_utilities.h"

using namespace std;

//----------------------------------------------------------------------------------------------------------------------

// Default values for texturing.
const string DefaultTextureWrapMode("clamp");     
const string DefaultTextureMinFilter("nearest"); 
const string DefaultTextureMagFilter("nearest");      
const int DefaultTextureDimensions = 2;
const string DefaultTextureMode("decal");

//----------------------------------------------------------------------------------------------------------------------

class CTextureManager;

/**
 * A list of texture names each with the level-of-detail they are associated. The index in the vector is
 * also the LOD they stand for.
 */
typedef vector<string> TLODList;

// CGCTexture encapsulates a png image used to texture a figure in Generic Canvas. It loads the image data
// and manages it as well as the OpenGL properties for it.
// Note: A texture can be shared amongst several figures including its properties (e.g. minification filter).
class CGCTexture
{
private:
  CTextureManager* FManager; // The manager by which this texture is controlled.
  TLODList FLODList;         // The list of file names and their associated LOD.
  bool FLoaded;              // Set to true once the image data is uploaded to OpenGL.
  string FName;              // The identification of this texture.
  GLenum FWrapModeS;         // Specifies how textures are handled at the edges of a figure (horizontal direction).
  GLenum FWrapModeT;         // Specifies how textures are handled at the edges of a figure (vertical direction).
  GLenum FMinFilter;         // Specifies how stretching of texture data is performed when the actual screen image is smaller than the image data.
  GLenum FMagFilter;         // Specifies how stretching of texture data is performed when the actual screen image is larger than the image data.
  GLenum FTarget;            // One-dimensional or two-dimensional textures.
  GLuint FHandle;            // The OpenGL handle for this texture.
  GLint FMode;               // The mode how to apply the texture to the surface.
protected:
  void LoadTexture(void);
  TImage* LoadTextureImage(const string& Name, unsigned char*& Buffer);
public:
  CGCTexture(CTextureManager* Controller, const TLODList& LODData, const string& ID, GLenum WrapModeS, GLenum WrapModeT, 
    GLenum MinFilter, GLenum MagFilter, int Dimensions, GLenum TextureMode);
  ~CGCTexture(void);
  void ActivateTexture(void);
  void DeactivateTexture(void);
};

#ifdef _WINDOWS
#pragma warning(disable: 4251) // Disable warning about DLL interface for template classes.
#endif

// The list of textures is an associated list of names and CTexture classes.
typedef map<string, CGCTexture*> CTextures;
typedef map<string, CGCTexture*>::iterator CTextureIterator;

class CTextureManager
{
private:
  CTextures FTextures;          // A list of CTexture classes.
public:
  virtual ~CTextureManager(void);
  CGCTexture* FindTexture(const string& Name);
  CGCTexture* CreateTextureEntry(const TLODList& LODData, const string& ID, const string& WrapH, const string& WrapV,
    const string& MinificationFilterStr, const string& MagnificationFilterStr, int Dimensions, const string& Mode);
  void ClearTextures(void);
};

//----------------------------------------------------------------------------------------------------------------------

CTextureManager* textureManager(); // The one (and only) texture manager instance.

#endif // #ifdef __GC_TEXTURE_H__
