unit XGrtShell;

interface

uses
  gnugettext, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, UniCodeEditor, UniCodeConsole, ComCtrls, TntComCtrls,
  TntForms, ExtCtrls, TntExtCtrls,
  myx_public_interface, ImgList, Menus, TntMenus, StdCtrls, TntStdCtrls,
  VirtualTrees,
  myx_grt_public_interface,
  myx_grt_builtin_module_public_interface,
  PNGImage, AuxFuncs, StrUtils,
  TntDialogs;

type
  GrtLoaderState = (
    GrtLsNotInitialized,
    GrtLsInitialized,
    GrtLsInitializeFailed
  );

  TXGrtShellForm = class(TTntForm)
    TntStatusBar1: TTntStatusBar;
    CommandLinePnl: TTntPanel;
    CommandLineUCC: TUniCodeConsole;
    ObjClassPnl: TTntPanel;
    TntSplitter2: TTntSplitter;
    GRTObjTree: TVirtualStringTree;
    ObjInspectorPnl: TTntPanel;
    ObjInspectorTree: TVirtualStringTree;
    ObjInspectorLookupPnl: TTntPanel;
    ObjectTreeLookup: TTntComboBox;
    TntMainMenu1: TTntMainMenu;
    FileMI: TTntMenuItem;
    HelpMI: TTntMenuItem;
    ExitMI: TTntMenuItem;
    CommandLineSplitter: TTntSplitter;
    RefreshMI: TTntMenuItem;
    TreeImageList: TImageList;
    ViewMI: TTntMenuItem;
    N1: TTntMenuItem;
    OnlyObjectStructureMI: TTntMenuItem;
    AboutMI: TTntMenuItem;
    SaveGRTEnvironmentMI: TTntMenuItem;
    OpenGRTEnvironmentMI: TTntMenuItem;
    N2: TTntMenuItem;
    N3: TTntMenuItem;
    ConvertGRTStructFileToJavaClassesMI: TTntMenuItem;
    SidebarPnl: TTntPanel;
    SidePageControl: TTntPageControl;
    ValuesTabsheet: TTntTabSheet;
    Structs: TTntTabSheet;
    FunctionsSheet: TTntTabSheet;
    LeftSpacerPnl: TTntPanel;
    RightSpacerPnl: TTntPanel;
    TopSpacerPnl: TTntPanel;
    ModulesTree: TVirtualStringTree;
    TntPanel1: TTntPanel;
    TntSplitter3: TTntSplitter;
    ShellPopupMenu: TTntPopupMenu;
    FontSizeMI: TTntMenuItem;
    FontSizeMediumMI: TTntMenuItem;
    FontSizeLargeMI: TTntMenuItem;
    FontSizeSmallMI: TTntMenuItem;
    TntLabel1: TTntLabel;
    ModuleNameLbl: TTntLabel;
    TntLabel2: TTntLabel;
    PathLbl: TTntLabel;
    TntLabel3: TTntLabel;
    ExtendsLbl: TTntLabel;
    TypeCaptionLbl: TTntLabel;
    ModuleTypeLbl: TTntLabel;
    StructsTree: TVirtualStringTree;
    TntPanel2: TTntPanel;
    TntShape1: TTntShape;
    TntPanel3: TTntPanel;
    InspectedObjectEd: TTntEdit;
    MainPnl: TTntPanel;
    TntPanel4: TTntPanel;
    MainPageControl: TTntPageControl;
    GtrShellSheet: TTntTabSheet;
    ScriptSheet: TTntTabSheet;
    TntPanel5: TTntPanel;
    TntPanel6: TTntPanel;
    UniCodeEdit1: TUniCodeEdit;
    StructPopupMenu: TTntPopupMenu;
    StructsOrderbyNameMI: TTntMenuItem;
    StructsOrderbyHierachyMI: TTntMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);

    function CommandLineUCCIsMultilineCommand(Cmd: WideString; Key: Word;
      Shift: TShiftState): Boolean;
    procedure CommandLineUCCExecuteCommand(Cmd: WideString; Key: Word;
      Shift: TShiftState);

    procedure BuildObjTree;
    procedure GRTObjTreeInitNode(Sender: TBaseVirtualTree; ParentNode,
      Node: PVirtualNode; var InitialStates: TVirtualNodeInitStates);
    procedure GRTObjTreeInitChildren(Sender: TBaseVirtualTree;
      Node: PVirtualNode; var ChildCount: Cardinal);
    procedure RefreshMIClick(Sender: TObject);
    procedure GRTObjTreeGetText(Sender: TBaseVirtualTree;
      Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
      var CellText: WideString);
    procedure GRTObjTreeAfterCellPaint(Sender: TBaseVirtualTree;
      TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
      CellRect: TRect);
    procedure GRTObjTreeMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure GRTObjTreeGetImageIndex(Sender: TBaseVirtualTree;
      Node: PVirtualNode; Kind: TVTImageKind; Column: TColumnIndex;
      var Ghosted: Boolean; var ImageIndex: Integer);
    procedure OnlyObjectStructureMIClick(Sender: TObject);
    procedure GRTObjTreeFreeNode(Sender: TBaseVirtualTree;
      Node: PVirtualNode);
    procedure GRTObjTreeChange(Sender: TBaseVirtualTree;
      Node: PVirtualNode);

    procedure RefreshGRTObjTree;
    procedure ObjInspectorTreeGetText(Sender: TBaseVirtualTree;
      Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
      var CellText: WideString);
    procedure GRTObjTreePaintText(Sender: TBaseVirtualTree;
      const TargetCanvas: TCanvas; Node: PVirtualNode;
      Column: TColumnIndex; TextType: TVSTTextType);
    procedure ExitMIClick(Sender: TObject);
    procedure AboutMIClick(Sender: TObject);
    procedure OpenGRTEnvironmentMIClick(Sender: TObject);
    procedure SaveGRTEnvironmentMIClick(Sender: TObject);
    procedure ConvertGRTStructFileToJavaClassesMIClick(Sender: TObject);

    procedure BuildModulesTree;
    procedure ModulesTreeGetText(Sender: TBaseVirtualTree;
      Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
      var CellText: WideString);
    procedure ModulesTreeAfterCellPaint(Sender: TBaseVirtualTree;
      TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
      CellRect: TRect);
    procedure ModulesTreeInitChildren(Sender: TBaseVirtualTree;
      Node: PVirtualNode; var ChildCount: Cardinal);
    procedure ModulesTreeInitNode(Sender: TBaseVirtualTree; ParentNode,
      Node: PVirtualNode; var InitialStates: TVirtualNodeInitStates);
    procedure TntFormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure ModulesTreeDblClick(Sender: TObject);
    procedure ModulesTreeMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FontSizeSmallMIClick(Sender: TObject);
    procedure ModulesTreeChange(Sender: TBaseVirtualTree;
      Node: PVirtualNode);

    procedure BuildStructsTree;

    procedure StructsTreeAfterCellPaint(Sender: TBaseVirtualTree;
      TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
      CellRect: TRect);
    procedure StructsTreeInitNode(Sender: TBaseVirtualTree; ParentNode,
      Node: PVirtualNode; var InitialStates: TVirtualNodeInitStates);
    procedure StructsTreeInitChildren(Sender: TBaseVirtualTree;
      Node: PVirtualNode; var ChildCount: Cardinal);
    procedure StructsTreeGetText(Sender: TBaseVirtualTree;
      Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
      var CellText: WideString);
    procedure ObjectTreeLookupCloseUp(Sender: TObject);
    procedure ObjectTreeLookupKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure StructsTreeCompareNodes(Sender: TBaseVirtualTree; Node1,
      Node2: PVirtualNode; Column: TColumnIndex; var Result: Integer);
    procedure StructsOrderbyNameMIClick(Sender: TObject);
    procedure StructsOrderbyHierachyMIClick(Sender: TObject);
    procedure StructPopupMenuPopup(Sender: TObject);

  protected
    procedure OutputCommandLine(Text: WideString);

    procedure SetOnlyObjectStructure(OnlyObjectStructure: Boolean);
    procedure SetObjInspectorObject(ObjInspectorObject: Pointer);

    function GetGrt: Pointer;
    procedure SetSortStructsByHierachy(SortStructsByHierachy: Boolean);
  private
    { Private declarations }
    FGrt: Pointer;

    FTreeBtnOpenPNGImg,
      FTreeBtnClosedPNGImg: TPNGObject;

    FGrtValueDictPNGImg,
      FGrtValueListPNGImg,
      FGrtValueSimplePNGImg,
      FGrtValueStructPNGImg,
      FGrtModulePNGImg,
      FGrtFunctionPNGImg: TPNGObject;

    FOnlyObjectStructure: Boolean;

    FObjInspectorObject: Pointer;

    FJavaLoaderState,
      FLuaLoaderState,
      FBuiltinLoaderState: GrtLoaderState;

    FSortStructsByHierachy: Boolean;
  public
    { Public declarations }
    property Grt: Pointer read GetGrt;
    property OnlyObjectStructure: Boolean read FOnlyObjectStructure write SetOnlyObjectStructure;
    property ObjInspectorObject: Pointer read FObjInspectorObject write SetObjInspectorObject;
    property JavaLoaderState: GrtLoaderState read FJavaLoaderState;
    property LuaLoaderState: GrtLoaderState read FLuaLoaderState;
    property BuiltinLoaderState: GrtLoaderState read FBuiltinLoaderState;
    property SortStructsByHierachy: Boolean read FSortStructsByHierachy write SetSortStructsByHierachy;

    procedure InitializeShell(ScriptFilename: WideString = '');
  end;

  FGrtObjTreeData = ^TGRTObjTreeData;
  TGRTObjTreeData = record
    Value: Pointer;
    ValueType: MYX_GRT_VALUE_TYPE;
    Key: WideString;
    StructName: WideString;
  end;

  FGrtObjInspectorData = ^TGRTObjInspectorData;
  TGRTObjInspectorData = record
    Value: Pointer;
  end;

  GRTStructTreeDataType =
  (
    GrtStStruct, GrtStMember
  );

  FGrtStructTreeData = ^TGrtStructTreeData;
  TGrtStructTreeData = record
    DataType: GRTStructTreeDataType;
    Struct: PMYX_GRT_STRUCT;
    Member: PMYX_GRT_STRUCT_MEMBER;
  end;


  GRTModuleTreeDataType =
  (
    GrtMtModule, GrtMtFunction
  );

  FGrtModuleTreeData = ^TGRTModuleTreeData;
  TGRTModuleTreeData = record
    DataType: GRTModuleTreeDataType;
    Module: PMYX_GRT_MODULE;
    ModuleFunction: PMYX_GRT_FUNCTION;
  end;

var
  XGrtShellForm: TXGrtShellForm;

implementation

{$R *.dfm}

procedure ProcessCommandLineOutput(text: PChar; userdata: Pointer) cdecl;

var
  XGrtShellForm: TXGrtShellForm;
  {StringList: TStringList;
  i: integer;}

begin
  XGrtShellForm := userdata;

  {StringList:=TStringList.Create;
  try
    StringList.Text:=text; //AnsiReplaceStr(text, #10, #13#10);

    for i:=0 to StringList.Count-1 do
      if(Not((i=StringList.Count-1)and(StringList[i]='')))then
        MainForm.CommandLineUCC.Content.AddLine(StringList[i]);
  finally
    StringList.Free;
  end;}


  XGrtShellForm.CommandLineUCC.Content.Text :=
    XGrtShellForm.CommandLineUCC.Content.Text + #13#10 + UTF8Decode(text);
end;

// -----------------------------------------------------------------------------

function TXGrtShellForm.GetGrt: Pointer;

begin
  Result := FGrt;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.FormCreate(Sender: TObject);

begin
  MainPageControl.ActivePageIndex := 0;
  SidePageControl.ActivePageIndex := 0;
  ActiveControl := CommandLineUCC;

  FJavaLoaderState := GrtLsNotInitialized;
  FLuaLoaderState := GrtLsNotInitialized;
  FBuiltinLoaderState := GrtLsNotInitialized;

  FTreeBtnOpenPNGImg := LoadPNGImageFromResource('tree_button_open');
  FTreeBtnClosedPNGImg := LoadPNGImageFromResource('tree_button_closed');

  FGrtValueDictPNGImg := LoadPNGImageFromResource('grt_value_dict');
  FGrtValueListPNGImg := LoadPNGImageFromResource('grt_value_list');
  FGrtValueSimplePNGImg := LoadPNGImageFromResource('grt_value_simple');
  FGrtValueStructPNGImg := LoadPNGImageFromResource('grt_value_struct');
  FGrtModulePNGImg := LoadPNGImageFromResource('grt_module');
  FGrtFunctionPNGImg := LoadPNGImageFromResource('grt_function');

  FOnlyObjectStructure := True;

  FObjInspectorObject := nil;

  FSortStructsByHierachy := False;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.FormDestroy(Sender: TObject);

begin
  if FGrt<>nil then
    myx_grt_finalize(FGrt);

  FTreeBtnOpenPNGImg.Free;
  FTreeBtnClosedPNGImg.Free;

  FGrtValueDictPNGImg.Free;
  FGrtValueListPNGImg.Free;
  FGrtValueSimplePNGImg.Free;
  FGrtValueStructPNGImg.Free;
  FGrtModulePNGImg.Free;
  FGrtFunctionPNGImg.Free;

end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.FormClose(Sender: TObject; var Action: TCloseAction);

begin
  //
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.InitializeShell(ScriptFilename: WideString);

var
  Loader: Pointer;
  Error: MYX_GRT_ERROR;
  I: integer;

begin
  //Initialize GRT
  FGrt := myx_grt_initialize();

  myx_grt_set_output_callback(FGrt, self, @ProcessCommandLineOutput);

  myx_grt_shell_print_welcome(FGrt);

  OutputCommandLine('');

  OutputCommandLine(_('GRT system initialized...'));

  //Load Structs
  //Scan for Java plugins
  OutputCommandLine(_('Scanning for struct definitions in ./xml ...'));
  I := myx_grt_scan_for_structs(FGrt, './xml', @Error);
  if (Error<>MYX_GRT_NO_ERROR) then
    OutputCommandLine(Format(
      _('Error while scanning for struct definitions (%d).'),
      [Ord(Error)]))
  else
    OutputCommandLine(Format(
      _('Registered %d struct definition files.'),
      [I]));


  //Load Modules

  //Load building modules
  OutputCommandLine(_('Initializing builtin modules...'));
  FBuiltinLoaderState := GrtLsInitializeFailed;
  Error := myx_register_mysql_grt_module(FGrt);
  if (Error=MYX_GRT_NO_ERROR) then
    FBuiltinLoaderState := GrtLsInitialized;

  //Load java module
  OutputCommandLine(_('Initializing Java loader...'));
  FJavaLoaderState := GrtLsInitializeFailed;
  Loader := myx_java_init_loader(FGrt, 'java', @Error);
  if (Loader<>nil) then
  begin
    if (myx_grt_register_module_loader(FGrt, Loader)<>MYX_GRT_NO_ERROR) then
      OutputCommandLine(_('Could not register Java module loader.'))
    else
      FJavaLoaderState := GrtLsInitialized;
  end
  else
    OutputCommandLine(Format(
      _('Error initializing Java module loader (%d)'),
      [Ord(error)]));

  //Scan for Java plugins
  OutputCommandLine(_('Scanning for java plugins in ./java/com/mysql/grt/modules ...'));
  I := myx_grt_scan_for_modules(FGrt, './java/com/mysql/grt/modules', @Error);
  if (Error<>MYX_GRT_NO_ERROR) then
    OutputCommandLine(Format(
      _('Error while scanning for java modules (%d).'),
      [Ord(Error)]))
  else
    OutputCommandLine(Format(
      _('Initialized %d modules.'),
      [I]));


  //Load lua module
  OutputCommandLine(_('Initializing Lua loader...'));
  FLuaLoaderState := GrtLsInitializeFailed;
  Loader := myx_lua_init_loader(FGrt, @Error);
  if (Loader<>nil) then
  begin
    if (myx_grt_register_module_loader(FGrt, Loader)<>MYX_GRT_NO_ERROR) then
      OutputCommandLine(_('Could not register Lua module loader.'))
    else
      FLuaLoaderState := GrtLsInitialized;
  end
  else
    OutputCommandLine(Format(
      _('Error initializing Lua module loader (%d)'),
      [Ord(error)]));

  OutputCommandLine('');

  myx_grt_init_lua_shell(FGrt);

  if (ScriptFilename<>'') then
    myx_grt_lua_shell_run_file(FGrt, ScriptFilename);

  CommandLineUCC.ConsolePrompt := myx_grt_lua_shell_get_prompt(FGrt);

  OutputCommandLine('');
  CommandLineUCC.PrepareNextConsoleCommand;

  RefreshMIClick(self);
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.OutputCommandLine(Text: WideString);

begin
  CommandLineUCC.Content.AddLine(Text);
end;

// -----------------------------------------------------------------------------

function TXGrtShellForm.CommandLineUCCIsMultilineCommand(Cmd: WideString;
  Key: Word; Shift: TShiftState): Boolean;

begin
  Result := False;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.CommandLineUCCExecuteCommand(Cmd: WideString;
  Key: Word; Shift: TShiftState);

var
  StartTime, EndTime: TDateTime;

begin
  StartTime := Now;

  if (myx_grt_lua_shell_execute(FGrt, Cmd)=MYX_GRT_SHELL_COMMAND_EXIT) then
    Close;

  EndTime := Now;

  if (Shift = [ssCtrl])then
    CommandLineUCC.Content.AddLine(_('Executed in ') +
      FormatDateTime('n:ss:zzz', EndTime - StartTime) + '.');

  CommandLineUCC.PrepareNextConsoleCommand;

  CommandLineUCC.ConsolePrompt := myx_grt_lua_shell_get_prompt(FGrt);
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.RefreshMIClick(Sender: TObject);
begin
  BuildObjTree;
  BuildModulesTree;
  BuildStructsTree;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.BuildObjTree;

begin
  if (FGrt=nil) then
    Exit;

  InspectedObjectEd.Text := '';
  ObjInspectorTree.RootNodeCount := 0;

  GRTObjTree.BeginUpdate;
  try
    GRTObjTree.Clear;
    GRTObjTree.NodeDataSize:=sizeof(TGRTObjTreeData);

    GRTObjTree.RootNodeCount:=1;
  finally
    GRTObjTree.EndUpdate;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeAfterCellPaint(Sender: TBaseVirtualTree;
  TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
  CellRect: TRect);

var
  NodeData: FGrtObjTreeData;
  TxtRect: TRect;
  x: integer;
  Icon: TPNGObject;

begin
  if (Column=0) then
  begin
    NodeData := Sender.GetNodeData(Node);

    TxtRect:=Sender.GetDisplayRect(Node, Column, True);

    x:=TxtRect.Left-Sender.OffsetX;

    //Draw > / v
    if (Node.ChildCount>0) or (vsHasChildren in Node.States) then
    begin
      if(Sender.Expanded[Node])then
        FTreeBtnOpenPNGImg.Draw(TargetCanvas,
          Rect(x-16-12, CellRect.Top+4, x-16-4, CellRect.Top+16+4))
      else
        FTreeBtnClosedPNGImg.Draw(TargetCanvas,
          Rect(x-16-12, CellRect.Top+4, x-16-4, CellRect.Top+16+4))
    end;


    if (NodeData.ValueType=MYX_LIST_VALUE) then
      Icon := FGrtValueListPNGImg
    else
      if (NodeData.ValueType=MYX_DICT_VALUE) and
        (NodeData.StructName='') then
        Icon := FGrtValueDictPNGImg
      else
        if (NodeData.ValueType=MYX_DICT_VALUE) and
          (NodeData.StructName<>'') then
          Icon := FGrtValueStructPNGImg
        else
          Icon := FGrtValueSimplePNGImg;

    Icon.Draw(TargetCanvas,
        Rect(x-16, CellRect.Top+1, x, CellRect.Top+1+16));
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

var HitInfo: THitInfo;
  TxtRect: TRect;
  xpos: integer;

begin
  if(Sender.InheritsFrom(TBaseVirtualTree))then
  begin
    TBaseVirtualTree(Sender).GetHitTestInfoAt(X, Y, True, HitInfo);

    if(HitInfo.HitNode<>nil)then
    begin
      TxtRect:=TBaseVirtualTree(Sender).GetDisplayRect(
        HitInfo.HitNode, -1, True);

      xpos:=TxtRect.Left-TBaseVirtualTree(Sender).OffsetX;

      if(X>xpos-16-4)and(X<xpos+2)and
        ((HitInfo.HitNode.ChildCount>0)or(vsHasChildren in HitInfo.HitNode.States))then
      begin
        TBaseVirtualTree(Sender).Expanded[HitInfo.HitNode]:=
          Not(TBaseVirtualTree(Sender).Expanded[HitInfo.HitNode]);
      end;
    end
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeGetImageIndex(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Kind: TVTImageKind; Column: TColumnIndex;
  var Ghosted: Boolean; var ImageIndex: Integer);

begin
  if(Column=0)then
    ImageIndex:=0;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.SetOnlyObjectStructure(OnlyObjectStructure: Boolean);

var
  Col: TVirtualTreeColumn;

begin
  FOnlyObjectStructure := OnlyObjectStructure;

  if (FOnlyObjectStructure) then
  begin
    if (GRTObjTree.Header.Columns.Count>1) then
      GRTObjTree.Header.Columns.Delete(1);
  end
  else
  begin
    if (GRTObjTree.Header.Columns.Count=1) then
    begin
      Col:=GRTObjTree.Header.Columns.Add;
      Col.Text:=_('Values');
      Col.Width:=150;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.OnlyObjectStructureMIClick(Sender: TObject);

begin
  OnlyObjectStructureMI.Checked := Not(OnlyObjectStructureMI.Checked);

  OnlyObjectStructure := OnlyObjectStructureMI.Checked;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeInitNode(Sender: TBaseVirtualTree;
  ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

var
  NodeData, ParentNodeData: FGrtObjTreeData;
  ParentNodeData_Type: MYX_GRT_VALUE_TYPE;
  PName: PChar;

begin
  NodeData := Sender.GetNodeData(Node);

  if ParentNode=nil then
  begin
    if (ObjectTreeLookup.ItemIndex=0) then
    begin
      NodeData.Key := _('root');
      NodeData.Value := myx_grt_get_root(FGrt);
    end
    else
    begin
      NodeData.Key := ObjectTreeLookup.Text;
      NodeData.Value := myx_grt_lua_shell_get_global_var(
        FGrt, ObjectTreeLookup.Text);

      if (NodeData.Value = nil) then
      begin
        GRTObjTree.RootNodeCount := 0;
        Exit;
      end;
    end;

    myx_grt_value_retain(NodeData.Value);
    NodeData.ValueType := myx_grt_value_get_type(NodeData.Value);

    //The root value must be a dict
    if NodeData.ValueType<>MYX_DICT_VALUE then
      raise EInOutError.Create(_('The root node has to be a dict value.'));

    InitialStates := InitialStates+[ivsExpanded];
  end
  else
  begin
    ParentNodeData := Sender.GetNodeData(ParentNode);
    ParentNodeData_Type := myx_grt_value_get_type(ParentNodeData.Value);

    if (ParentNodeData_Type=MYX_LIST_VALUE) then
    begin
      //Get value and retain it
      NodeData.Value := myx_grt_list_get_item(ParentNodeData.Value, Node.Index);
      myx_grt_value_retain(NodeData.Value);

      NodeData.ValueType := myx_grt_value_get_type(NodeData.Value);

      if (NodeData.ValueType = MYX_DICT_VALUE) then
      begin
        PName := _myx_grt_dict_name_item_as_string(NodeData.Value);
        if (PName<>nil) then
          NodeData.Key := UTF8Decode(PName)
        else
          NodeData.Key := '';
      end;
    end
    else
      if (ParentNodeData_Type=MYX_DICT_VALUE) then
      begin
        //Get value and retain it
        NodeData.Value := myx_grt_dict_item_value_by_index(
          ParentNodeData.Value, Node.Index);
        myx_grt_value_retain(NodeData.Value);

        NodeData.Key := UTF8Decode(_myx_grt_dict_item_key_by_index(
          ParentNodeData.Value, Node.Index));
        NodeData.ValueType := myx_grt_value_get_type(NodeData.Value);
      end;

    if ((NodeData.ValueType=MYX_LIST_VALUE) or
      (NodeData.ValueType=MYX_DICT_VALUE)) then
    begin
      PName := _myx_grt_value_struct_name(NodeData.Value);
      if (PName<>nil) then
        NodeData.StructName := UTF8Decode(PName)
      else
        NodeData.StructName := '';
    end;
  end;

  if NodeData.ValueType=MYX_DICT_VALUE then
  begin
    if myx_grt_dict_item_num(NodeData.Value)>0 then
      InitialStates := InitialStates+[ivsHasChildren];
  end
  else
    if NodeData.ValueType=MYX_LIST_VALUE then
      InitialStates := InitialStates+[ivsHasChildren];
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeInitChildren(Sender: TBaseVirtualTree;
  Node: PVirtualNode; var ChildCount: Cardinal);

var
  NodeData: FGrtObjTreeData;

begin
  NodeData := Sender.GetNodeData(Node);

  if (NodeData.ValueType=MYX_DICT_VALUE) then
    ChildCount := myx_grt_dict_item_num(NodeData.Value)
  else
    if (NodeData.ValueType=MYX_LIST_VALUE) then
      ChildCount := myx_grt_list_item_num(NodeData.Value);
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeGetText(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
  var CellText: WideString);

var
  NodeData: FGrtObjTreeData;

begin
  NodeData := Sender.GetNodeData(Node);

  if (Column=0) and (TextType=ttNormal) then
  begin
     CellText := NodeData.Key;
  end
  else
    if (Column=0) and (TextType=ttStatic) then
  begin
    CellText := UTF8Decode(_myx_get_value_type_as_string(NodeData.ValueType));

    if (NodeData.ValueType=MYX_LIST_VALUE) then
    begin
      CellText := CellText + ' [' +
        UTF8Decode(_myx_get_value_type_as_string(myx_grt_list_content_type(NodeData.Value)));

      if (NodeData.StructName<>'') then
        CellText := CellText + ': '+NodeData.StructName;

      CellText := CellText + ']';
    end
    else
      if (NodeData.ValueType=MYX_DICT_VALUE) and
        (NodeData.StructName<>'') then
        CellText := CellText + ': '+NodeData.StructName;
  end
  else if (Column=1) and (TextType=ttNormal) then
  begin
    case NodeData.ValueType of
      MYX_INT_VALUE:
        CellText := IntToStr(myx_grt_value_as_int(NodeData.Value));
      MYX_REAL_VALUE:
        CellText := FormatFloat('###,###,##0.00',
          (myx_grt_value_as_double(NodeData.Value)));
      MYX_STRING_VALUE:
        CellText := UTF8Decode(_myx_grt_value_as_string(NodeData.Value));
    else
      CellText := '';
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeFreeNode(Sender: TBaseVirtualTree;
  Node: PVirtualNode);

var
  NodeData: FGrtObjTreeData;

begin
  if (Node<>Sender.RootNode) then
  begin
    NodeData := Sender.GetNodeData(Node);

    if (NodeData.Value<>nil) then
      myx_grt_value_release(NodeData.Value);
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreeChange(Sender: TBaseVirtualTree;
  Node: PVirtualNode);

var
  NodeData: FGrtObjTreeData;

begin
  NodeData := Sender.GetNodeData(Node);

  if (NodeData<>nil) and
    ((NodeData.ValueType=MYX_DICT_VALUE) or
    (NodeData.ValueType=MYX_LIST_VALUE)) then
  begin
    ObjInspectorObject := NodeData.Value;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.RefreshGRTObjTree;

var
  PName: PChar;
  ValueType: MYX_GRT_VALUE_TYPE;
  DictStructName: WideString;

begin
  if (FObjInspectorObject<>nil) then
  begin
    ValueType := myx_grt_value_get_type(FObjInspectorObject);

    if ((ValueType = MYX_LIST_VALUE) or
      (ValueType = MYX_DICT_VALUE)) then
    begin
      //Get struct name if there is any
      PName := _myx_grt_value_struct_name(FObjInspectorObject);
      if (PName<>nil) then
        DictStructName := UTF8Decode(PName);

      if (ValueType = MYX_DICT_VALUE) then
      begin
        //Get name
        PName := _myx_grt_dict_name_item_as_string(FObjInspectorObject);
        if (PName<>nil) then
          InspectedObjectEd.Text := UTF8Decode(PName) + ' (' +
            DictStructName + ')'
        else
          InspectedObjectEd.Text := 'DICT';
      end
      else
      begin
        InspectedObjectEd.Text := '[' +
          UTF8Decode(_myx_get_value_type_as_string(myx_grt_list_content_type(FObjInspectorObject)));

        if (DictStructName<>'') then
          InspectedObjectEd.Text := InspectedObjectEd.Text +
            ': '+DictStructName;

        InspectedObjectEd.Text := InspectedObjectEd.Text +
          ']';
      end;



      ObjInspectorTree.NodeDataSize:=sizeof(TGRTObjInspectorData);

      ObjInspectorTree.BeginUpdate;
      try
        ObjInspectorTree.Clear;

        if (ValueType = MYX_DICT_VALUE) then
          ObjInspectorTree.RootNodeCount := myx_grt_dict_item_num(ObjInspectorObject)
        else
          ObjInspectorTree.RootNodeCount := myx_grt_list_item_num(ObjInspectorObject);
      finally
        ObjInspectorTree.EndUpdate;
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.SetObjInspectorObject(ObjInspectorObject: Pointer);

begin
  FObjInspectorObject := ObjInspectorObject;

  RefreshGRTObjTree;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ObjInspectorTreeGetText(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
  var CellText: WideString);

var
  ValueType: MYX_GRT_VALUE_TYPE;

begin
  if (FObjInspectorObject<>nil) then
  begin
    ValueType := myx_grt_value_get_type(FObjInspectorObject);

    if (ValueType = MYX_LIST_VALUE) then
    begin
      if (Integer(Node.Index)<myx_grt_list_item_num(FObjInspectorObject)) then
      begin
        if Column=0 then
          CellText := '[' + IntToStr(Node.Index) + ']'
        else
          CellText := myx_grt_value_format_as_string(
            myx_grt_list_get_item(FObjInspectorObject, Node.Index));
      end;
    end
    else if (ValueType = MYX_DICT_VALUE) then
    begin
      if (Integer(Node.Index)<myx_grt_dict_item_num(FObjInspectorObject)) then
      begin
        if Column=0 then
          CellText := UTF8Decode(_myx_grt_dict_item_key_by_index(FObjInspectorObject, Node.Index))
        else
          CellText := myx_grt_value_format_as_string(
            myx_grt_dict_item_value_by_index(FObjInspectorObject, Node.Index));
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.GRTObjTreePaintText(Sender: TBaseVirtualTree;
  const TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
  TextType: TVSTTextType);

begin
  if (TextType=ttStatic) then
    TargetCanvas.Font.Color := clGray;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ExitMIClick(Sender: TObject);

begin
  Close;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.AboutMIClick(Sender: TObject);

begin
  ShowModalDialog(_('MySQL GRT Environment GUI Shell'),
    _('MySQL Generic Runtime Environment GUI Shell 1.0 alpha'),
    myx_mtInformation, _('OK'));
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.OpenGRTEnvironmentMIClick(Sender: TObject);

var
  OpenDialog: TTntOpenDialog;

begin
  OpenDialog := TTntOpenDialog.Create(self);
  try
    OpenDialog.Filter := _('GRT Environment')+' (*.xml)|*.xml|'+
      _('Any file')+' (*.*)|*.*';

    if (OpenDialog.Execute) then
    begin
      myx_grt_lua_shell_execute(FGrt, 'setobj("/", load("'+
        OpenDialog.FileName+'"))'#13#10);
    end;
  finally
    OpenDialog.Free;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.SaveGRTEnvironmentMIClick(Sender: TObject);
begin
  //
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ConvertGRTStructFileToJavaClassesMIClick(Sender: TObject);

var
  PStructs: Pointer;
  Path: WideString;
  Error: MYX_GRT_ERROR;
  OpenDialog: TTntOpenDialog;
  Package: WideString;

begin
  OpenDialog := TTntOpenDialog.Create(self);

  try
    OpenDialog.Filter := _('GRT Struct Definition')+' (*.xml)|*.xml|'+
      _('Any file')+' (*.*)|*.*';

    if (OpenDialog.Execute) then
    begin
      Package := ChangeFileExt(ExtractFileName(OpenDialog.FileName), '');

      if (Copy(Package, 1, 8)<>'structs_') then
        raise EInOutError.Create('The struct definition file must start with structs_');

      Package := AnsiReplaceStr(Copy(Package, 9, Length(Package)), '_', '.');

      PStructs := myx_grt_struct_load_list(OpenDialog.FileName, Addr(Error));
      try
        Path := ExtractFilePath(Application.ExeName)+'java\';
        ForceDirectories(Path);

        myx_grt_struct_export_java_classes(PStructs, Package, Path);
      finally
        myx_grt_structs_free(PStructs);
      end;
    end;
  finally
    OpenDialog.Free;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.BuildModulesTree;

begin
  if (FGrt=nil) then
    Exit;

  ModulesTree.BeginUpdate;
  try
    ModulesTree.Clear;
    ModulesTree.NodeDataSize:=sizeof(TGRTModuleTreeData);

    ModulesTree.RootNodeCount := myx_grt_module_get_count(FGrt);
  finally
    ModulesTree.EndUpdate;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeGetText(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
  var CellText: WideString);

var
  NodeData: FGrtModuleTreeData;
  ParamTypes,
    ReturnType: WideString;

begin
  NodeData := Sender.GetNodeData(Node);

  if (NodeData.DataType = GrtMtModule) then
  begin
    if (TextType = ttNormal) then
      CellText := NodeData.Module.name;
  end
  else
    if (NodeData.DataType = GrtMtFunction) then
    begin
      if (TextType = ttNormal) then
      begin
        CellText := NodeData.ModuleFunction.name;

        ParamTypes := Trim(myx_grt_module_function_get_params(NodeData.ModuleFunction));

        if (ParamTypes<>'') then
          CellText := CellText + ' '+ParamTypes
        else
          CellText := CellText + ' ()';

        ReturnType := Trim(myx_grt_module_function_get_return_type(NodeData.ModuleFunction));
        if (ReturnType<>'') then
          CellText := CellText + ': '+ReturnType;

        CellText := CellText + ';';
      end;
    end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeAfterCellPaint(Sender: TBaseVirtualTree;
  TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
  CellRect: TRect);

var
  NodeData: FGrtModuleTreeData;
  TxtRect: TRect;
  x: integer;
  Icon: TPNGObject;

begin
  if(Column=0)then
  begin
    NodeData := Sender.GetNodeData(Node);

    TxtRect:=Sender.GetDisplayRect(Node, Column, True);

    x:=TxtRect.Left-Sender.OffsetX;

    //Draw > / v
    if (Node.ChildCount>0) or (vsHasChildren in Node.States) then
    begin
      if(Sender.Expanded[Node])then
        FTreeBtnOpenPNGImg.Draw(TargetCanvas,
          Rect(x-16-12, CellRect.Top+4, x-16-4, CellRect.Top+16+4))
      else
        FTreeBtnClosedPNGImg.Draw(TargetCanvas,
          Rect(x-16-12, CellRect.Top+4, x-16-4, CellRect.Top+16+4))
    end;

    if (NodeData.DataType = GrtMtModule) then
      Icon := FGrtModulePNGImg
    else
      Icon := FGrtFunctionPNGImg;

    Icon.Draw(TargetCanvas,
        Rect(x-16, CellRect.Top+1, x, CellRect.Top+1+16));
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeInitNode(Sender: TBaseVirtualTree;
  ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

var
  NodeData, ParentNodeData: FGrtModuleTreeData;

begin
  NodeData := Sender.GetNodeData(Node);

  if (ParentNode = nil) then
  begin
    NodeData.DataType := GrtMtModule;

    NodeData.Module := myx_grt_module_get_by_index(FGrt, Node.Index);

    Include(Node.States, vsHasChildren);
  end
  else
  begin
    ParentNodeData := Sender.GetNodeData(ParentNode);

    if (ParentNodeData.DataType = GrtMtModule) then
    begin
      NodeData.DataType := GrtMtFunction;

      NodeData.ModuleFunction := myx_grt_module_function_get_by_index(
        ParentNodeData.Module, Node.Index);
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeInitChildren(Sender: TBaseVirtualTree;
  Node: PVirtualNode; var ChildCount: Cardinal);

var
  NodeData: FGrtModuleTreeData;

begin
  NodeData := Sender.GetNodeData(Node);

  if (NodeData.DataType = GrtMtModule) then
    ChildCount := myx_grt_module_function_get_count(NodeData.Module)
  else
    ChildCount := 0;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.TntFormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);

begin
  if (Key = VK_F5) then
    RefreshMIClick(self);
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeDblClick(Sender: TObject);

var NodeData, ParentNodeData: FGrtModuleTreeData;
  Cmd: WideString;

begin
  if (Sender = ModulesTree) then
  begin
    NodeData := ModulesTree.GetNodeData(ModulesTree.FocusedNode);

    if (NodeData.DataType = GrtMtFunction) then
    begin
      ParentNodeData := ModulesTree.GetNodeData(
        ModulesTree.FocusedNode.Parent);

      Cmd := CommandLineUCC.ConsoleCommand;
      if (Copy(Cmd, Length(Cmd)-1, 2) = #13#10) then
        Cmd := Copy(Cmd, 1, Length(Cmd)-2);

      Cmd := Cmd +
        ParentNodeData.Module.name + ':' +
        NodeData.ModuleFunction.name + '(';

      CommandLineUCC.ConsoleCommand := Cmd;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

begin
  CommandLineUCC.SetFocus;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.FontSizeSmallMIClick(Sender: TObject);

var
  NewFontHeight: Integer;

begin
  if (Sender is TTntMenuItem) then
  begin
    TTntMenuItem(Sender).Checked := True;

    NewFontHeight := TTntMenuItem(Sender).Tag * -1;

    if (NewFontHeight<=-10) then
      CommandLineUCC.Font.Height := NewFontHeight;

    if (NewFontHeight<=-11) then
      CommandLineUCC.Font.Style := [fsBold]
    else
      CommandLineUCC.Font.Style := [];
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ModulesTreeChange(Sender: TBaseVirtualTree;
  Node: PVirtualNode);

var
  NodeData: FGrtModuleTreeData;
  Module: PMYX_GRT_MODULE;

begin
  if (Node<>nil) then
  begin
    NodeData := Sender.GetNodeData(Node);

    if (NodeData.DataType = GrtMtModule) then
      Module := NodeData.Module
    else
      Module := NodeData.ModuleFunction.module;

    ModuleNameLbl.Caption := Module.name;
    PathLbl.Caption := Module.path;
    ExtendsLbl.Caption := Module.extends;
    case myx_grt_module_get_type(Module) of
      MYX_BUILTIN_MODULE_TYPE:
        ModuleTypeLbl.Caption := 'C Module';
      MYX_JAVA_MODULE_TYPE:
        ModuleTypeLbl.Caption := 'Java Module';
      MYX_LUA_MODULE_TYPE:
        ModuleTypeLbl.Caption := 'Lua Module';
      MYX_PYTHON_MODULE_TYPE:
        ModuleTypeLbl.Caption := 'Python Module';
    else
      ModuleTypeLbl.Caption := '-';
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.BuildStructsTree;

begin
  if (FGrt=nil) then
    Exit;

  StructsTree.BeginUpdate;
  try
    StructsTree.Clear;
    StructsTree.NodeDataSize:=sizeof(TGRTStructTreeData);

    if (SortStructsByHierachy) then
      StructsTree.RootNodeCount := myx_grt_struct_get_child_count(FGrt, '')
    else
      StructsTree.RootNodeCount := myx_grt_struct_get_count(FGrt);
  finally
    StructsTree.EndUpdate;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsTreeAfterCellPaint(Sender: TBaseVirtualTree;
  TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
  CellRect: TRect);
  
var
  NodeData: FGrtStructTreeData;
  TxtRect: TRect;
  x: integer;
  Icon: TPNGObject;

begin
  if(Column=0)then
  begin
    NodeData := Sender.GetNodeData(Node);

    TxtRect:=Sender.GetDisplayRect(Node, Column, True);

    x:=TxtRect.Left-Sender.OffsetX;

    //Draw > / v
    if (Node.ChildCount>0) or (vsHasChildren in Node.States) then
    begin
      if(Sender.Expanded[Node])then
        FTreeBtnOpenPNGImg.Draw(TargetCanvas,
          Rect(x-16-12, CellRect.Top+4, x-16-4, CellRect.Top+16+4))
      else
        FTreeBtnClosedPNGImg.Draw(TargetCanvas,
          Rect(x-16-12, CellRect.Top+4, x-16-4, CellRect.Top+16+4))
    end;

    if (NodeData.DataType = GrtStStruct) then
      Icon := FGrtValueStructPNGImg
    else
      Icon := FGrtValueSimplePNGImg;

    Icon.Draw(TargetCanvas,
      Rect(x-16, CellRect.Top+1, x, CellRect.Top+1+16));
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsTreeInitNode(Sender: TBaseVirtualTree;
  ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

var
  NodeData, ParentNodeData: FGrtStructTreeData;
  ParentName: WideString;

begin
  NodeData := Sender.GetNodeData(Node);

  if (SortStructsByHierachy) then
  begin
    NodeData.DataType := GrtStStruct;

    if (ParentNode = nil) then
    begin
      NodeData.Struct := myx_grt_struct_get_child_by_index(
        FGrt, '', Node.Index);

      if (myx_grt_struct_get_child_count(FGrt, '')>0) then
        Include(InitialStates, ivsHasChildren);
    end
    else
    begin
      ParentNodeData := Sender.GetNodeData(ParentNode);

      if (ParentNodeData.DataType = GrtStStruct) then
      begin
        ParentName := Utf8Decode(
          _myx_grt_struct_get_name(ParentNodeData.Struct));

        NodeData.Struct := myx_grt_struct_get_child_by_index(
          FGrt, ParentName, Node.Index);

        if (myx_grt_struct_get_child_count(FGrt, ParentName)>0) then
          Include(InitialStates, ivsHasChildren);
      end;
    end;
  end
  else
  begin
    if (ParentNode = nil) then
    begin
      NodeData.DataType := GrtStStruct;
      NodeData.Struct := myx_grt_struct_get_by_index(FGrt, Node.Index);

      Include(InitialStates, ivsHasChildren);
    end
    else
    begin
      ParentNodeData := Sender.GetNodeData(ParentNode);

      if (ParentNodeData.DataType = GrtStStruct) then
      begin
        NodeData.DataType := GrtStMember;
        NodeData.Member := myx_grt_struct_get_member_by_index(
          ParentNodeData.Struct, Node.Index);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsTreeInitChildren(Sender: TBaseVirtualTree;
  Node: PVirtualNode; var ChildCount: Cardinal);

var
  NodeData: FGrtStructTreeData;
  Name: WideString;

begin
  NodeData := Sender.GetNodeData(Node);

  if (SortStructsByHierachy) then
  begin
    Name := Utf8Decode(_myx_grt_struct_get_name(NodeData.Struct));
    ChildCount := myx_grt_struct_get_child_count(FGrt, Name);
  end
  else
  begin
    if (NodeData.DataType = GrtStStruct) then
      ChildCount := myx_grt_struct_get_member_count(NodeData.Struct)
    else
      ChildCount := 0;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsTreeGetText(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
  var CellText: WideString);

var
  NodeData: FGrtStructTreeData;

begin
  NodeData := Sender.GetNodeData(Node);

  if (NodeData.DataType = GrtStStruct) then
  begin
    if (TextType = ttNormal) then
      CellText := Utf8Decode(_myx_grt_struct_get_name(NodeData.Struct))
  end
  else
  begin
    if (TextType = ttNormal) then
      CellText := Utf8Decode(_myx_grt_struct_get_member_name(NodeData.Member))
    else
      CellText := Utf8Decode(_myx_get_value_type_as_string(
        myx_grt_struct_get_member_type(NodeData.Member)));
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ObjectTreeLookupCloseUp(Sender: TObject);

begin
  BuildObjTree;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.ObjectTreeLookupKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);

var
  Index: Integer;

begin
  if (Key = VK_RETURN) then
  begin
    Index := ObjectTreeLookup.Items.IndexOf(ObjectTreeLookup.Text);
    if (Index = -1) then
      Index := ObjectTreeLookup.Items.Add(ObjectTreeLookup.Text);

    ObjectTreeLookup.ItemIndex := Index;
    BuildObjTree;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsTreeCompareNodes(Sender: TBaseVirtualTree;
  Node1, Node2: PVirtualNode; Column: TColumnIndex; var Result: Integer);

var
  NodeData1, NodeData2: FGrtStructTreeData;
  DotCount1, DotCount2: Integer;
  Name1, Name2: WideString;

begin
  NodeData1 := Sender.GetNodeData(Node1);
  NodeData2 := Sender.GetNodeData(Node2);

  if (NodeData1.DataType=GrtStStruct) and
    (NodeData2.DataType=GrtStStruct) then
  begin
    Name1 := Utf8Decode(_myx_grt_struct_get_name(NodeData1.Struct));
    Name2 := Utf8Decode(_myx_grt_struct_get_name(NodeData2.Struct));

    DotCount1 := GetSubstringCount('.', Name1);
    DotCount2 := GetSubstringCount('.', Name2);

    if (DotCount1<>DotCount2) then
      Result := DotCount1-DotCount2
    else
      Result := WideCompareText(Name1, Name2);
  end
  else
    if (NodeData1.DataType=GrtStMember) and
      (NodeData2.DataType=GrtStMember) then
    begin
      Name1 := Utf8Decode(_myx_grt_struct_get_member_name(NodeData1.Member));
      Name2 := Utf8Decode(_myx_grt_struct_get_member_name(NodeData2.Member));

      Result := WideCompareText(Name1, Name2);
    end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.SetSortStructsByHierachy(SortStructsByHierachy: Boolean);

begin
  if (FSortStructsByHierachy<>SortStructsByHierachy) then
  begin
    FSortStructsByHierachy := SortStructsByHierachy;

    BuildStructsTree;
  end;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsOrderbyNameMIClick(Sender: TObject);
begin
  SortStructsByHierachy := False;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructsOrderbyHierachyMIClick(Sender: TObject);
begin
  SortStructsByHierachy := True;
end;

// -----------------------------------------------------------------------------

procedure TXGrtShellForm.StructPopupMenuPopup(Sender: TObject);
begin
  if (SortStructsByHierachy) then
    StructsOrderbyHierachyMI.Checked := True
  else
    StructsOrderbyNameMI.Checked := True;
end;

end.
