# weak.py
#
#   Copyright (C) 2004 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# This file contains code to work around a leak in the Python weak
# reference abstraction.

import weakref

def WeakCallableRef(f, callback=lambda *args:None):
    """This works around a Python bug: bound methods are instantiated
    as tuple-like objects that point to the object, so a weak
    reference to a bound method dies instantly.

    A WeakCallableRef holds a weak reference to both an object and its
    method, and \"dies\" at exactly the same time as the object.  The
    routine callback is invoked when the object dies."""

    class WeakWrapper:
        """This wraps the transient method object."""
        def __init__(self, obj, method):
            self.obj=weakref.ref(obj, self.__kill)
            self.method=method

        def __kill(self, deadman):
            self.method=lambda *args:None
            callback(self)

        def __call__(self, *args):
            obj=self.obj()

            if obj == None:
                return None
            else:
                return lambda *args:apply(self.method, (obj,)+args)

    # Is it a method?
    try:
        f.im_self
    except AttributeError:
        # No, so the bug doesn't apply; use a standard weak reference.
        return weakref.ref(f, callback)

    # Ok, return a true wrapper.
    return WeakWrapper(f.im_self, f.im_func)

