# treemodel.py - This module contains code to maintain the tree seen
#                in the GUI.
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import filecollection
import tags
import types

# A real functional if.  Used below.
#
# The first argument is the condition; the other two are thunks.
def ifelse(cond, thenf, elsef):
    if cond:
        return thenf()
    else:
        return elsef()

# Comments on the tree:
#
# The tree always has at least four columns.  Column 0 is the Python
# object represented by the row; it is an object from the music
# library database or a group constructor.  Column 1 is the name to be
# placed in the tree for this row.  Column 2 is the column which
# determines whether cells are editable (always true, but..)  Column 3
# is like column 2, but is only true for rows which contain an actual
# file (not just a group header) The remaining columns are
# user-definable and will generally have to do with tags and so forth.
#
# Editing tags proceeds as follows:
#
#  1. when editing a group label, the group label is changed first.
#  If there is another group with the same label in the parent, revert
#  the change.  (this will result in the group being emptied later)
#  Alternatively, test first and don't perform the operation if the
#  group already exists.
#
#  2. For each affected file, set the tag appropriately.  Then query
#  each parent of the file to see if the file still belongs in the
#  parent.  If not, remove the file and re-add it at the root.  (when
#  files are removed, empty groups should be purged, although this can
#  wait until the end -- it might be easier to do that way)
#
# Reverting changes proceeds as follows:
#
# First, the underlying tags are reverted.  Then, each file is checked
# to see if it needs to be moved, in the same manner as (2) above.

COLUMN_PYOBJ=0
COLUMN_LABEL=1
COLUMN_EDITABLE=2
COLUMN_EDITABLE_NONLABEL=3
COLUMN_BACKGROUND=4
COLUMN_FIRST_NONRESERVED=5

# Fills in a row in the model based on the given row object.  The object
# must be compatible with all columns in extracolumns.
#
# Move this to a shared base class of Group and FileProxy?
#
# FIXME: THIS FUNCTION IS THE BOTTLENECK FOR BUILDING THE TREE!
#
# It seems that model.set is a very expensive operation.  I'm not sure
# why; it may have to do with sorting, or something else entirely.
def fill_row(row, iter, model, label, extracolumns):
    # model.set is VERY expensive, call it as few times as possible
    args=[iter,
          COLUMN_PYOBJ, row,
          COLUMN_LABEL, label,
          COLUMN_EDITABLE, 1,
          COLUMN_EDITABLE_NONLABEL, 1,
          COLUMN_BACKGROUND, ifelse(row.modified(),
                                    lambda:"lightpink", lambda:"white")]


    for col in range(0, len(extracolumns)):
        args+=[COLUMN_FIRST_NONRESERVED+col, extracolumns[col].get_value(row)]

    apply(model.set, args)

    return

# Group classes maintain a "shadow" of the model that's easier to access.

# The tree is stored based on three types of classes:
#
# Groups are just a shadow of the GTK+ group.  They are stored in the GTK+
# row and define various operations on the row.  Their parent gives them a
# fixed label, but they know how to generate their column information.
#
# The subclasses of Group should implement the following methods:
#  add_file() must be overridden to actually add the file.
#        (maybe this should be a different name?)
#  remove_child() must be created.
#  belongs_to() must be created.
#  children() must be created and must return a copy of a list (ie, one
#  which is secure against deletion)
#
#  set_child_label() must be created
# FileProxies are proxies for the files in the GUI.  They obey a similar
# interface to Groups.
#
# GroupGenerators are responsible for populating a group.  Given a file,
# they know how to add it to the group they are attached to.

class Group(filecollection.FileCollection):
    def __init__(self, iter, parent, label, model, gui, extracolumns):
        filecollection.FileCollection.__init__(self, gui)
        self.iter=iter
        self.parent=parent
        self.label=label
        self.model=model
        self.extracolumns=extracolumns

        self.__fill_row()

    # Part of the list-item interface
    def get_label(self):
        return self.label

    def __fill_row(self):
        if self.label <> None:
            fill_row(self, self.iter, self.model, self.label, self.extracolumns)

    # TODO: track modified state of files in this group
    #
    # Part of the list-item interface.
    def modified(self):
        return False

    # should this be done by children for symmetry? (removal is
    # handled by children)
    def add_file(self, file):
        # update contained count
        self.inc_count(file.items())

        # The row might need to be updated since it depends on the
        # contents of this subtree
        self.__fill_row()

    # Purges any empty children of this node.  TODO: cache the size of
    # each subtree for great optimization!
    def purge_empty(self):
        for child in self.children():
            child.purge_empty()

            if child.empty():
                self.remove_child(child)

    # Indicates that the given file was removed from this tree or a
    # subtree.
    def file_removed(self, file, olddict):
        # question: we pass the items around everywhere; could these
        # be used to start with? (saves lots of copying, but is it
        # really worth it?)
        self.dec_count(olddict.items())

        # The row might need to be updated since it depends on the
        # contents of this subtree
        self.__fill_row()

        if self.parent:
            self.parent.file_removed(file, olddict)

    # Indicates that the given file in this tree/subtree was updated.
    #
    # Like file_removed, but also adds it back in.
    def file_updated(self, file, olddict):
        self.dec_count(olddict.items())
        self.inc_count(file.items())

        # the row might need to be updated
        self.__fill_row()

        if self.parent:
            self.parent.file_updated(file, olddict)

    def add_underlying_files(self, lst):
        for child in self.children():
            child.add_underlying_files(lst)

# A group with subtrees based on a given tag.  TODO: make tags visible and
# editable if they're the same for all members of a group.
#
# Groups keep track of how many files they contain, which may be
# useful for output purposes, in addition to helping purge empty
# groups.
class TagGroup(Group):
    # next is a function of two arguments (the new item's iterator and the
    # parent Python object)
    # which will generate an empty group the next level down (use
    # lambda to wrap the real constructor)
    def __init__(self, iter, parent, label, model, gui, extracolumns, tag, next):
        Group.__init__(self, iter, parent, label, model, gui, extracolumns)
        self.tag=tag
        self.next=next
        self.__contents={}

    def __val_of(self, file):
        # ignore all but the first value here
        values=file.get_tag(self.tag)
        if len(values)==0:
            # FIXME: Here I assume that nothing has this as a tag value:
            return 'No %s entered'%self.tag.lower()
        else:
            return values[0]

    # Used to check if a file still belongs in the sub-group it has
    # been assigned to.
    def belongs_to(self, file, child):
        val=self.__val_of(file)

        return self.__contents.get(val, None)==child

    def add_file(self, file):
        Group.add_file(self, file)

        val=self.__val_of(file)
        if self.__contents.has_key(val):
            self.__contents[val].add_file(file)
        else:
            # make a new group, add it, and add the file
            newiter=self.model.append(self.iter)
            newgrp=self.next(newiter, self, val)

            self.__contents[val]=newgrp
            newgrp.add_file(file)

    # Test whether the given value is valid for a child.
    def validate_child_label(self, child, newlabel):
        return tags.known_tags[self.tag.lower()].validate(child, newlabel)

    # Set the label of a child to the given value unless it would
    # result in a duplication of tags, then sets the corresponding tag
    # (possibly removing the child from the tree in the process!)
    #
    # This routine has to handle validation.
    def set_child_label(self, child, newlabel):
        oldlabel=self.model.get_value(child.iter, COLUMN_LABEL)

        # sanity-check
        assert(self.__contents.get(oldlabel, None)==child)

        # Check for duplicates -- if there would be a duplicate, don't
        # change the label.
        if not self.__contents.has_key(newlabel):
            # Change the label in the tree, and move the child in the
            # dictionary
            self.model.set(child.iter, COLUMN_LABEL, newlabel)
            # FIXME: where should this be set?  Should the UI call a
            # set_label() on the object itself, and have that call this?
            child.label=newlabel
            del self.__contents[oldlabel]
            self.__contents[newlabel]=child

        # now set the tag
        child.set_tag(self.tag, newlabel)

        # FIXME: update the row of the child and my row?

    def remove_child(self, child):
        label=self.model.get_value(child.iter, COLUMN_LABEL)

        assert(self.__contents.get(label, None)==child)

        del self.__contents[label]
        self.model.remove(child.iter)

    def children(self):
        return self.__contents.values()

# A group which adds its members with the title in column 1.  Note: if
# there are multiple titles, it does NOT create multiple entries.  On
# the other hand, multiple songs with the same title do get multiple
# entries.
#
# Should this just be a tag group on "title"?  But it does a few more
# things.  Could it be a subclass? sibling?  A lot of functionality is
# shared...
class NullGroup(Group):
    def __init__(self, iter, parent, label, model, gui, extracolumns):
        Group.__init__(self, iter, parent, label, model, gui, extracolumns)
        self.__contents=[]

    def add_file(self, file):
        Group.add_file(self, file)

        newiter=self.model.append(self.iter)
        # create a FileProxy; this will fill in the column data for
        # the file.
        f=FileProxy(file, newiter, self, self.model, self.gui, self.extracolumns)
        self.__contents.append(f)

    def remove_child(self, file, olddict):
        assert(self.model.get_value(self.model.iter_parent(file.iter), COLUMN_PYOBJ)==self)

        self.__contents.remove(file)
        self.model.remove(file.iter)

        self.file_removed(file, olddict)

    # hm.
    def children(self):
        return list(self.__contents)

    # always true (hm, could I check if it's in items? that wouldn't
    # do the right thing, though)
    def belongs_to(*args):
        return 1

    def validate_child_label(self, child, newlabel):
        return tags.known_tags['title'].validate(child, newlabel)

    # unconditionally set the child's label.  Should I check that it really
    # is a child?
    def set_child_label(self, child, newlabel):
        self.model.set(child.iter, COLUMN_LABEL, newlabel)
        # this should actually regenerate the column anyway:
        child.set_tag('title', newlabel)

# A proxy for files in the GUI.  Mostly passthrough, but stores
# GUI-specific info, and has hooks to send updates to file tags to the
# GUI.
#
# Currently, this also hides the ability to set multiple tags from the
# rest of the GUI.
class FileProxy:
    def __init__(self, file, iter, parent, model, gui, extracolumns):
        self.file=file
        self.iter=iter
        self.parent=parent
        self.extracolumns=extracolumns
        self.model=model

        self.__fill_in_row()
        self.file.add_listener(self.__reposition_if_necessary)

    # So we can add it to a FileCollection
    def items(self):
        return self.file.items()

    # Part of the list-item interface
    def get_label(self):
        titles=self.file.get_tag('title')
        if len(titles)==0:
            return self.file.fn
        else:
            return titles[0]

    def __fill_in_row(self):
        fill_row(self, self.iter, self.model,
                 self.get_label(), self.extracolumns)

    # Remove this item entirely from the tree and re-add it.
    def __reposition(self, olddict):
        assert(self.parent <> None)

        # remove first.  Local operation.
        self.parent.remove_child(self, olddict)
        # we might be able to let the weak references reap this
        # object, but don't count on it.
        self.file.remove_listener(self.__reposition_if_necessary)

        root=self

        while root.parent <> None:
            root=root.parent

        # Now root is the root of the tree; re-add ourselves. (this
        # object will be discarded and a new proxy created)
        root.add_file(self.file)

    # If this item is in the wrong place, remove it from the tree and
    # re-add it.  Otherwise, update our column in the tree.
    def __reposition_if_necessary(self, file, olddict):
        assert(self.parent <> None)

        # Note that this is guaranteed to have at least one parent
        # (grandparent may be None)
        parent=self.parent
        grandparent=parent.parent

        while grandparent <> None:
            if not grandparent.belongs_to(self, parent):
                self.__reposition(olddict)
                return

            parent=grandparent
            grandparent=grandparent.parent

        # no repositioning needed, update the display to reflect
        # changes.
        self.__fill_in_row()
        # signal to the parent that we have changed (eg, to allow the
        # updating of counts of tag values)
        self.parent.file_updated(self.file, olddict)

    # Part of the interface for list items.
    def title_editable(self):
        return True

    def modified(self):
        return self.file.modified

    def add_listener(self, listener):
        self.file.add_listener(listener)

    def remove_listener(self, listener):
        self.file.remove_listener(listener)

    def get_comments(self):
        return self.file.comments

    def valid_genre(self, genre):
        return self.file.valid_genre(genre)

    # A file node always has one element, so is not empty.
    def empty(self):
        return False

    # Purging the empty children of a file node is a no-op
    def purge_empty(self):
        pass

    # A file node has no children.
    def children(self):
        return []

    def get_tag(self, key):
        return self.file.get_tag(key)

    def set_tag(self, key, val):
        assert(type(val)==types.StringType)

        if val=='':
            val=None

        rval=self.file.set_tag_first(key, val)
        return rval

    def add_underlying_files(self, lst):
        lst.append(self.file)


