# libraryview.py
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# A class that handles the main window of the program.  Currently, it
# modifies and stores configuration data based on the assumption that
# it is the only "view" that is open, meaning that you cannot easily
# create more than one instance without Weird Stuff happening.

import gtk
import gtk.glade
import gobject
import os
import string
import sys

import config
import filecollection
import filelist
import library
import organize
import progress
import status
import tageditor
import tags
import treemodel
import types
import undo

# This needs to be coordinated with the glade file, or past libraries
# will show up in the wrong place!
PAST_LIBRARIES_LOC=4


# Tags that are visible in the tree view.  They are stored as a
# colon-separated list.
#
# adjust some tag visibilities with organization changes?
config.add_option('LibraryView', 'VisibleColumns',
                  ['tracknumber', 'genre'],
                  lambda x:
                      type(x)==types.ListType and
                      reduce(lambda a,b:a and b,
                             map(lambda y:tags.known_tags.has_key(y),
                                 x)))

# The current organization of the view.
config.add_option('LibraryView', 'Organization',
                  'ArtistAlbum',
                  lambda x:
                      type(x)==types.StringType and
                      organize.CANNED_ORGANIZATIONS.has_key(x))

# The last few (5?) libraries that were loaded.
config.add_option('LibraryView', 'PastLibraries',
                  [],
                  lambda x:
                      type(x)==types.ListType and
                      reduce(lambda a,b:a and b,
                             map(lambda y:type(y)==types.StringType,
                                 x),
                             True))

# Column generators compartmentalize the generation of columns.  They
# contain information about how to create a column in the model.  The
# get_type() method on a column generator returns the GType which is
# appropriate for the column, and the get_value() method returns a
# value of the appropriate type.

# A class to generate a column from a string tag.  If several entries
# are available, the first one is arbitrarily chosen.  (a Tag object
# (as below) is stored)
class StringTagColumn:
    def __init__(self, tag):
        self.tag=tag

    def get_title(self):
        return self.tag.title

    def get_tag(self):
        return self.tag.tag

    def get_type(self):
        return gobject.TYPE_STRING

    def get_value(self, f):
        vals=f.get_tag(self.tag.tag)
        if len(vals)>0:
            return vals[0]
        else:
            return None

    def set_value(self, f, val):
        f.set_tag(self.tag.tag, val)

class MusicLibraryView:
    def __init__(self, glade_location, library_location=None):
        self.current_organization=organize.CANNED_ORGANIZATIONS[config.get_option('LibraryView', 'Organization')]

        xml=gtk.glade.XML(glade_location)

        # Connect up the signals.

        self.undo_manager=undo.UndoManager(self.__cleanup_after_undo,
                                           self.__cleanup_after_undo,
                                           self.update_undo_sensitivity)

        xml.signal_autoconnect({'close' : self.close,
                                'quit_program' : self.quit_program,
                                'on_open_library1_activate' : self.handle_open_library,
                                'on_revert_changes1_activate' : self.handle_revert_changes,
                                'on_save_changes' : self.handle_save_changes,
                                'on_undo1_activate' : self.undo_manager.undo,
                                'on_redo1_activate' : self.undo_manager.redo,
                                'on_edit_tags_activate' : self.edit_tags})

        # Extract widgets from the tree
        self.music_list=xml.get_widget('music_list')
        self.status=status.Status(xml.get_widget('statusbar1'))
        self.file_menu=xml.get_widget('file_item').get_submenu()
        self.file_open_item=xml.get_widget('open_library_item')
        self.file_save_item=xml.get_widget('save_changes_item')
        self.file_revert_item=xml.get_widget('discard_changes_item')
        self.undo_item=xml.get_widget('undo_item')
        self.redo_item=xml.get_widget('redo_item')
        self.edit_tags_item=xml.get_widget('edit_tags_item')
        self.organizeview_item=xml.get_widget('organize_view')
        self.showhide_item=xml.get_widget('show/hide_columns')

        # FIXME: the text on toolbar buttons doesn't match the text in
        # the menu for space reasons, but it's ugly to have this
        # discrepancy.
        self.open_button=xml.get_widget('toolbar_open')
        self.save_button=xml.get_widget('toolbar_save')
        self.revert_button=xml.get_widget('toolbar_revert')
        self.undo_button=xml.get_widget('toolbar_undo')
        self.redo_button=xml.get_widget('toolbar_redo')
        self.edit_tags_button=xml.get_widget('toolbar_edit_tags')

        self.update_undo_sensitivity()
        self.past_libraries=[]

        # why does Python not have built in reverse iteration? oh well.
        pl=config.get_option('LibraryView', 'PastLibraries')
        pl.reverse()
        for x in pl:
            self.add_past_library(x)


        self.model=None
        self.library=None

        self.building_tree=False
        self.restart_build_tree=False

        self.update_organizations()

        self.update_known_tags()

        selection=self.music_list.get_selection()
        # Allow multiple selection
        selection.set_mode(gtk.SELECTION_MULTIPLE)
        # Update the sensitivity of edit_tags whenever the selection
        # changes.
        selection.connect('changed', self.tree_selection_changed)

        # set_column_drag_function isn't wrapped -- use this once it
        # is:
        #
        #def handledrop(view, col, prev, next):
        #    return prev <> None
        #
        #self.music_list.set_column_drag_function(handledrop)

        for x in [self.edit_tags_button,self.edit_tags_item]:
            x.set_sensitive(False)

        if library_location:
            self.open_library(library_location)
        else:
            for x in [self.file_save_item, self.save_button,
                      self.file_revert_item, self.revert_button]:
                x.set_sensitive(False)
            self.status.set_message('You have not opened a library')

    # Adds the given library to the stored set, and the File menu
    # if appropriate.  past_libraries is a list of (name,widget).
    def add_past_library(self, lib):
        lib=os.path.realpath(lib)
        # semi-arbitrary limit
        if len(lib)>30:
            libname='.../%s'%os.path.basename(lib)
        else:
            libname=lib

        if len(self.past_libraries)==0:
            w=gtk.SeparatorMenuItem()
            self.file_menu.insert(w,
                                  PAST_LIBRARIES_LOC)
            w.show()

        for x in self.past_libraries:
            if x[0] == lib:
                x[1].destroy()

        w=gtk.MenuItem(libname)
        self.file_menu.insert(w, PAST_LIBRARIES_LOC)
        w.connect('activate', lambda *args:self.open_library(lib))
        w.show()

        self.past_libraries=([(lib, w)]+filter(lambda x:x[0]<>lib,
                                               self.past_libraries))[:6]
        config.set_option('LibraryView', 'PastLibraries',
                          map(lambda x:x[0], self.past_libraries))

    # Should these be here?  Would deriving the GUI window from
    # UndoManager be better?
    def open_undo_group(self):
        self.undo_manager.open_undo_group()

    def close_undo_group(self):
        self.undo_manager.close_undo_group()

    def __cleanup_after_undo(self):
        self.update_undo_sensitivity()

        self.toplevel.purge_empty()

    def tree_selection_changed(self, *args):
        selection=self.music_list.get_selection()

        # Lame lame lame -- pygtk's bindings are annoyingly incomplete
        lst=[]
        selection.selected_foreach(lambda model,path,iter:lst.append(self.model.get_value(iter, treemodel.COLUMN_PYOBJ)))
        edit_tags_active=len(lst)>0

        for x in [self.edit_tags_button, self.edit_tags_item]:
            x.set_sensitive(edit_tags_active)

    def edit_tags(self, *args):
        selection=self.music_list.get_selection()
        lst=[]
        selection.selected_foreach(lambda model,path,iter:self.model.get_value(iter, treemodel.COLUMN_PYOBJ).add_underlying_files(lst))

        if len(lst)>0:
            tageditor.TagEditor(filelist.FileList(self, lst), self.toplevel).show()

    # Sometimes does slightly more than necessary, but safer than
    # trying to be overly clever.
    def update_undo_sensitivity(self):
        self.undo_item.set_sensitive(self.undo_manager.has_undos())
        self.undo_button.set_sensitive(self.undo_manager.has_undos())

        self.redo_item.set_sensitive(self.undo_manager.has_redos())
        self.redo_button.set_sensitive(self.undo_manager.has_redos())

    # TODO: check for modifications and pop up a dialog as appropriate
    # For now, just quit the program when a window is closed (need to
    # be cleverer, maybe even kill file->quit)
    def close(self, *args):
        self.quit_program()

    # main_quit is just broken and doesn't behave.
    def quit_program(self,*args):
        sys.exit(0)

    def column_toggled(self, menu_item, tag):
        nowvisible=menu_item.get_active()

        currcols=config.get_option('LibraryView',
                                   'VisibleColumns')

        if nowvisible:
            if not tag in currcols:
                currcols.append(tag)
        else:
            currcols.remove(tag)

        config.set_option('LibraryView', 'VisibleColumns',
                          currcols)

        self.guicolumns[tag].set_visible(menu_item.get_active())

    # Handles a change to the set of organizations.  Used to construct
    # the "organizations" menu up front, and to select the appropriate
    # initial value (from "config").
    def update_organizations(self):
        # Can I sort them in a better way?
        organizations=organize.CANNED_ORGANIZATIONS.keys()
        organizations.sort(lambda a,b:cmp(organize.CANNED_ORGANIZATIONS[a][2],
                                          organize.CANNED_ORGANIZATIONS[b][2]))

        menu=gtk.Menu()

        curr_organization=config.get_option('LibraryView', 'Organization')

        # taking this from the pygtk demo -- not sure how it works in
        # general :-/
        group=None

        for o in organizations:
            title=organize.CANNED_ORGANIZATIONS[o][2]

            menuitem=gtk.RadioMenuItem(group, title)

            if o == curr_organization:
                menuitem.set_active(o == curr_organization)

            group=menuitem
            menuitem.show()

            menu.add(menuitem)

            menuitem.connect('activate', self.choose_canned_organization, o)

        self.organizeview_item.set_submenu(menu)


    # Handles a change to the set of known tags
    def update_known_tags(self):
        self.extracolumns=map(lambda x:StringTagColumn(x),
                              tags.known_tags.values())
        self.extracolumns.sort(lambda a,b:cmp(a.get_title(),b.get_title()))
        self.guicolumns={}
        self.tagmenuitems={}

        # Create a menu for toggling column visibility.
        menu=gtk.Menu()

        renderer=gtk.CellRendererText()
        renderer.connect('edited', self.handle_edit, None)
        col=gtk.TreeViewColumn('Title',
                               renderer,
                               cell_background=treemodel.COLUMN_BACKGROUND,
                               text=treemodel.COLUMN_LABEL,
                               editable=treemodel.COLUMN_EDITABLE)
        self.music_list.append_column(col)

        initial_visible_columns=config.get_option('LibraryView', 'VisibleColumns')

        # Add the remaining columns
        for n in range(0, len(self.extracolumns)):
            col=self.extracolumns[n]

            # Set up a menu item for this.
            is_visible=(col.get_tag() in initial_visible_columns)
            menuitem=gtk.CheckMenuItem(col.get_title())
            menuitem.set_active(is_visible)
            menuitem.show()
            menuitem.connect('activate', self.column_toggled, col.get_tag())
            menu.add(menuitem)

            src=n+treemodel.COLUMN_FIRST_NONRESERVED
            renderer=gtk.CellRendererText()
            renderer.connect('edited', self.handle_edit, col)
            gcol=gtk.TreeViewColumn(col.get_title(),
                                    renderer,
                                    text=src,
                                    editable=treemodel.COLUMN_EDITABLE_NONLABEL,
                                    cell_background=treemodel.COLUMN_BACKGROUND)
            gcol.set_visible(is_visible)
            gcol.set_reorderable(True)
            self.guicolumns[col.get_tag()]=gcol
            self.music_list.append_column(gcol)

        self.showhide_item.set_submenu(menu)

    def set_organization(self, organization):
        if organization <> self.current_organization:
            self.current_organization=organization
            self.build_tree()

    # signal handler:
    def choose_canned_organization(self, widget, name):
        # GTK+ bug: "activate" is emitted for a widget when it's
        # either activated OR deactivated.
        if widget.get_active():
            # FIXME: show an error dialog if the name doesn't exist.
            self.set_organization(organize.CANNED_ORGANIZATIONS[name])
            config.set_option('LibraryView', 'Organization', name)

    # Handle an edit of a field.  "column" is a column generator or None for
    # the label column.
    def handle_edit(self, cell, path_string, new_text, column):
        iter=self.model.get_iter(path_string)
        rowobj=self.model.get_value(iter, treemodel.COLUMN_PYOBJ)

        # Handle the label specially
        if column==None:
            if new_text <> '' and not rowobj.parent.validate_child_label(rowobj, new_text):
                return
            rowobj.parent.set_child_label(rowobj, new_text)
        else:
            tag=tags.known_tags[column.get_tag()]
            # hm, should '' be handled specially? ew.
            if new_text <> '' and not tag.validate(rowobj, new_text):
                return
            column.set_value(rowobj, new_text)

        # find the root and purge empty groups from it
        #
        # why not just use toplevel?
        root=rowobj
        while root.parent:
            root=root.parent

        root.purge_empty()

        self.model.sort_column_changed()

    # Handle the "open library" menu function.
    def handle_open_library(self, *args):
        def on_ok(widget):
            fn=filesel.get_filename()
            filesel.destroy()

            if not os.path.isdir(fn):
                msgdlg=gtk.MessageDialog(None, 0, gtk.MESSAGE_WARNING,
                                         gtk.BUTTONS_OK, 'You must pick a directory' )
                msgdlg.show()
                msgdlg.connect('response', lambda *args:self.handle_open_library())
                msgdlg.connect('response', lambda *args:msgdlg.destroy())
            else:
                self.open_library(fn)

        filesel=gtk.FileSelection("Choose the directory containing the library")
        filesel.ok_button.connect('clicked', on_ok)
        filesel.cancel_button.connect('clicked', lambda *args:filesel.destroy())
        filesel.show()

    # Handle the "revert changes" menu function.
    def handle_revert_changes(self, widget):
        self.status.push()
        try:
            self.library.revert(progress.ProgressUpdater("Discarding changes",
                                                         self.show_progress))
            self.toplevel.purge_empty()
        finally:
            self.status.pop()

    # Handle the "save changes" menu function.
    def handle_save_changes(self, widget):
        self.status.push()
        try:
            self.library.commit(progress.ProgressUpdater("Saving changes",
                                                         self.show_progress))
        finally:
            self.status.pop()

    # Set the progress display to the given amount, with the given
    # percentage and message.
    #
    # Assumes it needs to call gtk_poll()
    def show_progress(self, message, percent):
        self.status.set_message('%s: %d%%'%(message, int(percent*100)))
        while(gtk.events_pending()):
            gtk.main_iteration_do(False)

    # makegroup takes a model argument and returns the base group.
    def build_tree(self, callback=None):
        if not self.library:
            return

        # pygtk is disgustingly broken; this "restart_build_tree"
        # business is an attempt to fake what would happen if it
        # correctly passed exceptions through gtk.mainiter().
        if self.building_tree:
            self.restart_build_tree=True
            return

        self.status.push()

        try:
            success=False
            self.building_tree=True
            while not success:
                self.restart_build_tree=False

                # Needed here because self isn't bound in the argument list:
                if callback==None:
                    callback=progress.ProgressUpdater("Building music tree",
                                                      self.show_progress)

                self.model=apply(gtk.TreeStore,[gobject.TYPE_PYOBJECT,
                                                gobject.TYPE_STRING,
                                                gobject.TYPE_BOOLEAN,
                                                gobject.TYPE_BOOLEAN,
                                                gobject.TYPE_STRING]+map(lambda x:x.get_type(),
                                                                         self.extracolumns))

                self.toplevel=self.current_organization[0](self.model,
                                                           self,
                                                           self.extracolumns)

                cur=0
                max=len(self.library.files)
                for file in self.library.files:
                    callback(cur, max)

                    if self.restart_build_tree:
                        break

                    cur+=1
                    self.toplevel.add_file(file)

                if self.restart_build_tree:
                    continue

                callback(max, max)

                if self.restart_build_tree:
                    continue
                success=True

                # I'd like to have these up above in order to provide more
                # visual feedback while building the tree.  Unfortunately,
                # adding items to a tree with a sorting function is
                # hideously expensive, so I have to do this here (IMO this
                # is a bug in the TreeStore)
                #
                # eg: with 660 items, if I set the sorter before building
                # the tree, it is called 26000 times; if I set it here, it
                # is called 2300 times.  For 96 items it is called 3975
                # and 445 times, respectively.
                self.model.set_sort_func(0, self.current_organization[1])
                self.model.set_sort_column_id(0, gtk.SORT_ASCENDING)
                self.music_list.set_model(self.model)
                # Changing the organization can change the optimal
                # column width:
                self.music_list.columns_autosize()

            self.building_tree=False
        finally:
            self.status.pop()

    def __do_add_undo(self, file, olddict):
        self.undo_manager.add_undo(undo.TagUndo(file, olddict, file.comments))

    def open_library(self, library_location):
        self.add_past_library(library_location)

        self.status.push()

        try:
            # TODO: do I need to recover sensitivity if an exception occurs?
            for x in [self.file_open_item, self.file_save_item,
                      self.file_revert_item, self.open_button,
                      self.save_button, self.revert_button]:
                x.set_sensitive(False)

            self.library=library.MusicLibrary(library_location,
                                              progress.ProgressUpdater("Indexing music files in %s"%library_location, self.show_progress))

            for x in [self.file_open_item, self.file_save_item,
                      self.file_revert_item, self.open_button,
                      self.save_button, self.revert_button]:
                x.set_sensitive(True)

            self.build_tree()

            for file in self.library.files:
                file.add_listener(self.__do_add_undo)

        finally:
            self.status.pop()

        self.status.set_message('Editing %s'%library_location)
