# library.py
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Code to manage a library.

import ID3
import os
import os.path
import ogg.vorbis
import weakref

# Utility function
def fname_ext(fn):
    if not '.' in fn or fn.rfind('.')==len(fn)-1:
        return None

    return fn[fn.rfind('.')+1:]

# Work around a Python bug with weak references and bound methods.
def WeakCallableRef(f, callback=lambda *args:None):
    # We expect a callable that is a method to die exactly when the
    # object does.  To facilitate this, use the following class:
    class WeakWrapper:
        def __init__(self, obj, method):
            self.obj=weakref.ref(obj, self.__kill)
            self.method=method

        def __kill(self, deadman):
            self.method=lambda *args:None
            callback(self)

        def __call__(self, *args):
            obj=self.obj()

            if obj == None:
                return None
            else:
                return lambda *args:apply(self.method, (obj,)+args)

    # Is it a method?
    try:
        f.im_self
    except AttributeError:
        # No, so the bug doesn't apply; use a standard weak reference.
        return weakref.ref(f, callback)

    # Ok, return a true wrapper.
    return WeakWrapper(f.im_self, f.im_func)

# An object which you can attach to to listen to changes in its
# comments.  (used below, but the code is useful for stuff outside
# this module as well)
class Listenable:
    def __init__(self):
        self.listeners=[]

    def add_listener(self, listener, weak=True):
        if weak:
            wrapper=WeakCallableRef(listener)
        else:
            wrapper=lambda:listener
        self.listeners.append(wrapper)

    def remove_listener(self, listener):
        self.listeners=filter(lambda x:x() <> None and x() <> listener,
                              self.listeners)

    def call_listeners(self, oldcomments):
        # Remove currently dead listeners
        self.listeners=filter(lambda x:x() <> None,
                              self.listeners)

        for wrapper in list(self.listeners):
            listener=wrapper()

            # This would be a race condition otherwise -- I don't use
            # threads now, but it would be annoying if I started using
            # them and got weird bugs because of this.
            if listener <> None:
                listener(self, oldcomments)    

# A generic file.  Initializes some stuff all files need.
class File(Listenable):
    def __init__(self, library):
        Listenable.__init__(self)

        self.library=library

# A file with a dict interface.  This is an abstract class; subclasses
# need to implement the write_to_file() method and initialize "comments"
# in their constructor.  It is ASSUMED that the keys in "comments" are
# upper-case.
class DictFile(File):
    def __init__(self, library, comments):
        File.__init__(self, library)
        self.comments=comments
        self.__rationalize_comments()
        self.backup_comments=self.comments.copy()
        self.modified=0

    def __rationalize_comments(self):
        for key,val in self.comments.items():
            if val == []:
                del self.comments[key]

    # Returns ZERO or more strings associated with the key (never fails)
    #
    # Note that the dictionary is guaranteed to contain upper-case
    # keys and the tags are guaranteed to be lists of strings.  (at
    # least, in the current implementation??  boo poorly documented
    # libraries)
    def get_tag(self, key):
        return self.comments.get(key.upper(), [])

    # Sets all tags based on the given dictionary
    def set_tags(self, dict):
        if self.comments <> dict:
            oldcomments=self.comments
            self.comments=dict.copy()
            self.__rationalize_comments()

            self.modified = (self.comments <> self.backup_comments)

            self.call_listeners(oldcomments)

    # Sets the given tag
    def set_tag(self, key, val):
        if self.get_tag(key) <> val:
            upkey=key.upper()
            # used to handle updating structures in the GUI:
            oldcomments=self.comments.copy()

            if val==[]:
                del self.comments[upkey]
            else:
                self.comments[upkey]=val

            if not self.modified: # the set of modified files
                                  # increases monotonically; that's
                                  # ok, it's cheaper than trying to
                                  # weed out files that were modified
                                  # to be unmodified. (is it? can I do
                                  # this cheaper using a dictionary?)
                self.library.add_modified_file(self)

            # careful here. (could just always compare dictionaries,
            # but this is a little more efficient in some common
            # cases)
            if (not self.modified):
                # if it wasn't modified, we can just compare the new value
                # to the old value.
                self.modified=(val <> self.backup_comments.get(upkey, []))
            else:
                # it was modified; if this key is now the same as its
                # original value, compare the whole dictionary. (no
                # way around this right now) Note that if it isn't the
                # same, you might as well just leave it modified.
                if val == self.backup_comments.get(upkey, []):
                    self.modified = (self.comments <> self.backup_comments)

            self.call_listeners(oldcomments)

    # Get a list of tags
    def tags(self):
        return self.comments.keys()

    # Get a list of values
    def values(self):
        return self.comments.values()

    # Get a list of the items
    def items(self):
        return self.comments.items()

    # Sets only the first entry in the given tag, leaving the rest (if
    # any) unmodified.
    def set_tag_first(self, key, val):
        cur=self.get_tag(key)
        if cur==[]:
            if val <> None:
                self.set_tag(key, [val])
        elif val==None:
            new=list(cur)
            del new[0]
            self.set_tag(key, new)
        elif cur[0] <> val:
            new=list(cur)
            new[0]=val
            self.set_tag(key, new)

    # Commits any changes.
    def commit(self):
        if self.modified:
            self.write_to_file()
            self.modified=0
            self.call_listeners(self.comments)
            self.backup_comments=self.comments.copy()

    # Reverts any changes
    def revert(self):
        if self.modified:
            # no copy, we aren't modifying them.
            oldcomments=self.comments
            self.comments=self.backup_comments.copy()
            self.modified=0

            self.call_listeners(oldcomments)

import sys

# An object representing an Ogg file.
class OggFile(DictFile):
    def __init__(self, library, fn):
        # cache the list of comments
        #
        # TODO: catch an exception here
        DictFile.__init__(self, library, ogg.vorbis.VorbisFile(fn).comment().as_dict())
        self.fn=fn

    def write_to_file(self):
        class FooExcept:
            def __init__(self):
                pass
        try:
            c=ogg.vorbis.VorbisComment(self.comments)

            # work around a pyvorbis bug by trashing exception information:
            raise FooExcept()
        except FooExcept:
            pass

        c.write_to(self.fn)

    def valid_genre(self, genre):
        return True

# Fix a glaring omission in the id3 module: map strings -> numbers
backgenres={'Unknown Genre' : 255}

genre=0
for name in ID3.ID3.genres:
    backgenres[name]=genre
    genre=genre+1

del genre
del name

# This works around a bug in some versions of python-id3, where only
# trailing NULs are stripped.  (some broken software produced
# incorrectly formatted tags, where values weren't padded with NULs --
# only one NUL was used to terminate)
def strip_padding(s):
    nulloc=s.find('\0')
    if nulloc <> -1:
        return s[:nulloc].rstrip()
    else:
        return s.rstrip()

# An object representing a file with id3 tags (ie, an mp3 file)
class ID3File(DictFile):
    def __init__(self, library, fn):
        id3file=ID3.ID3(fn)
        # it's not clear what modifying the return value of as_dict()
        # will do, so it's copied below
        d=id3file.as_dict()

        # This works around a bug in some versions of python-id3,
        # where only trailing NULs are stripped.  (some broken
        # software produced incorrectly formatted tags, where values
        # weren't padded with NULs)
        dict={}
        for key,value in d.items():
            # Convert to a list to be consistent with ogg
            dict[key]=[strip_padding(value)]

        DictFile.__init__(self, library, dict)
        self.fn=fn

    def write_to_file(self):
        id3file=ID3.ID3(self.fn, as_tuple)

        for key,val in self.comments.items():
            if key=='GENRE':
                if len(val)>0:
                    id3file.genre=backgenres[val[0]]
            else:
                id3file[key]=val[0]

        id3file.write()

    def valid_genre(self, genre):
        return backgenres.has_key(genre)

# Represents a library.  On startup, all files in the library are
# catalogued; additional files can be added through the add_file
# function. (files already in the library are automatically ignored)
class MusicLibrary:
    # A library is initialized from a directory.  The callback argument
    # can be used to display how close to being done the process is.
    def __init__(self, dir, callback=lambda cur,max:None):
        if not os.path.isdir(dir):
            if os.path.exists(dir):
                raise 'Invalid library: %s is not a directory'%dir
            else:
                raise 'Invalid library: %s does not exist'%dir

        # Initialize the library
        self.dir=dir
        self.files=[]
        self.modified_files=[]
        self.inodes={}

        candidate_files=self.__find_files(dir, {}, [])
        cur=0
        max=len(candidate_files)
        for file in candidate_files:
            callback(cur, max)
            cur+=1
            self.add_file(file)
        # One last time to show that we're done
        callback(max, max)

    # Finds all files in the given directory and subdirectories,
    # following symlinks and avoiding cycles.
    def __find_files(self, dir, seen_dirs, output):
        assert(os.path.isdir(dir))

        dir_ino=os.stat(dir).st_ino

        if not seen_dirs.has_key(dir_ino) and os.access(dir, os.R_OK|os.X_OK):
            seen_dirs[dir_ino]=1

            for name in os.listdir(dir):
                fullname=os.path.join(dir, name)

                if os.path.isfile(fullname) and file_types.has_key(fname_ext(fullname)):
                    output.append(fullname)
                elif os.path.isdir(fullname):
                    self.__find_files(fullname, seen_dirs, output)

        return output

    # Try to add the given file to the library.
    def add_file(self, file):
        file_ino=os.stat(file).st_ino
        if not self.inodes.has_key(file_ino) and os.access(file, os.R_OK):
            self.inodes[file_ino]=1
            
            # Find the file extension and associated handler
            ext=fname_ext(file)
            if file_types.has_key(ext):
                self.files.append(file_types[ext](self, file))

    # Commit all changes.  This may involve restructuring the
    # library's layout in the filesystem.
    def commit(self, callback=lambda cur,max:None):
        cur=0
        max=len(self.modified_files)
        for file in self.modified_files:
            callback(cur, max)
            cur+=1
            file.commit()
        callback(max, max)

        self.modified_files=[]

    # Revert all changes.
    def revert(self, callback=lambda cur,max:None):
        cur=0
        max=len(self.modified_files)
        for file in self.modified_files:
            callback(cur, max)
            cur+=1
            file.revert()
        callback(max, max)

        self.modified_files=[]

    # adds the file to the list of modified files -- no check of whether
    # it's already in!
    def add_modified_file(self, file):
        self.modified_files.append(file)

# The set of known file extensions and associated classes.  It is
# assumed that None is not included in this dictionary.
file_types = {
    'ogg' : OggFile,
    'mp3' : ID3File
    }
