# filecollection.py
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import library

# This is a base class for objects that contain some number of files
# and need to keep track of how many of the files they contain have
# particular (key, value) mappings.  Children are responsible for
# calling inc_count and dec_count as values change or files are removed.
#
# This actually implements part of the list-item interface.
class FileCollection(library.Listenable):
    def __init__(self, gui):
        library.Listenable.__init__(self)

        # This contains a dictionary mapping tags to dictionaries
        # mapping values to their occurance count.
        self.contained_values={}
        self.gui=gui

    # Returns the visible comments as a dictionary.  Part of the
    # list-item interface.
    def get_comments(self):
        rval={}

        for tag,countdict in self.contained_values.items():
            if len(countdict)==1:
                rval[tag]=[countdict.keys()[0]]

        return rval

    # So we can add TagCollections to TagCollections.
    def items(self):
        rval=[]
        for tag,countdict in self.contained_values.items():
            for val,count in countdict.items():
                rval.append((tag,[val],count))

        return rval

    # Used to determine whether the title should be editable.  The
    # idea is that you almost never want to set multiple songs to have
    # an identical title, so the interface shouldn't encourage you to
    # do it accidentally.
    #
    # The title could be editable iff this collection contains exactly
    # one item, but that gets sketchy if items are added and removed.
    # Subclasses can of course override this.
    def title_editable(self):
        return False

    # Adjusts the contained count to remove the given counts.  The
    # counts are given as a set of tuples (key, val, count), or as a
    # set of tuples (key, val).  If there is no count, it is assumed
    # that the count is 1.  Note that values --must-- be lists of
    # strings.
    def dec_count(self, items):
        oldcomments=self.get_comments()
        for item in items:
            tag=item[0]
            val=item[1]
            if len(item)>2:
                count=item[2]
            else:
                count=1

            uptag=tag.upper()
            countdict=self.contained_values[uptag]
            if len(val)>0:
                val0=val[0]
                assert(countdict[val0]>=count)
                countdict[val0]=countdict[val0]-count
                if countdict[val0]==0:
                    del countdict[val0]
                    if len(countdict)==0:
                        del self.contained_values[uptag]

        if oldcomments <> self.get_comments():
            self.call_listeners(oldcomments)

    # Adjusts the contained count to add the given dictionary (given
    # as items)
    def inc_count(self, items):
        oldcomments=self.get_comments()
        for item in items:
            tag=item[0]
            val=item[1]
            if len(item)>2:
                count=item[2]
            else:
                count=1

            uptag=tag.upper()
            countdict=self.contained_values.setdefault(uptag, {})
            if len(val)>0:
                countdict[val[0]]=countdict.get(val[0], 0)+count

        if oldcomments <> self.get_comments():
            self.call_listeners(oldcomments)

    # This is used for sorting and generating column data; it works by
    # checking for information which is identical across all children
    # using contained_values.
    #
    # Part of the list-item interface
    def get_tag(self, tag):
        countdict=self.contained_values.get(tag.upper(), {})
        if len(countdict)==1:
            return [countdict.keys()[0]]
        else:
            return []

    # Part of the list-item interface
    def set_tag(self, tag, val):
        self.gui.open_undo_group()
        for item in self.children():
            item.set_tag(tag, val)
        self.gui.close_undo_group()

    # Return True iff the given genre is valid for every member of the
    # group.  (is this a little strict?)
    def valid_genre(self, genre):
        for child in self.children():
            if not child.valid_genre(genre):
                return False

        return True

    # Tests if this node is empty.  Part of the list-item interface
    #
    # Note: this is a little bit of a hack -- it assumes that every
    # file has at least one tag.
    def empty(self):
        return len(self.children())==0
