;;; muse-latex.el --- publish entries in LaTex or PDF format

;; Copyright (C) 2004, 2005  Free Software Foundation, Inc.

;; This file is not part of GNU Emacs.

;; This is free software; you can redistribute it and/or modify it under
;; the terms of the GNU General Public License as published by the Free
;; Software Foundation; either version 2, or (at your option) any later
;; version.
;;
;; This is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;; for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Commentary:

;;; Contributors:

;; Li Daobing (lidaobing AT gmail DOT com) provided CJK support.
;;
;; Trent Buck (trentbuck AT gmail DOT com) gave valuable advice for
;; how to treat LaTeX specials and the like.

;;; Code:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Muse LaTeX Publishing
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'muse-publish)

(defgroup muse-latex nil
  "Rules for marking up a Muse file as a LaTeX article."
  :group 'muse-publish)

(defcustom muse-latex-extension ".tex"
  "Default file extension for publishing LaTeX files."
  :type 'string
  :group 'muse-latex)

(defcustom muse-latex-pdf-extension ".pdf"
  "Default file extension for publishing LaTeX files to PDF."
  :type 'string
  :group 'muse-latex)

(defcustom muse-latex-header
  "\\documentclass{article}

\\usepackage[english]{babel}
\\usepackage[latin1]{inputenc}
\\usepackage[T1]{fontenc}
\\usepackage{hyperref}
\\usepackage[pdftex]{graphicx}

\\newcommand{\\comment}[1]{}

\\begin{document}

\\title{<lisp>(muse-publishing-directive \"title\")</lisp>}
\\author{<lisp>(muse-publishing-directive \"author\")</lisp>}
\\date{<lisp>(muse-publishing-directive \"date\")</lisp>}

\\maketitle

<lisp>(and muse-publish-generate-contents
           \"\\\\tableofcontents\n\\\\newpage\")</lisp>\n\n"
  "Header used for publishing LaTeX files.  This may be text or a filename."
  :type 'string
  :group 'muse-latex)

(defcustom muse-latex-footer "\n\\end{document}\n"
  "Footer used for publishing LaTeX files.  This may be text or a filename."
  :type 'string
  :group 'muse-latex)

(defcustom muse-latexcjk-header
  "\\documentclass{article}

\\usepackage{CJK}
\\usepackage{indentfirst}
\\usepackage[CJKbookmarks=true]{hyperref}
\\usepackage[pdftex]{graphicx}

\\begin{document}
\\begin{CJK*}<lisp>(muse-latexcjk-encoding)</lisp>

\\title{<lisp>(muse-publishing-directive \"title\")</lisp>}
\\author{<lisp>(muse-publishing-directive \"author\")</lisp>}
\\date{<lisp>(muse-publishing-directive \"date\")</lisp>}

\\maketitle

<lisp>(and muse-publish-generate-contents
           \"\\\\tableofcontents\n\\\\newpage\")</lisp>\n\n"
  "Header used for publishing LaTeX files (CJK).  This may be text or a
filename."
  :type 'string
  :group 'muse-latex)

(defcustom muse-latexcjk-footer
  "\n\\end{CJK*}
\\end{document}\n"
  "Footer used for publishing LaTeX files (CJK).  This may be text or a
filename."
  :type 'string
  :group 'muse-latex)

(defcustom muse-latex-markup-regexps
  `(;; numeric ranges
    (10000 "\\([0-9]+\\)-\\([0-9]+\\)" 0 "\\1--\\2")

    ;; characters which need quoting
    (10010 "\\([$#%]\\)" 0 "\\\\\\1")
    (10020 "_" 0 "\\\\textunderscore{}")
    (10030 "<" 0 "\\\\textless{}")
    (10040 ">" 0 "\\\\textgreater{}")
    (10050 "\\^" 0 "\\\\^{}")

    ;; be careful of closing quote pairs
    (10100 "\"'" 0 "\"\\\\-'")

    ;; join together the parts of a list or table
    (10200 ,(concat
             "\\\\end{\\(tabular\\|description\\|itemize\\|enumerate\\)}"
             "\\([" muse-regexp-blank "]*\n\\)\\{0,2\\}"
             "[" muse-regexp-blank "]*"
             "\\\\begin{\\1}\\({[^\n}]+}\\)?\n+") 0 ""))
  "List of markup regexps for identifying regions in a Muse page.
For more on the structure of this list, see `muse-publish-markup-regexps'."
  :type '(repeat (choice
                  (list :tag "Markup rule"
                        integer
                        (choice regexp symbol)
                        integer
                        (choice string function symbol))
                  function))
  :group 'muse-latex)

(defcustom muse-latex-markup-functions
  '((anchor . muse-latex-markup-anchor)
    (table . muse-latex-markup-table))
  "An alist of style types to custom functions for that kind of text.
For more on the structure of this list, see
`muse-publish-markup-functions'."
  :type '(alist :key-type symbol :value-type function)
  :group 'muse-latex)

(defcustom muse-latex-markup-strings
  '((image-with-desc . "\\includegraphics[width=\\textwidth]{%s}")
    (image-link      . "\\includegraphics[width=\\textwidth]{%s}")
    (url-with-image  . "%% %s\n\\includegraphics[width=\\textwidth]{%s}")
    (url-link        . "\\href{%s}{%s}")
    (internal-link   . "\\hyperlink{%s}{%s}")
    (email-addr      . "\\verb|%s|")
    (emdash          . "---")
    (comment-begin   . "\\comment{")
    (comment-end     . "}")
    (rule            . "\\bigskip")
    (no-break-space  . "~")
    (enddots         . "\\ldots{}")
    (dots            . "\\dots{}")
    (part            . "\\part{")
    (part-end        . "}")
    (chapter         . "\\chapter{")
    (chapter-end     . "}")
    (section         . "\\section{")
    (section-end     . "}")
    (subsection      . "\\subsection{")
    (subsection-end  . "}")
    (subsubsection   . "\\subsubsection{")
    (subsubsection-end . "}")
    (section-other   . "\\paragraph{")
    (section-other-end . "}")
    (footnote        . "\\footnote{")
    (footnote-end    . "}")
    (footnotetext    . "\\footnotetext[%d]{")
    (begin-underline . "\\underline{")
    (end-underline   . "}")
    (begin-literal   . "\\texttt{")
    (end-literal     . "}")
    (begin-emph      . "\\emph{")
    (end-emph        . "}")
    (begin-more-emph . "\\textbf{")
    (end-more-emph   . "}")
    (begin-most-emph . "\\textbf{\\emph{")
    (end-most-emph   . "}}")
    (begin-verse     . "\\begin{verse}\n")
    (end-verse-line  . " \\\\")
    (verse-space     . "~~~~")
    (end-verse       . "\n\\end{verse}")
    (begin-example   . "\\begin{quote}\n\\begin{verbatim}")
    (end-example     . "\\end{verbatim}\n\\end{quote}")
    (begin-center    . "\\begin{center}\n")
    (end-center      . "\n\\end{center}")
    (begin-quote     . "\\begin{quote}\n")
    (end-quote       . "\n\\end{quote}")
    (begin-uli       . "\\begin{itemize}\n\\item ")
    (end-uli         . "\n\\end{itemize}")
    (begin-oli       . "\\begin{enumerate}\n\\item ")
    (end-oli         . "\n\\end{enumerate}")
    (begin-ddt       . "\\begin{description}\n\\item[")
    (start-dde       . "] ")
    (end-ddt         . "\\end{description}"))
  "Strings used for marking up text.
These cover the most basic kinds of markup, the handling of which
differs little between the various styles."
  :type '(alist :key-type symbol :value-type string)
  :group 'muse-latex)

(defcustom muse-latexcjk-encoding-map
  '((utf-8              . "{UTF8}{song}")
    (japanese-iso-8bit  . "[dnp]{JIS}{min}")
    (chinese-big5       . "{Bg5}{bsmi}")
    (mule-utf-8         . "{UTF8}{song}")
    (chinese-iso-8bit   . "{GB}{song}")
    (chinese-gbk        . "{GBK}{song}"))
  "An alist mapping emacs coding systems to appropriate CJK codings.
Use the base name of the coding system (ie, without the -unix)."
  :type '(alist :key-type coding-system :value-type string)
  :group 'muse-latex)

(defcustom muse-latexcjk-encoding-default "{GB}{song}"
  "The default Emacs buffer encoding to use in published files.
This will be used if no special characters are found."
  :type 'string
  :group 'muse-latex)

(defun muse-latexcjk-encoding ()
  (when (boundp 'buffer-file-coding-system)
    (muse-latexcjk-transform-content-type buffer-file-coding-system)))

(defun muse-latexcjk-transform-content-type (content-type)
  "Using `muse-cjklatex-encoding-map', try and resolve an emacs coding
system to an associated CJK coding system."
  (let ((match (and (fboundp 'coding-system-base)
                    (assoc (coding-system-base content-type)
                           muse-latexcjk-encoding-map))))
    (if match
        (cdr match)
      muse-latexcjk-encoding-default)))

(defcustom muse-latex-markup-specials
  '((?\\ . "\\\\"))
  "A table of characters which must be represented specially."
  :type '(alist :key-type character :value-type string)
  :group 'muse-latex)

(defcustom muse-latex-markup-texttt-specials
  '((?\n . "\\\n")
    (?_  . "\\textunderscore{}")
    (?\< . "\\textless{}")
    (?\> . "\\textgreater{}")
    (?^  . "\\^{}")
    (?\$ . "\\$")
    (?\% . "\\%")
    (?\{ . "\\{")
    (?\} . "\\}")
    (?\& . "\\&")
    (?\# . "\\#"))
  "A table of characters which must be represented specially.
This applies to text in \\texttt{} regions."
  :type '(alist :key-type character :value-type string)
  :group 'muse-latex)

(defun muse-latex-insert-anchor (anchor)
  "Insert an anchor, either around the word at point, or within a tag.
If the anchor occurs at the end of a line, ignore it."
  (unless (or (bolp)  ; point is placed after newline if anchor at end
              (get-text-property (match-end 1) 'noemphasis))
    (skip-chars-forward (concat muse-regexp-blank "\n"))
    (if (looking-at "<\\([^ />]+\\)>")
        (let ((tag (match-string 1)))
          (goto-char (match-end 0))
          (insert "\\hypertarget{" anchor "}{")
          (or (and (search-forward (format "</%s>" tag)
                                   (muse-line-end-position) t)
                   (goto-char (match-beginning 0)))
              (forward-word 1))
          (insert "}"))
      (insert "\\hypertarget{" anchor "}{")
      (forward-word 1)
      (insert "}"))))

(defun muse-latex-markup-anchor ()
  (save-match-data
    (muse-latex-insert-anchor (match-string 1)))
  (match-string 1))

(defun muse-latex-markup-table ()
  (let* ((str (prog1
                  (match-string 1)
                (delete-region (match-beginning 0) (match-end 0))))
         (fields (split-string str "\\s-*|+\\s-*"))
         (type (and (string-match "\\s-*\\(|+\\)\\s-*" str)
                    (length (match-string 1 str)))))
    (insert "\\begin{tabular}{" (make-string (length fields) ?l) "}\n")
    (when (= type 3)
      (insert "\\hline\n"))
    (insert (mapconcat 'identity fields " & "))
    (insert " \\\\\n")
    (when (= type 2)
      (insert "\\hline\n"))
    (insert "\\end{tabular}")))

(defun muse-latex-fixup-dquotes ()
  "Fixup double quotes."
  (let ((open t))
    (while (search-forward "\"" nil t)
      (unless (get-text-property (match-beginning 0) 'read-only)
        (if (and (bolp) (eq (char-before) ?\n))
            (setq open t))
        (if open
            (progn
              (replace-match "``")
              (setq open nil))
          (replace-match "''")
          (setq open t))))))

(defun muse-latex-fixup-texttt ()
  "Escape extra characters in texttt sections."
  (let ((inhibit-read-only t)
        beg end)
    (while (and (< (point) (point-max))
                (setq beg (search-forward "\\texttt{" nil t)))
      (goto-char (next-single-property-change (point) 'read-only))
      (backward-char)
      (setq end (point-marker))
      (goto-char beg)
      (while (< (point) end)
        (let ((repl (assoc (char-after) muse-latex-markup-texttt-specials)))
          (if (null repl)
              (forward-char 1)
            (delete-char 1)
            (insert-before-markers (cdr repl))))))))

(defun muse-latex-finalize-buffer ()
  (goto-char (point-min))
  (muse-latex-fixup-dquotes)
  (goto-char (point-min))
  (muse-latex-fixup-texttt))

(defun muse-latex-pdf-browse-file (file)
  (shell-command (concat "open " file)))

(defun muse-latex-pdf-generate (file output-path final-target)
  (muse-publish-transform-output
   file output-path final-target "PDF"
   (function
    (lambda (file output-path)
      (let ((command (format "cd \"%s\"; pdflatex \"%s\""
                             (file-name-directory output-path) file))
            (times 0)
            result)
        ;; XEmacs can sometimes return a non-number result.  We'll err
        ;; on the side of caution by continuing to attempt to generate
        ;; the PDF if this happens and treat the final result as
        ;; successful.
        (while (and (< times 2)
                    (or (not (numberp result))
                        (not (eq result 0))
                        ;; table of contents takes 2 passes
                        (file-readable-p
                         (muse-replace-regexp-in-string
                          "\\.tex\\'" ".toc" file t t))))
          (setq result (shell-command command)
                times (1+ times)))
        (if (or (not (numberp result))
                (eq result 0))
            t
          nil))))
   ".aux" ".toc" ".out" ".log"))

(unless (assoc "latex" muse-publishing-styles)
  (muse-define-style "latex"
                     :suffix    'muse-latex-extension
                     :regexps   'muse-latex-markup-regexps
                     :functions 'muse-latex-markup-functions
                     :strings   'muse-latex-markup-strings
                     :specials  'muse-latex-markup-specials
                     :after     'muse-latex-finalize-buffer
                     :header    'muse-latex-header
                     :footer    'muse-latex-footer
                     :browser   'find-file)

  (muse-derive-style "pdf" "latex"
                     :final   'muse-latex-pdf-generate
                     :browser 'muse-latex-pdf-browse-file
                     :link-suffix 'muse-latex-pdf-extension
                     :osuffix 'muse-latex-pdf-extension)

  (muse-derive-style "latexcjk" "latex"
                     :header    'muse-latexcjk-header
                     :footer    'muse-latexcjk-footer)

  (muse-derive-style "pdfcjk" "latexcjk"
                     :final   'muse-latex-pdf-generate
                     :browser 'muse-latex-pdf-browse-file
                     :link-suffix 'muse-latex-pdf-extension
                     :osuffix 'muse-latex-pdf-extension))

(provide 'muse-latex)

;;; muse-latex.el ends here
