/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: init.c,v 1.19 2004/09/12 07:59:55 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 *
 * Fix: PCI_CLASS and PCI_SUBSYS_ID variable
 * 			Shun-ichi TAHARA <jado@flowernet.gr.jp>
 */
#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<dirent.h>
#include	<fcntl.h>
#include	<string.h>
#include	<unistd.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<wait.h>
#define	__USE_XOPEN_EXTENDED 1
#include	<ftw.h>

#include	"murasaki.h"
#include	"murasaki_pci.h"

#define	PCI_PROC_DIR	"/proc/bus/pci"
#define	IGNORE_CLASS		"0000"
#define	IGNORE_SUBSYS_ID	"0000:0000"

#define	SCRATCH_BUFSIZE		256

/* global */
static int msg_level = MU_MSG_STD;
static char *sysfs_dir = NULL;	/* SYSFS dir */

struct pci_init {
	char *	bus;
	char *	slot_fn;
	u_int16_t vendor;
	u_int16_t device;
	u_int16_t sub_vendor;
	u_int16_t sub_device;
	u_int16_t class;
};

static void
execute_hotplug(struct pci_init *pci)
{
	char buf[SCRATCH_BUFSIZE],*scratch;
	char *argv[3],*envp[16];
	int i;

	switch (fork()) {
	case -1:
		LOG(MU_MSG_QUIET,msg_level,"fork error");
		exit(1);
	case 0:
		i = init_argument(argv,envp,MU_ARG_PCI);

		scratch = buf;
		envp[i++] = scratch;
		scratch += sprintf(scratch,"PCI_CLASS=%04X",pci->class);
		scratch++;
		envp[i++] = scratch;
		scratch += sprintf(scratch,"PCI_ID=%04X:%04X",pci->vendor,pci->device);
		scratch++;
		envp[i++] = scratch;
		scratch += sprintf(scratch,"PCI_SUBSYS_ID=%04X:%04X",pci->sub_vendor,pci->sub_device);
		envp[i++] = scratch;
		scratch++;
		sprintf(scratch,"PCI_SLOT_NAME=%s:%s",pci->bus,pci->slot_fn);
		envp[i] = 0;

		execve(argv[0], argv, envp);
		break;
	default:
		if (msg_level >= MU_MSG_TRACE) {
			wait(NULL);
		}
		break;
	}

}

#ifdef	ONLY_CARDBUS
static int
check_cardbus(char *config)
{
	unsigned char pci_header;
	u_int32_t cardbus_cis;

	pci_header = config[PCI_HEADER_TYPE];
	cardbus_cis = config[PCI_CARDBUS_CIS] |
		config[PCI_CARDBUS_CIS+1] << 8 |
		config[PCI_CARDBUS_CIS+2] << 16 |
		config[PCI_CARDBUS_CIS+3] << 24;
	LOG(MU_MSG_DESC,msg_level,"pci_header=0x%02x cardbus_cis=0x%08x",
			pci_header,cardbus_cis);
	if (pci_header != PCI_HEADER_TYPE_NORMAL ||
		cardbus_cis == 0) {

		LOG(MU_MSG_TRACE,msg_level,"Not CardBus device");

		return 0;
	}

	LOG(MU_MSG_TRACE,msg_level,"This is a CardBus device");

	return 1;
}
#else
static int
check_cardbus(char *config)
{
	return 1;
}
#endif	/* ONLY_CARDBUS */

static int
scan_device(char *dirname,struct pci_init *pci)
{
	int fd,size;
	char path[PATH_MAX];
	unsigned char buf[PCI_PROC_BUF];

	strcpy(path,dirname);
	strcat(path,"/");
	strcat(path,pci->slot_fn);

	if ((fd = open(path,O_RDONLY)) == -1) {
		LOG(MU_MSG_TRACE,msg_level,"open error %s: %m",path);

		return -1;
	}
	size = read(fd,buf,PCI_PROC_BUF);
	if (!check_cardbus(buf)) {
		close(fd);

		return 0;
	}
	pci->vendor = *(u_int16_t *)buf;
	pci->device = *((u_int16_t *)buf+1);
	pci->class = *((u_int16_t *)buf+5);
	pci->sub_vendor = *((u_int16_t *)buf+22);
	pci->sub_device = *((u_int16_t *)buf+23);
	LOG(MU_MSG_DESC,msg_level,"bus:%s slot_fn:%s vendor:%04x device:%04x subvendor:%04x subdevice %04x class:%04x",
			pci->bus,pci->slot_fn,pci->vendor,pci->device,pci->sub_vendor,pci->sub_device,pci->class);
	execute_hotplug(pci);

	close(fd);

	return 0;
}

static int
scan_bus(char *dirname,struct pci_init *pci)
{
	char path[PATH_MAX];
	int n, i;
	struct dirent **names, **ent;

	strcpy(path,dirname);
	strcat(path,"/");
	strcat(path,pci->bus);
	if ((n = scandir(path, &names, NULL, alphasort)) < 0) {
		LOG(MU_MSG_TRACE,msg_level,"scandir error %s",path);

		return -1;
	}
	for (i = 0, ent = names; i < n; i++, ent++) {
		if ((*ent)->d_name[0] == '.')
			continue;
		pci->slot_fn = (*ent)->d_name;
		if (scan_device(path,pci) != 0)
			break;
	}
	if (names != NULL)
		free(names);

	return 0;
}

/*
 * sysfs
 */

#ifdef	ONLY_CARDBUS
static int
check_cardbus_file(const char *path)
{
	int fd,ret;
	unsigned char buf[PCI_PROC_BUF];

	fd = open(path,O_RDONLY);
	if(fd == -1)
		return 0;
	if (read(fd,buf,PCI_PROC_BUF) == -1) {
		close(fd);

		return 0;
	}
	ret = check_cardbus(buf);
	close(fd);

	return ret;
}
#else
static int
check_cardbus_file(const char *path)
{
	return 1;
}
#endif	/* ONLY_CARDBUS */

static int
sysfs_sub(const char *path,const struct stat *st,int flag,struct FTW *s)
{
	char dname[PATH_MAX],*ptr;

	if (flag == FTW_F) {
		if (strcmp(path + s->base,"config") == 0
			&& check_cardbus_file(path)) {
			strcpy(dname,path+strlen(sysfs_dir));
			ptr = strrchr(dname,'/');
			*ptr = '\0';
			LOG(MU_MSG_TRACE,msg_level,"devpath=%s\n",dname);
			execute_hotplug_sysfs(dname,MU_ARG_PCI,WAIT_OFF,msg_level);
		}
	}

	return 0;
}

int
main(int argc,char **argv)
{
	int n, i;
	struct dirent **names, **ent;
	char *dirname = PCI_PROC_DIR;
	struct pci_init pci;
	char sysfs_devices[PATH_MAX];

	sysfs_dir = coldplug_init(MU_INIT_PCI,MU_ARG_PCI,&msg_level);
	if (msg_level == -1)
		exit(0);
	if (sysfs_dir != NULL) {
		strcpy(sysfs_devices,sysfs_dir);
		strcat(sysfs_devices,"/devices");
		nftw(sysfs_devices,sysfs_sub,16,FTW_PHYS);
	} else {
		if ((n = scandir(dirname, &names, NULL, alphasort)) < 0) {
			LOG(MU_MSG_TRACE,msg_level,"scandir error %s",dirname);

			exit(1);
		}
		for (i = 0, ent = names; i < n; i++, ent++) {
			if ((*ent)->d_name[0] == '.')
				continue;
			LOG(MU_MSG_TRACE,msg_level,"dirname %s",(*ent)->d_name);
			if (isdigit((*ent)->d_name[0]) &&
				isdigit((*ent)->d_name[1])) {
				pci.bus = (*ent)->d_name;
				scan_bus(dirname,&pci);
			}
		}
		if (names != NULL)
			free(names);
	}

	return 0;
}
