/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: depend.c,v 3.23 2004/09/13 14:43:28 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<strings.h>
#include	<ctype.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<sys/mman.h>

#include	"murasaki.h"

#define	CHAR_RSEP		'\n'
#define	CHAR_DEP		':'

#define DEPEND_PATH	MU_CONF_DIR "/" DEPEND_FILE
#define	CONFIG_PATH	MU_CONF_DIR "/" CONFIG_FILE
#define	CALL_PATH	MU_CONF_DIR "/" CALL_FILE
#define	PRECALL_PATH	MU_CONF_DIR "/" PRECALL_FILE

#define	DEVICE_NAME_MAX	32	/* Now max length = 13 "pcmcia_socket" */

enum string_status {ST_START, ST_END };

/*
 * expand_list: appended list
 * except_list: except list (including NULL)
 * ptr: point to next for ':'
 * endp: end pointer of a mapped file.
 */
static char *
append_depends(
	list_t *expand_list,
	list_t *except_list,
	char *ptr,
	char *endp,
	int msg_level)
{
	char *module,*mod_start = NULL;/* for warning */
	enum string_status status = ST_END;

	for(;ptr <= endp;ptr++) {
		if (status == ST_END) {
			if (*ptr == ' ' || *ptr == '\t') /* skip */
				continue;
			if (*ptr == CHAR_RSEP) 	/* invalid line end */
				goto line_end;
			status = ST_START;
			mod_start = ptr;
		} else {	/* status == ST_START */
			if (*ptr != ' ' && *ptr != '\t' && *ptr != CHAR_RSEP)
				continue;
			module = alloc_module(mod_start,ptr - mod_start);
			if (module == NULL)
				return NULL;
			if (strcasecmp(module,"off") == 0 || strcasecmp(module,"on") == 0) {
				LOG(MU_MSG_TRACE,msg_level,"Dependence(%s) was found",module);
			} else {
				LOG(MU_MSG_STD,msg_level,"Dependence(%s) was found",module);
			}
			/* 
			 * 2nd pass for except list(blacklist and sticky)
			 */
			if (except_list != NULL && except_list->used > 0) {
				if (find_str_list(except_list,module) == NOT_FOUND) {
					if (add_to_list(expand_list,module,msg_level) == INVALID)
						return NULL;
				} else {
					LOG(MU_MSG_STD,msg_level,"%s was excepted",module);
					free(module);
				}
			} else {
				if (add_to_list(expand_list,module,msg_level) == INVALID)
					return NULL;
			}
			if (*ptr == CHAR_RSEP)	/* all modules be appended */
				break;
			status = ST_END;	/* -> next modules */
		}
	}

line_end:
	return ptr;
}

/* 
 * expand_list: destination list for 'list'
 * src_list: source list should be expanded
 * except_list: except list (including NULL)
 *
 * search list[0] to list[list->size-1]
 * copy expanded list[#] to expand_list
 */

static int
get_helper(char *fname,
	list_t *expand_list,
	list_t *src_list,
	list_t *except_list,
	int msg_level)
{
	int base_len, index;
	char *mp,*endp,*ptr;
	unsigned int size;

	/*
	 * If the file is not exist, do nothing
	 */
	if ((mp = map_file(fname,&size,msg_level)) == NULL)
		return GOOD;
	endp = mp + size;

	/* try to match an argument for 'modprobe' */
	for(index = 0;index < src_list->used;index++) {
		base_len = strlen(src_list->list[index]);
		for(ptr = mp;ptr < endp;ptr++) {
			/* module is matched */
			if ((ptr[base_len] == CHAR_DEP) &&
				(ptr + base_len < endp) &&
				(memcmp(ptr,src_list->list[index],base_len) == 0)) {
				/* base_len + 1 -> 
				 * point to next character for ':'
				 */
				ptr = append_depends(expand_list,except_list,ptr+base_len+1,endp,msg_level);
				if (ptr == NULL) {
					LOG(MU_MSG_QUIET,msg_level,"%s is invalid",fname);
					munmap(mp,size);
					return INVALID;
				}
			} else {	/* skip an unmatched line */
				while(ptr < endp) {
					if (*ptr == CHAR_RSEP)
						break;
					ptr++;
				}
			}
		}
	}	/* list loop */

	munmap(mp,size);

	return GOOD;
}

/*
 * creating depended module list from modules
 */
int
get_depend(mu_op_t *opp)
{
	return get_helper(DEPEND_PATH,
		&(opp->module_list),
		&(opp->module_list),
		&(opp->except_list),
		opp->msg_level);
}

/*
 * create [0]:...    script-file
 */
int
get_module_script(mu_op_t *opp,int timing)
{
	char  *fname;

	if (timing == PRECALL)
		fname = PRECALL_PATH;
	else
		fname = CALL_PATH;

	CLEAR_LIST(opp->script_list);
	if (get_helper(fname, &(opp->script_list),&(opp->module_list),NULL,opp->msg_level) == INVALID)
		return INVALID;
	return get_helper(fname,&(opp->script_list),&(opp->alias_list),NULL,opp->msg_level); 
}

int
get_device_script(mu_op_t *opp,int timing)
{
	char  *fname;
	char device_name[DEVICE_NAME_MAX+3];	/* 3: '['+']'+'\0' */
	list_t device_list;
	unsigned int len;
	int ret;

	if (init_list(&device_list,2) == INVALID) {
		LOG_OPP_QUIET("Memory allocation failed"); 

		return INVALID;
	}

	if (timing == PRECALL)
		fname = PRECALL_PATH;
	else
		fname = CALL_PATH;
	if ((len = strlen(opp->device_name)) > DEVICE_NAME_MAX) {
		LOG_OPP_QUIET("Device name is too long \"%s\"",opp->device_name); 
		return INVALID;
	}
	device_name[0] = '[';
	strcpy(device_name+1,opp->device_name);
	device_name[1+len] = ']';
	device_name[1+len+1] = '\0';
	device_list.list[device_list.used++] = device_name;
	device_list.list[device_list.used] = NULL;

	CLEAR_LIST(opp->script_list);
	ret = get_helper(fname,&(opp->script_list),&device_list,NULL,opp->msg_level);
#ifdef	KERNEL_JOB
	free_list(&device_list);
#endif

	return ret;
}

#define	CONFIG_LISTMAX	3

/*
 * check if the argument is on/off
 * default: on
 */
int
check_config(char *arg,int msg_level)
{
	list_t config_list;
	int ret = ARG_ON;	/* default */

	if (init_list(&config_list,CONFIG_LISTMAX) == INVALID) {
		LOG(MU_MSG_QUIET,msg_level,"Memory allocation failed");

		return INVALID;
	}

	config_list.list[config_list.used++] = arg;
	config_list.list[config_list.used] = NULL;
	if (get_helper(CONFIG_PATH,&config_list,&config_list,NULL,msg_level) == INVALID) {
		LOG(MU_MSG_WHISPER,msg_level,"%s cound Not be opened", config_list.list[0]);
		goto RET;
	}
	if (config_list.list[1] && strcasecmp(config_list.list[1],"off") == 0) {
		LOG(MU_MSG_STD,msg_level,"%s is defined as \"%s\"",
			config_list.list[0],config_list.list[1]);
		ret = ARG_OFF;
	}

RET:
#ifdef	KERNEL_JOB
	free_list(&config_list);
#endif
	return ret;
}

/*
 * check number
 * default: -1
 */
int
check_config_val(char *arg,int msg_level)
{
	list_t config_list;
	int ret = -1;	/* default */

	if (init_list(&config_list,CONFIG_LISTMAX) == INVALID) {
		LOG(MU_MSG_QUIET,msg_level,"Memory allocation failed");

		return INVALID;
	}

	config_list.list[config_list.used++] = arg;
	config_list.list[config_list.used] = NULL;
	if (get_helper(CONFIG_PATH,&config_list,&config_list,NULL,msg_level) == INVALID) {
		LOG(MU_MSG_WHISPER,msg_level,"%s cound Not be opened", config_list.list[0]);
		goto RET;
	}
	if (config_list.used == 2 && isdigit(config_list.list[1][0])) {
		ret = (int)strtoul(config_list.list[1],NULL,0);
	}

RET:
#ifdef	KERNEL_JOB
	free_list(&config_list);
#endif
	return ret;
}

/*
 * for pcmcia
 */
int
get_resource(char *path,list_t *lst,int msg_level)
{
	int ret;

	ret = get_helper(path,lst,lst,NULL,msg_level);
	if (ret == INVALID)
		LOG(MU_MSG_QUIET,msg_level,"%s cound Not be identified.", lst->list[0]);

	return ret;
}
