/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: match.c,v 1.2 2004/09/12 08:04:48 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<stdlib.h>

#include	"murasaki.h"
#include	"murasaki_input.h"

#define	CHAR_COMMENT		'#'
#define	CHAR_RSEP		'\n'

#define	ST_COMMENT		255
#define	ST_MODULE		1
#define	ST_MATCH_FLAGS		2
#define	ST_BUSTYPE		3
#define	ST_VENDOR		4
#define	ST_PRODUCT		5
#define	ST_VERSION		6
#define	ST_EVBITS		7
#define	ST_KEYBITS		8
#define	ST_RELBITS		9
#define	ST_ABSBITS		10
#define	ST_MSCBITS		11
#define	ST_LEDBITS		12
#define	ST_SNDBITS		13
#define	ST_FFBITS		14
#define	ST_INFO			15

#define	PRINT_BUF_SIZE		512

static int
match_bits(unsigned long *array,unsigned long *map_array,int array_size)
{
	int i;

	for(i = 0;i < array_size;i++) {
		if ((map_array[i] & array[i]) != map_array[i])
			break;
	}
	if (i != array_size)
		return INVALID;

	return GOOD;
}

static int
real_match(MU_input_config_t *active,MU_input_config_t *map)
{
	if (map->match_flags == 0 && map->driver_info != 0)
		return GOOD;

	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_BUS) &&
		(active->bustype != map->bustype))
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_VENDOR) &&
		(active->vendor != map->vendor))
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_PRODUCT) &&
		(active->product != map->product))
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_VERSION) &&
		(active->version != map->version))
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_EVBIT) &&
		match_bits(active->evbit,map->evbit,NBITS(EV_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_KEYBIT) &&
		match_bits(active->keybit,map->keybit,NBITS(KEY_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_RELBIT) &&
		match_bits(active->relbit,map->relbit,NBITS(REL_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_ABSBIT) &&
		match_bits(active->absbit,map->absbit,NBITS(ABS_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_MSCBIT) &&
		match_bits(active->mscbit,map->mscbit,NBITS(MSC_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_LEDBIT) &&
		match_bits(active->ledbit,map->ledbit,NBITS(LED_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_SNDBIT) &&
		match_bits(active->sndbit,map->sndbit,NBITS(SND_MAX)) == INVALID)
		return INVALID;
	if ((map->match_flags & INPUT_DEVICE_ID_MATCH_FFBIT) &&
		match_bits(active->ffbit,map->ffbit,NBITS(FF_MAX)) == INVALID)
		return INVALID;

	return GOOD;
}


/*
 * called if msg_level >= STD
 */
static void
print_matching(MU_input_config_t *map,char *module)
{
	char buf[PRINT_BUF_SIZE],*scratch;

	scratch = buf;

	scratch = push_bit(scratch,"evbit",map->evbit,NBITS(EV_MAX));
	scratch = push_bit(scratch,"keybit",map->keybit,NBITS(KEY_MAX));
	scratch = push_bit(scratch,"relbit",map->relbit,NBITS(REL_MAX));
	scratch = push_bit(scratch,"absbit",map->absbit,NBITS(ABS_MAX));
	scratch = push_bit(scratch,"mscbit",map->mscbit,NBITS(MSC_MAX));
	scratch = push_bit(scratch,"ledbit",map->ledbit,NBITS(LED_MAX));
	scratch = push_bit(scratch,"sndbit",map->sndbit,NBITS(SND_MAX));
	scratch = push_bit(scratch,"ffbit",map->ffbit,NBITS(FF_MAX));

	syslog(LOG_LEVEL,"MATCH(%s) -> match_flags:0x%x bustype:0x%x vendor:0x%x product:0x%x version:0x%x %s driver_info:0x%lx",
		module,map->match_flags,
		map->bustype,map->vendor,map->product,map->version,
		buf,
		map->driver_info
		);
}

int
match_config(mu_op_t *opp,char *start,char *end)
{
	MU_input_config_t *config = opp->config;
	MU_input_config_t map;
	char *ptr,*module,*module_start;
	int status = ST_MODULE;
	unsigned int module_len = 0,i;
	unsigned long value;
	char *fwdp;

	for(module_start = ptr = start; ptr < end;ptr++) {
		if (status == ST_COMMENT) {
			while(*ptr != CHAR_RSEP) {
				if (ptr >= end)
					goto error_return;
				ptr++;
			}
			status = ST_MODULE;
			continue;
		} else if (*ptr == CHAR_COMMENT) {
			status = ST_COMMENT;
			continue;
		}
		if (status == ST_MODULE) {
			module_start = ptr;
			while(*ptr != ' ' && *ptr != '\t') {
				if (ptr >= end)
					goto error_return;
				ptr++;
			}
			/* ptr == next to module end */
			module_len = ptr - module_start;
			/*
			 * check excepted modules
			 */
			if ((i = find_mem_list(&(opp->except_list),module_start,ptr)) != NOT_FOUND) {
				status = ST_COMMENT;
				LOG_OPP_STD("%s was skipped in inputmap",opp->except_list.list[i]);
				continue;
			}
		} else {
			if ((ST_MATCH_FLAGS <= status && status <= ST_VERSION) || status == ST_INFO) {
				value = (unsigned long)strtoul(ptr,&fwdp,0);
				switch(status) {
				case ST_MATCH_FLAGS:
					map.match_flags = (unsigned int)value;
					break;
				case ST_BUSTYPE:
					map.bustype = value;
					break;
				case ST_VENDOR:
					map.vendor = value;
					break;
				case ST_PRODUCT:
					map.product = value;
					break;
				case ST_VERSION:
					map.version = value;
					break;
				case ST_INFO:
					map.driver_info = value;
					status = ST_MODULE;
					/* need exclusive ? */
					if (real_match(config,&map) == GOOD) {
						config->match_flags = map.match_flags;
						module = alloc_module(module_start,module_len);
						if (module == NULL)
							goto error_return;
						if (opp->msg_level >= MU_MSG_STD)
							print_matching(&map,module);
						if (add_to_list(&(opp->module_list),module,opp->msg_level) == INVALID)
							goto error_return;
							
					}
					ptr = fwdp;
					continue;
				}
			} else {
				while(*ptr == ' ' || *ptr == '\t')
					ptr++;
				fwdp = ptr;
				while(*fwdp != ' ' && *fwdp != '\t') {
					if (*fwdp == '\n')
						goto error_return;
					fwdp++;
				}
				switch(status) {
				case ST_EVBITS:
					get_id_input_array_colon(ptr,fwdp,map.evbit,NBITS(EV_MAX));
					break;
				case ST_KEYBITS:
					get_id_input_array_colon(ptr,fwdp,map.keybit,NBITS(KEY_MAX));
					break;
				case ST_RELBITS:
					get_id_input_array_colon(ptr,fwdp,map.relbit,NBITS(REL_MAX));
					break;
				case ST_ABSBITS:
					get_id_input_array_colon(ptr,fwdp,map.absbit,NBITS(ABS_MAX));
					break;
				case ST_MSCBITS:
					get_id_input_array_colon(ptr,fwdp,map.mscbit,NBITS(MSC_MAX));
					break;
				case ST_LEDBITS:
					get_id_input_array_colon(ptr,fwdp,map.ledbit,NBITS(LED_MAX));
					break;
				case ST_SNDBITS:
					get_id_input_array_colon(ptr,fwdp,map.sndbit,NBITS(SND_MAX));
					break;
				case ST_FFBITS:
					get_id_input_array_colon(ptr,fwdp,map.ffbit,NBITS(FF_MAX));
					break;
				default:
					goto error_return;
				}
			}
			ptr = fwdp;
		}
		status++;
	}

	return GOOD;

error_return:
	return INVALID;
}
