/*
 * net.c
 * 
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2000, 2003, 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#ifdef WINDOWS
#include <winsock2.h>
#ifdef HAVE_GETADDRINFO
#include <ws2tcpip.h>
#endif
#elif defined (DJGPP)
#include <unistd.h>
#include <tcp.h>
#include <netdb.h>
#include <errno.h>
extern int errno;
#else /* UNIX */
#include <unistd.h>
#ifndef HAVE_GETADDRINFO
#include <netinet/in.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
extern int errno;
#endif /* UNIX */

#include "net.h"
#include "merror.h"


/*
 * [WINDOWS only] wsa_strerror() 
 *
 * This function translates WSA error codes to strings.
 * It should translate all codes that could be caused by the Windows socket 
 * functions used in this file:
 * WSAStartup, getaddrinfo() or gethostbyname(), socket(), connect(), 
 * recv(), send()
 */

#ifdef WINDOWS
char *wsa_strerror(int error_code)
{
    char *error_string;
    
    switch (error_code)
    {
	case WSA_NOT_ENOUGH_MEMORY:
	    error_string = "not enough memory";
	    break;
	    
	case WSAEINVAL:
	    error_string = "invalid argument";
	    break;
	    
	case WSATYPE_NOT_FOUND:
    	    error_string = "class type not found";
	    break;
	    
	case WSAENETDOWN:
    	    error_string = "the network subsystem has failed";
	    break;
	    
	case WSAHOST_NOT_FOUND:
    	    error_string = "host not found (authoritative)";
	    break;
	    
	case WSATRY_AGAIN:
    	    error_string = "host not found (nonauthoritative) or server failure";
	    break;
	    
	case WSANO_RECOVERY:
    	    error_string = "nonrecoverable error";
	    break;
	    
	case WSANO_DATA:
    	    error_string = "valid name, but no data record of requested type";
	    break;

	case WSAEAFNOSUPPORT:
	    error_string = "address family not supported";
	    break;
	    
	case WSAEMFILE:
	    error_string = "no socket descriptors available";
	    break;

	case WSAENOBUFS:
	    error_string = "no buffer space available";
	    break;

	case WSAEPROTONOSUPPORT:
	    error_string = "protocol not supported";
	    break;

	case WSAEPROTOTYPE:
	    error_string = "wrong protocol type for this socket";
	    break;

	case WSAESOCKTNOSUPPORT:
	    error_string = "socket type is not supported in this address family";
	    break;
	    
	case WSAEADDRNOTAVAIL:
	    error_string = "remote address is not valid";
	    break;

	case WSAECONNREFUSED:
	    error_string = "connection refused";
	    break;

	case WSAENETUNREACH:
	    error_string = "network unreachable";
	    break;

	case WSAETIMEDOUT:
	    error_string = "timeout";
	    break;
	    
	case WSAENOTCONN:
	    error_string = "socket not connected";
	    break;

	case WSAESHUTDOWN:
	    error_string = "the socket was shut down";
	    break;

	case WSAEHOSTUNREACH:
	    error_string = "host unreachable";
	    break;

	case WSAECONNRESET:
	    error_string = "connection reset by peer";
	    break;
	    
	case WSASYSNOTREADY:
    	    error_string = "the underlying network subsystem is not ready";
	    break;
	    
	case WSAVERNOTSUPPORTED:
    	    error_string = "the requested version is not available";
	    break;
	    
	case WSAEINPROGRESS:
    	    error_string = "a blocking operation is in progress";
	    break;
	    
	case WSAEPROCLIM:
    	    error_string = "limit on the number of tasks has been reached";
	    break;
	    
	case WSAEFAULT:
    	    error_string = "invalid request";
	    break;

	default:
    	    error_string = "unknown error";
	    break;
    }

    return error_string;
}
#endif /* WINDOWS */


/*
 * net_lib_init()
 *
 * see net.h
 */

merror_t net_lib_init(void)
{
#ifdef WINDOWS
   
    WORD wVersionRequested;
    WSADATA wsaData;
    int error_code;
    
    wVersionRequested = MAKEWORD(2, 0);
    if ((error_code = WSAStartup(wVersionRequested, &wsaData)) != 0)
    {
	return merror(NET_ELIBFAILED, 
		"Failed to initialize networking DLL: %s", 
		wsa_strerror(error_code));
    }
    else
    {
	return merror(EOK, NULL);
    }
    
#else
    
    return merror(EOK, NULL);
    
#endif /* WINDOWS */    
}


/*
 * net_close_socket()
 *
 * [This function is needed because Windows cannot just close() a socket].
 *
 * see net.h
 */

void net_close_socket(int fd)
{
#ifdef WINDOWS
    (void)closesocket(fd);
#else
    (void)close(fd);
#endif
}


/*
 * open_socket() 
 *
 * see net.h
 */

merror_t net_open_socket(char *hostname, int port, int *ret_fd)
{    
#ifdef HAVE_GETADDRINFO
    
    int fd;
    char port_string[6];
    struct addrinfo hints;
    struct addrinfo *res0;
    struct addrinfo *res;
    int error_code;
    int cause;
    
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = 0;
    hints.ai_protocol = 0;
    hints.ai_addrlen = 0;
    hints.ai_canonname = NULL;
    hints.ai_addr = NULL;
    hints.ai_next = NULL;
    snprintf(port_string, 6, "%d", port);
    error_code = getaddrinfo(hostname, port_string, &hints, &res0);
    if (error_code)
    {
	return merror(NET_EHOSTNOTFOUND, "cannot locate host %s: %s", 
		hostname,
#ifdef WINDOWS
		wsa_strerror(error_code)
#else
		gai_strerror(error_code)
#endif
		);
    }

    fd = -1;
    cause = 0;
    for (res = res0; res; res = res->ai_next)
    {
	fd = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (fd < 0)
	{
	    cause = 1;
	    continue;
	}
	if (connect(fd, res->ai_addr, res->ai_addrlen) < 0) 
	{
	    cause = 2;
	    net_close_socket(fd);
	    fd = -1;
	    continue;
	}
	break;
    }
    freeaddrinfo(res0);
    if (fd < 0)
    {
	if (cause == 1)
	{
	    return merror(NET_ESOCKET, "cannot create socket: %s", 
#ifdef WINDOWS
		    wsa_strerror(WSAGetLastError())
#else
		    strerror(errno)
#endif
		    );
	}
	else if (cause == 2)
	{
	    return merror(NET_ECONNECT, "cannot connect to %s, port %d: %s", 
		    hostname, port,
#ifdef WINDOWS
		    wsa_strerror(WSAGetLastError())
#else
		    strerror(errno)
#endif
		    );
	}
	else /* cause == 0, can't happen */
	{
	    return merror(NET_EHOSTNOTFOUND, 
		    "cannot locate host %s: "
		    "getaddrinfo() returned crap", hostname);
	}
    }
    
    *ret_fd = fd;
    return merror(EOK, NULL);

#else /* !HAVE_GETADDRINFO */

    int fd;
    struct sockaddr_in sock;
    struct hostent *remote_host;
    
    if (!(remote_host = gethostbyname(hostname)))
    {
	return merror(NET_EHOSTNOTFOUND, 
		"cannot locate host %s: %s", hostname, 
#ifdef WINDOWS
		wsa_strerror(WSAGetLastError())
#else
		hstrerror(h_errno)
#endif
		);
    }

    if ((fd = socket(PF_INET, SOCK_STREAM, 0)) < 0)
    {
	return merror(NET_ESOCKET, "cannot create socket: %s", 
#ifdef WINDOWS
		wsa_strerror(WSAGetLastError())
#else
		strerror(errno)
#endif
		);
    }
    
    sock.sin_family = AF_INET;
    sock.sin_port = htons((unsigned short int)port);
    memcpy(&sock.sin_addr, remote_host->h_addr_list[0], (size_t)remote_host->h_length);

    if (connect(fd, (struct sockaddr *)(&sock), sizeof(sock)) < 0)
    {
	return merror(NET_ECONNECT, "cannot connect to %s, port %d: %s",
		hostname, port,
#ifdef WINDOWS
		wsa_strerror(WSAGetLastError())
#else
		strerror(errno)
#endif
		);
    }

    *ret_fd = fd;
    return merror(EOK, NULL);

#endif /* HAVE_GETADDRINFO */
}


/*
 * net_getchar()
 *
 * see net.h
 */

merror_t net_getchar(int fd, char *c, int *eof)
{
#ifdef WINDOWS
    int error_code;
    
    if ((error_code = recv(fd, c, 1, 0)) == 1)
    {
	*eof = 0;
	return merror(EOK, NULL);
    }
    else if (error_code == 0)
    {
	*eof = 1;
	return merror(EOK, NULL);
    }
    else
    {
	return merror(NET_EIO, 
		"network read error: %s", wsa_strerror(WSAGetLastError()));
    }
    
#else /* !WINDOWS */
    
    ssize_t error_code;
    
    if ((error_code = read(fd, c, (size_t)1)) == 1)
    {
	*eof = 0;
	return merror(EOK, NULL);
    }
    else if (error_code == 0)
    {
	*eof = 1;
	return merror(EOK, NULL);
    }
    else /* error_code == -1 */
    {
	return merror(NET_EIO, "network read error: %s", strerror(errno));
    }
#endif /* !WINDOWS */
}


/*
 * net_gets()
 *
 * see net.h
 */

merror_t net_gets(int fd, char *line, int size)
{
    char c;
    int i;
    int eof;
    merror_t e;

    i = 0;
    while (merror_ok(e = net_getchar(fd, &c, &eof)) && !eof)
    {
	line[i++] = c;
	if (c == '\n' || i == size - 1)
	{
	    break;
	}
    }
    line[i] = '\0';

    return e;
}


/*
 * net_puts()
 *
 * see net.h
 */

merror_t net_puts(int fd, char *s)
{
#ifdef WINDOWS
    int error_code;
    size_t count;

    count = strlen(s);    
    error_code = send(fd, s, count, 0);
    if (error_code < 0)
    {
	return merror(NET_EIO, 
		"network write error: %s", wsa_strerror(WSAGetLastError()));
    }
    else if ((size_t)error_code == count)
    {
	return merror(EOK, NULL);
    }
    else /* 0 <= error_code < count */
    {
	return merror(NET_EIO, "network write error");
    }
    
#else /* !WINDOWS */

    ssize_t error_code;
    size_t count;

    count = strlen(s);    
    error_code = write(fd, s, count);
    if (error_code < 0)
    {
	return merror(NET_EIO, "network write error: %s", strerror(errno));
    }
    else if ((size_t)error_code == count)
    {
	return merror(EOK, NULL);
    }
    else /* 0 <= error_code < count */
    {
	return merror(NET_EIO, "network write error");
    }
#endif /* !WINDOWS */
}


/*
 * net_lib_deinit()
 *
 * see net.h
 */

void net_lib_deinit(void)
{
#ifdef WINDOWS
    (void)WSACleanup();
#endif /* WINDOWS */
}
