/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/system/memory.h>

using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

#define MRPT_DISABLE_ALIGNED_MALLOCS  1
//#define MRPT_DISABLE_ALIGNED_MALLOCS  0

/** Returns an aligned memory block.
  * \param alignment The desired alignment, typ. 8 or 16 bytes. 1 means no alignment required. It must be a power of two.
  * \sa aligned_free, aligned_realloc
  * \note Based on code by William Chan
*/
void* mrpt::system::os::aligned_malloc(size_t bytes, size_t alignment)
{
// allocate memory size = bytes + alignment + sizeof(alignment)
//	bytes               - obvious to allocate at least the user requested amount
//	alignment           - this is the padding in front or behind or both the memory block
//	sizeof(alignment)   - this is to save the real value pointer value (ptr vs aligned_ptr) before the aligned memory block
//						- this is necessary to free the entire memory block and not just the aligned memory block
//						- sizeof(size_t) was chosen over sizeof(unsigned int) for compatibility of 32 and 64-bit architectures
#if MRPT_DISABLE_ALIGNED_MALLOCS
	return ::malloc(bytes);
#else
	ASSERT_(alignment>0)
	ASSERTMSG_( ((alignment - 1) & alignment) == 0, "alignment is not a power of 2")

	uintptr_t ptr = (uintptr_t)::malloc(bytes + alignment + sizeof(uintptr_t)); //allocate enough memory for everything

	if(!ptr) //failed malloc! :(
		return NULL;

	uintptr_t aligned_ptr = (ptr + alignment + sizeof(uintptr_t)) & (~(alignment - 1)); //align!

	*(uintptr_t*)(aligned_ptr - sizeof(uintptr_t)) = ptr; //store the entire memory block address before the aligned block

	return (void*)aligned_ptr;
#endif
}

/** Frees a memory block reserved by aligned_malloc.
  * \param alignment The desired alignment, typ. 8 or 16 bytes. 1 means no alignment required.
  * If old_ptr is NULL, a new block will be reserved from scratch.
  * \sa aligned_malloc, aligned_free
  */
void* mrpt::system::os::aligned_realloc(void* old_ptr, size_t bytes, size_t alignment)
{
#if MRPT_DISABLE_ALIGNED_MALLOCS
	return ::realloc(old_ptr,bytes);
#else
	ASSERT_(alignment>0)
	ASSERTMSG_( ((alignment - 1) & alignment) == 0, "alignment is not a power of 2")

	if (!old_ptr)
	{
		// It's a malloc
		return mrpt::system::os::aligned_malloc(bytes, alignment);
	}
	else
	{
		// Real realloc
		uintptr_t original_ptr = (*((uintptr_t*)((uintptr_t)old_ptr - sizeof(uintptr_t)))); // Recover the address of the block:

		uintptr_t new_ptr = (uintptr_t ) ::realloc((void*)original_ptr, bytes + alignment + sizeof(uintptr_t)); //allocate enough memory for everything

		if(!new_ptr) //failed malloc! :(
			return NULL;

		uintptr_t aligned_ptr = (new_ptr + alignment + sizeof(uintptr_t)) & (~(alignment - 1)); //align!

		*(uintptr_t*)(aligned_ptr - sizeof(uintptr_t)) = new_ptr; //store the entire memory block address before the aligned block

		return (void*)aligned_ptr;
	}
#endif
}

/** Frees a memory block reserved by aligned_malloc
  * \sa aligned_malloc
  */
void mrpt::system::os::aligned_free(void* p)
{
#if MRPT_DISABLE_ALIGNED_MALLOCS
	return ::free(p);
#else
	if (p)
		free((void*)(*((uintptr_t*)((uintptr_t)p - sizeof(uintptr_t))))); //get the pointer of the entire block and then FREE!
#endif
}

// A malloc + zeroes
void * mrpt::system::os::aligned_calloc(size_t bytes, size_t alignment)
{
	void *ptr = mrpt::system::os::aligned_malloc(bytes, alignment);
	if (ptr)
		::memset(ptr,0,bytes);
	return ptr;
}

#ifdef  MRPT_OS_WINDOWS
#include <windows.h>
	// Windows:
	typedef struct _PROCESS_MEMORY_COUNTERS {
	  DWORD cb;
	  DWORD PageFaultCount;
	  SIZE_T PeakWorkingSetSize;
	  SIZE_T WorkingSetSize;
	  SIZE_T QuotaPeakPagedPoolUsage;
	  SIZE_T QuotaPagedPoolUsage;
	  SIZE_T QuotaPeakNonPagedPoolUsage;
	  SIZE_T QuotaNonPagedPoolUsage;
	  SIZE_T PagefileUsage;
	  SIZE_T PeakPagefileUsage;
	} PROCESS_MEMORY_COUNTERS,
	*PPROCESS_MEMORY_COUNTERS;

	namespace mrpt
	{
		namespace system
		{
			/** This is an auxiliary class for mrpt::system::getMemoryUsage() under Windows.
			  *  It loads in runtime PSAPI.DLL. This is to avoid problems in some platforms, i.e Windows 2000,
			  *  where this DLL must not be present.
			  */
			class CAuxPSAPI_Loader
			{
			protected:
				typedef BOOL (WINAPI *TGetProcessMemoryInfo)(
				  HANDLE Process,
				  PPROCESS_MEMORY_COUNTERS ppsmemCounters,
				  DWORD cb );

				TGetProcessMemoryInfo		m_ptr;

			public:
				HMODULE m_dll;

				CAuxPSAPI_Loader()
				{
					m_ptr = NULL;

					m_dll = LoadLibraryA("PSAPI.DLL");
					if (m_dll)
					{
						m_ptr = (TGetProcessMemoryInfo) GetProcAddress(m_dll,"GetProcessMemoryInfo");
					}
				}
				~CAuxPSAPI_Loader()
				{
					if (m_dll)
					{
						FreeLibrary(m_dll);
						m_dll = NULL;
						m_ptr = NULL;
					}
				}

				BOOL WINAPI GetProcessMemoryInfo(
				  HANDLE Process,
				  PPROCESS_MEMORY_COUNTERS ppsmemCounters,
				  DWORD cb )
				{
					try
					{
						if (!m_ptr)
								return false;
						else	return (*m_ptr )(Process,ppsmemCounters,cb);
					}
					catch(...)
					{
						return false;
					}
				}
			};
		}
	}

#endif

/*---------------------------------------------------------------
						getMemoryUsage
 ---------------------------------------------------------------*/
unsigned long  mrpt::system::getMemoryUsage()
{
	MRPT_START
	unsigned long MEM = 0;

#ifdef  MRPT_OS_WINDOWS
	// Windows:
	static CAuxPSAPI_Loader		PSAPI_LOADER;

	PROCESS_MEMORY_COUNTERS		pmc;
	pmc.cb = sizeof(pmc);

	if ( PSAPI_LOADER.GetProcessMemoryInfo( GetCurrentProcess(),&pmc,sizeof(pmc)  ) )
	{
		MEM = (long)pmc.PagefileUsage;
	}
#else
	// Linux:
	//int page_size = getpagesize();

	FILE *f = ::fopen ("/proc/self/stat", "r");
	if (!f) return 0;

	// Note: some of these scanf specifiers would normally be 'long' versions if
	// not for the fact that we are using suppression (gcc warns).  see 'man
	// proc' for scanf specifiers and meanings.
	if (!::fscanf(f,
		"%*d %*s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %*u %*u %*d %*d %*d %*d %*d %*d %*u %lu", &MEM))
	{
		// Error parsing:
		MEM=0;
	}
	::fclose (f);
	//::system("cat /proc/self/statm");
#endif

	return MEM;
	MRPT_END
}

