/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/slam/CICP.h>
#include <mrpt/slam/CMetricMapBuilderICP.h>
#include <mrpt/slam/CSimplePointsMap.h>
#include <mrpt/poses/CPose3DPDFGaussian.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/utils/CEnhancedMetaFile.h>

using namespace mrpt::slam; using namespace mrpt::utils; using namespace mrpt::poses;
using namespace mrpt::utils;


/*---------------------------------------------------------------
							Constructor
  ---------------------------------------------------------------*/
CMetricMapBuilderICP::CMetricMapBuilderICP(
	TSetOfMetricMapInitializers	*mapInitializers,
	float						_insertionLinDistance,
	float						_insertionAngDistance,
	CICP::TConfigParams			*icpParams
	) :
		ICP_options(),
		ICP_params(),
		SF_Poses_seq(),
		metricMap( mapInitializers ),
		isTheFirstObservation ( true ),
		currentMapFile(),
		estPose(),
		estRobotPath(),
		insertionLinDistance ( _insertionLinDistance ),
		insertionAngDistance ( _insertionAngDistance ),
		linDistSinceLast(0),
		angDistSinceLast(0)
{
	if (icpParams)
		ICP_params = *icpParams;
}

/*---------------------------------------------------------------
							Destructor
  ---------------------------------------------------------------*/
CMetricMapBuilderICP::~CMetricMapBuilderICP()
{
	MRPT_START;

	// Asure, we have exit all critical zones:
	enterCriticalSection();
	leaveCriticalSection();

	// Save current map to current file:
	setCurrentMapFile("");

	MRPT_END;
}

/*---------------------------------------------------------------

						processActionObservation

  ---------------------------------------------------------------*/
void  CMetricMapBuilderICP::processActionObservation(
			CActionCollection	&action,
			CSensoryFrame		&in_SF )
{
	// Critical section: We are using our global metric map
	enterCriticalSection();

	MRPT_START;

	float		ret = 0;


	CTicTac					tictac;
	CICP					ICP;
	CICP::TReturnInfo		icpReturn;
	CSimplePointsMap		sensedPoints;
	float					runningTime;
	CActionRobotMovement2DPtr movEstimation = action.getBestMovementEstimation();

	// A mov. estimation must exist!
	ASSERT_(movEstimation);

	// Approximate the robot current pose:
	// --------------------------------------------------------------
	CPose2D	lastRobotPose = estPose.getMeanVal();
	CPose2D	initialEstimatedRobotPose = lastRobotPose +movEstimation->poseChange->getMeanVal();


	// Reflect observations into a points map:
	// --------------------------------------------------------------
	sensedPoints.insertionOptions.minDistBetweenLaserPoints = 0.02f;
	sensedPoints.insertionOptions.also_interpolate = false;

	sensedPoints.clear();
	// Insert the points depending on finding matchings in a grid or a points map:
//	in_SF.insertObservationsInto( &sensedPoints );

	// Execute ICP over the current points map and the sensed points:
	// ----------------------------------------------------------------------
	icpReturn.cbSize = sizeof(icpReturn);

	ICP.options = ICP_params;

	CMetricMap	*matchWith;

	if (ICP_options.matchAgainstTheGrid && !metricMap.m_gridMaps.empty() )
	{
		matchWith = static_cast<CMetricMap*>(metricMap.m_gridMaps[0].pointer());

		// Insert only those planar range scans in the altitude of the grid map:
		if (metricMap.m_gridMaps[0]->insertionOptions.useMapAltitude)
		{
			// Use altitude:
			for (CSensoryFrame::iterator it=in_SF.begin();it!=in_SF.end();++it)
				if ( (*it)->GetRuntimeClass() == CLASS_ID( CObservation2DRangeScan ) )
					if ( fabs( metricMap.m_gridMaps[0]->insertionOptions.mapAltitude - static_cast<CObservation2DRangeScan*>(it->pointer())->sensorPose.z())<0.01)
						sensedPoints.insertObservation( it->pointer() );
		}
		else
		{
			// Do not use grid altitude:
			in_SF.insertObservationsInto( &sensedPoints );
		}
	}
	else
	{
		ASSERT_( metricMap.m_pointsMaps.size() );
		matchWith = static_cast<CMetricMap*>(metricMap.m_pointsMaps[0].pointer());
		in_SF.insertObservationsInto( &sensedPoints );
	}

    CPosePDFPtr pestPose=	ICP.Align(
        matchWith,					// Map 1
        &sensedPoints,				// Map 2
        initialEstimatedRobotPose,	// a first gross estimation of map 2 relative to map 1.
        &runningTime,				// Running time
        &icpReturn					// Returned information
        );
    estPose.copyFrom( *pestPose );
	//delete pestPose; pestPose=NULL;

	// Estimated pose: (It is stored in estPose)
	// ----------------------------------------------------
	CPose2D	estimatedPose;
	estPose.getMean(estimatedPose);

	CPose2D			Apose = estimatedPose - lastRobotPose;
	linDistSinceLast += Apose.norm();
	angDistSinceLast += fabs( Apose.phi() );

	// Debug
	/** /
	{
		CPose2D p = pestPose->getEstimatedPose();
		CMatrix c = pestPose->getEstimatedCovariance();
		FILE	*f=os::fopen("debug_icps.txt","at");
		os::fprintf(f,"%f %f %f %f %f\n",
			        p.x,p.y,
					c(0,0),c(1,1),c(0,1));

		os::fclose(f);
	} / **/


	// Decide to update the map or not:
	// ----------------------------------------------------------------------
/*	float		current_info_index;
	float		newStaticPointsRatio;
	newStaticPointsRatio = metricMap.getNewStaticPointsRatio(
			&sensedPoints,
			estimatedPose );

	current_info_index = newStaticPointsRatio * icpReturn.goodness;
*/

	// Debug output to console:
	// ----------------------------------------------------
	if (options.verbose)
	{
		std::cout << "[performSLAM]  " << initialEstimatedRobotPose << "->" << estimatedPose << std::endl;
		printf("   Fit:%.1f%% Itr:%i In %.02fms \n\n",
			icpReturn.goodness*100,
			icpReturn.nIterations,
			1000*runningTime );
	}

	// ----------------------------------------------------------
	//							CRITERION
	// ----------------------------------------------------------
	//bool	update = (current_info_index > newInfoUpdateThreshold ); // && (icpReturn.goodness>0.5);
	bool update = 	(linDistSinceLast>insertionLinDistance ||
					angDistSinceLast>insertionAngDistance )
					&& (icpReturn.goodness>0.40f);

	// Always insert into map when SF includes an image:
	if (options.insertImagesAlways)
	{
		for ( CSensoryFrame::iterator it=in_SF.begin();it!=in_SF.end() && !update;++it)
			if ( (*it)->GetRuntimeClass() == CLASS_ID(CObservationImage))
				update = true;
	}

	// ----------------------------------------------------------
	// ----------------------------------------------------------
	if ( options.enableMapUpdating && ( update || isTheFirstObservation ))// ((0==metricMap.pointsMap->getPointsCount() ) )
	{
		// Reset distance counters:
		linDistSinceLast = angDistSinceLast = 0;
		isTheFirstObservation = false;

		printf("[performSLAM] Updating map...\n");
		tictac.Tic();

		if (metricMap.m_pointsMaps.size())
		{
			metricMap.m_pointsMaps[0]->insertionOptions.disableDeletion = true;
			metricMap.m_pointsMaps[0]->insertionOptions.fuseWithExisting = true;
			metricMap.m_pointsMaps[0]->insertionOptions.isPlanarMap = true;
			metricMap.m_pointsMaps[0]->insertionOptions.matchStaticPointsOnly = false;
			metricMap.m_pointsMaps[0]->insertionOptions.also_interpolate = false;
			metricMap.m_pointsMaps[0]->insertionOptions.minDistBetweenLaserPoints = 0.05f;
		}

		// Inserta el scan en el mapa de puntos y en el grid.
		CPose3D		estimatedPose3D(estimatedPose);
		in_SF.insertObservationsInto( &metricMap, &estimatedPose3D);

		// Add to the vector of "poses"-"SFs" pairs:
		CPosePDFGaussian	posePDF(estimatedPose);
		CPose3DPDFPtr  pose3D = CPose3DPDFPtr( CPose3DPDF::createFrom2D( posePDF ) );

		SF_Poses_seq.insert( pose3D, CSensoryFramePtr( new CSensoryFrame(in_SF)) );

		printf("[performSLAM] Map updated OK!! In %.03fms\n",tictac.Tac()*1000.0f );
	}

	estRobotPath.push_back(estimatedPose);

	ret = icpReturn.goodness;

	leaveCriticalSection();

	MRPT_END_WITH_CLEAN_UP( \
		leaveCriticalSection(); \
		);
}

/*---------------------------------------------------------------
						setCurrentMapFile
  ---------------------------------------------------------------*/
void  CMetricMapBuilderICP::setCurrentMapFile( const char *mapFile )
{
	// Save current map to current file:
	if (currentMapFile.size())
		saveCurrentMapToFile( currentMapFile.c_str() );

	// Sets new current map file:
	currentMapFile = mapFile;

	// Load map from file or create an empty one:
	if (currentMapFile.size())
		loadCurrentMapFromFile( mapFile );
}


/*---------------------------------------------------------------
						getCurrentPoseEstimation
  ---------------------------------------------------------------*/
CPose3DPDFPtr CMetricMapBuilderICP::getCurrentPoseEstimation() const
{
	CPose3DPDFGaussian	*pdf_gauss = new CPose3DPDFGaussian();
	pdf_gauss->copyFrom(estPose);

	return CPose3DPDFPtr(pdf_gauss);
}

/*---------------------------------------------------------------
						initialize
  ---------------------------------------------------------------*/
void  CMetricMapBuilderICP::initialize(
				CSensFrameProbSequence		&initialMap,
				CPosePDF					*x0 )
{
	MRPT_START;

	int		i,n;

	// Enter critical section (updating map)
	enterCriticalSection();

	// copy map:
	SF_Poses_seq = initialMap;

	// Parse SFs to the hybrid map:
	metricMap.clear();

	// Set options:
	// ---------------------
	if (metricMap.m_pointsMaps.size())
	{
		metricMap.m_pointsMaps[0]->insertionOptions.fuseWithExisting			= false;
		metricMap.m_pointsMaps[0]->insertionOptions.minDistBetweenLaserPoints = 0.05f;
		metricMap.m_pointsMaps[0]->insertionOptions.disableDeletion			= true;
		metricMap.m_pointsMaps[0]->insertionOptions.isPlanarMap				= true;
		metricMap.m_pointsMaps[0]->insertionOptions.matchStaticPointsOnly		= true;
	}


	// Load estimated pose from given PDF:
	if (x0)
	{
		estPose.copyFrom( *x0 );
	}
	else
	{
		// Initial pose not supplied...
	}


	n = SF_Poses_seq.size();
	for (i=0;i<n;i++)
	{
		CPose3DPDFPtr		posePDF;
		CSensoryFramePtr	SF;

		// Get the SF and its pose:
		SF_Poses_seq.get(i, posePDF,SF);

		CPose3D	 estimatedPose3D;
		posePDF->getMean(estimatedPose3D);

		// Insert observations into the map:
		SF->insertObservationsInto( &metricMap, &estimatedPose3D );
	}

	printf("[CMetricMapBuilderICP::loadCurrentMapFromFile] OK\n");

	// Exit critical zone.
	leaveCriticalSection();

	MRPT_END_WITH_CLEAN_UP( \
		leaveCriticalSection(); \
		);
}

/*---------------------------------------------------------------
						getCurrentMapPoints
  ---------------------------------------------------------------*/
void  CMetricMapBuilderICP::getCurrentMapPoints(
			vector_float		&x,
			vector_float		&y)
{
	// Critical section: We are using our global metric map
	enterCriticalSection();

	ASSERT_( metricMap.m_pointsMaps.size()>0 );
	metricMap.m_pointsMaps[0]->getAllPoints(x,y);

	// Exit critical zone.
	leaveCriticalSection();
}

/*---------------------------------------------------------------
				getCurrentlyBuiltMap
  ---------------------------------------------------------------*/
void  CMetricMapBuilderICP::getCurrentlyBuiltMap(CSensFrameProbSequence &out_map) const
{
	out_map = SF_Poses_seq;

}

/*---------------------------------------------------------------
						getCurrentlyBuiltMetricMap
  ---------------------------------------------------------------*/
CMultiMetricMap*   CMetricMapBuilderICP::getCurrentlyBuiltMetricMap()
{
	return &metricMap;
}


/*---------------------------------------------------------------
			getCurrentlyBuiltMapSize
  ---------------------------------------------------------------*/
unsigned int  CMetricMapBuilderICP::getCurrentlyBuiltMapSize()
{
	return SF_Poses_seq.size();
}

/*---------------------------------------------------------------
				saveCurrentEstimationToImage
  ---------------------------------------------------------------*/
void  CMetricMapBuilderICP::saveCurrentEstimationToImage(const std::string &file, bool formatEMF_BMP )
{
	MRPT_START;

	CImageFloat			imgFl;
	size_t					nPoses = estRobotPath.size();

	ASSERT_( metricMap.m_gridMaps.size()>0 );

	if (!formatEMF_BMP)
		THROW_EXCEPTION("Not implemented yet for BMP!");

	// grid map as bitmap:
	// ----------------------------------
	metricMap.m_gridMaps[0]->getAsImage( imgFl );

	// Draw paths (using vectorial plots!) over the EMF file:
	// -------------------------------------------------
//	float					SCALE = 1000;
	CEnhancedMetaFile		EMF( file, 1000 );

	EMF.drawImage( 0,0, imgFl );

	unsigned int imgHeight = imgFl.getHeight();

	// Path hypothesis:
	// ----------------------------------
	int		x1,x2,y1,y2;

	// First point: (0,0)
	x2 = metricMap.m_gridMaps[0]->x2idx( 0.0f );
	y2 = metricMap.m_gridMaps[0]->y2idx( 0.0f );

	// Draw path in the bitmap:
	for (size_t j=0;j<nPoses;j++)
	{
		// For next segment
		x1 = x2;
		y1 = y2;

		// Coordinates -> pixels
		x2 = metricMap.m_gridMaps[0]->x2idx( estRobotPath[j].x() );
		y2 = metricMap.m_gridMaps[0]->y2idx( estRobotPath[j].y() );

		// Draw line:
		EMF.line(
			x1, imgHeight-1-y1,
			x2, imgHeight-1-y2,
			0 );
	}

	MRPT_END;
}
