/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef opengl_CPointCloudColoured_H
#define opengl_CPointCloudColoured_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/utils/CImage.h>
#include <mrpt/utils/stl_extensions.h>

namespace mrpt
{
	namespace slam { class CPointsMap; }

	namespace opengl
	{
		class MRPTDLLIMPEXP CPointCloudColoured;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CPointCloudColoured, CRenderizable )


		/** A cloud of points, each one with an individual colour (R,G,B). The alpha component is shared by all the points and is stored in the base member m_color_A.
		  *  \sa opengl::COpenGLScene, opengl::CPointCloud
		  */
		class MRPTDLLIMPEXP CPointCloudColoured : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CPointCloudColoured )

		public:
			struct TPointColour
			{
				TPointColour( float _x=0,float _y=0,float _z=0,float _R=0,float _G=0,float _B=0 ) :
					x(_x),y(_y),z(_z),R(_R),G(_G),B(_B)
				{ }
				float x,y,z,R,G,B;	// Float is precission enough for rendering
			};

		private:
			typedef std::vector<TPointColour> TListPointColour;

			TListPointColour	m_points;
			float				m_pointSize; //!< By default is 1.0

			/** Constructor
			  */
			CPointCloudColoured( ) :
				m_points(),
				m_pointSize(1)
			{
			}
			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CPointCloudColoured() { }

		public:
			typedef TListPointColour::iterator iterator;
			typedef TListPointColour::const_iterator const_iterator;

			iterator begin() { return m_points.begin(); }
			const_iterator begin() const { return m_points.begin(); }
			iterator end() { return m_points.end(); }
			const_iterator end() const { return m_points.end(); }

			void push_back(double x,double y,double z, double R, double G, double B)
			{
				m_points.push_back(TPointColour(float(x),float(y),float(z),float(R),float(G),float(B)));
			}

			void push_back(float x,float y,float z, float R, float G, float B)
			{
				m_points.push_back(TPointColour(x,y,z,R,G,B));
			}

			void reserve(size_t N) { m_points.reserve(N); }
			void resize(size_t N) { m_points.resize(N); }

			/** Read or write access to each individual point. */
			TPointColour &operator [](size_t i) { return m_points[i]; }

			size_t size() const { return m_points.size(); }
			void clear() { m_points.clear(); }

			void setPointSize(float pointSize) { m_pointSize = pointSize; }
			float getPointSize() const { return m_pointSize; }


			/** Load the points from a points map, depending on the type of point map passed: for the case of a mrpt::slam::CColouredPointMap the colours of individual points will be also copied.
			  */
			void  loadFromPointsMap( const mrpt::slam::CPointsMap *map);

			/** Render
			  */
			void  render() const;

		};

		MRPTDLLIMPEXP mrpt::utils::CStream& operator>>(mrpt::utils::CStream& in,  CPointCloudColoured::TPointColour &o);
		MRPTDLLIMPEXP mrpt::utils::CStream& operator<<(mrpt::utils::CStream& out, const CPointCloudColoured::TPointColour &o);

	} // end namespace

	namespace utils
	{
		using namespace mrpt::opengl;

		// Specialization must occur in the same namespace
		MRPT_DECLARE_TTYPENAME(CPointCloudColoured::TPointColour)
	}

} // End of namespace


#endif
