/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  mrpt_math_distributions_H
#define  mrpt_math_distributions_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>

/*---------------------------------------------------------------
		Namespace
  ---------------------------------------------------------------*/
namespace mrpt
{
	namespace math
	{
		/** @name Statistics functions
		@{ */

		/** Evaluates the univariate normal (Gaussian) distribution at a given point "x".
		  */
		double MRPTDLLIMPEXP  normalPDF(double x, double mu, double std);

		/** Evaluates the multivariate normal (Gaussian) distribution at a given point "x" ("x" and "mu" can be 1xN or Nx1 matrixes).
		  */
		template <typename  T>
		T  normalPDF(
			const CMatrixTemplateNumeric<T>		&x,
			const CMatrixTemplateNumeric<T>		&mu,
			const CMatrixTemplateNumeric<T>		&cov)
		{
			MRPT_START;

			CMatrixTemplateNumeric<T>	S = cov.inv();
			size_t						d = cov.getRowCount();		// Dimension:
			CMatrixTemplateNumeric<T>	X, MU, X_MU;

			if (x.getRowCount()==1)
					X = ~x;
			else	X = x;

			if (mu.getRowCount()==1)
					MU = ~mu;
			else	MU = mu;

			X_MU = X - MU;

			return ::exp( static_cast<T>(-0.5)  * ( (~X_MU)*S*X_MU )(0,0) ) / (::pow(static_cast<T>(M_2PI),static_cast<T>(d)) * ::sqrt(cov.det()));

			MRPT_END;
		}

		/** Evaluates the multivariate normal (Gaussian) distribution at a given point "x" ("x" and "mu" can be 1xN or Nx1 matrixes).
		  */
		template <typename T,size_t N>
		T  normalPDF(
			const CMatrixFixedNumeric<T,N,1>	&x,
			const CMatrixFixedNumeric<T,N,1>	&mu,
			const CMatrixFixedNumeric<T,N,N>	&cov)
		{
			MRPT_START;

			CMatrixFixedNumeric<T,N,N>  S;
			cov.inv(S);

			CMatrixFixedNumeric<T,N,1> X_MU = x;
			X_MU-= mu;

			return ::exp( static_cast<T>(-0.5)  * ( (~X_MU)*S*X_MU )(0,0) ) / (::pow(static_cast<T>(M_2PI),static_cast<T>(N)) * ::sqrt(cov.det()));

			MRPT_END;
		}

		/** The complementary error function of a Normal distribution
		  */
		double MRPTDLLIMPEXP  erfc(double x);

		/** The error function of a Normal distribution
		  */
		double MRPTDLLIMPEXP   erf(double x);

		/** Evaluates the Gaussian distribution quantile for the probability value p=[0,1].
		  *  The employed approximation is that from Peter J. Acklam (pjacklam@online.no),
		  *  freely available in http://home.online.no/~pjacklam.
		  */
		double MRPTDLLIMPEXP  normalQuantile(double p);

		/** Evaluates the Gaussian cumulative density function.
		  *  The employed approximation is that from W. J. Cody
		  *  freely available in http://www.netlib.org/specfun/erf
		  */
		double  MRPTDLLIMPEXP normalCDF(double p);

		/** The "quantile" of the Chi-Square distribution, for dimension "dim" and probability 0<P<1
		  * An aproximation from the Wilson-Hilferty transformation is used.
		  */
		double  MRPTDLLIMPEXP chi2inv(double P, unsigned int dim=1);

		/** @} */

	} // End of MATH namespace

} // End of namespace


#endif
