/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/math/utils.h>
#include <mrpt/gui/CDisplayWindow.h>
#include <mrpt/gui/CDisplayWindowPlots.h>
#include <mrpt/utils/CConfigFile.h>
#include <mrpt/utils/CEnhancedMetaFile.h>
#include <mrpt/slam/CGridMapAligner.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/slam/COccupancyGridMap2D.h>
#include <mrpt/slam/CICP.h>
#include <mrpt/poses/CPosePDFParticles.h>
#include <mrpt/poses/CPosePDFSOG.h>
#include <mrpt/slam/CLandmarksMap.h>
#include <mrpt/scan_matching/scan_matching.h>
#include <mrpt/poses/CPosePDFGrid.h>

#include <mrpt/utils/CTicTac.h>
#include <mrpt/math/CVectorTemplate.h>
#include <mrpt/math_mrpt.h>

using namespace mrpt::math;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::gui;
using namespace std;

/*---------------------------------------------------------------
The method for aligning a pair of 2D points map.
*   The meaning of some parameters are implementation dependant,
*    so look for derived classes for instructions.
*  The target is to find a PDF for the pose displacement between
*   maps, <b>thus the pose of m2 relative to m1</b>. This pose
*   is returned as a PDF rather than a single value.
*
* \param m1			[IN] The first map
* \param m2			[IN] The second map. The pose of this map respect to m1 is to be estimated.
* \param grossEst		[IN] IGNORED
* \param runningTime	[OUT] A pointer to a container for obtaining the algorithm running time in seconds, or NULL if you don't need it.
* \param info			[OUT] See derived classes for details, or NULL if it isn't needed.
*
* \sa CPointsMapAlignmentAlgorithm
  ---------------------------------------------------------------*/
CPosePDFPtr CGridMapAligner::AlignPDF(
    const CMetricMap		*mm1,
    const CMetricMap		*mm2,
    const CPosePDFGaussian	&initialEstimationPDF,
    float					*runningTime,
    void					*info )
{
	MRPT_TRY_START;

	switch( options.methodSelection )
	{
	case CGridMapAligner::amCorrelation:
		return AlignPDF_correlation(mm1,mm2,initialEstimationPDF,runningTime,info);

	case CGridMapAligner::amRobustMatch:
		return AlignPDF_robustMatch(mm1,mm2,initialEstimationPDF,runningTime,info);

	default:
		THROW_EXCEPTION("Wrong value found in 'options.methodSelection'!!");
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					AlignPDF_robustMatch
---------------------------------------------------------------*/
CPosePDFPtr CGridMapAligner::AlignPDF_robustMatch(
    const CMetricMap		*mm1,
    const CMetricMap		*mm2,
    const CPosePDFGaussian	&initialEstimationPDF,
    float					*runningTime,
    void					*info )
{
	MRPT_TRY_START;

	TReturnInfo								outInfo;
	CMetricMap::TMatchingPairList			&correspondences = outInfo.correspondences; // Use directly this placeholder to save 1 variable & 1 copy.
	CMetricMap::TMatchingPairList			largestConsensusCorrs;

	CMetricMap::TMatchingPairList::iterator	corrIt;
	CTicTac									*tictac=NULL;

	CPose2D									grossEst = initialEstimationPDF.mean;

	CLandmarksMapPtr						lm1( new CLandmarksMap() );
	CLandmarksMapPtr						lm2( new CLandmarksMap() );

	unsigned int							idx1,idx2;
	unsigned int							nLM1,nLM2;
	std::vector<size_t>						idxs1,idxs2;
	std::vector<size_t>::iterator			it1,it2;

	// Asserts:
	// -----------------
	const CMultiMetricMap			*multimap1 = NULL;
	const CMultiMetricMap			*multimap2 = NULL;
	const COccupancyGridMap2D		*m1 = NULL;
	const COccupancyGridMap2D		*m2 = NULL;

	if (IS_CLASS(mm1, CMultiMetricMap) && IS_CLASS(mm2, CMultiMetricMap) )
	{
		multimap1 = static_cast<const CMultiMetricMap*>(mm1);
		multimap2 = static_cast<const CMultiMetricMap*>(mm2);

		ASSERT_(multimap1->m_gridMaps.size() && multimap1->m_gridMaps[0].present());
		ASSERT_(multimap2->m_gridMaps.size() && multimap2->m_gridMaps[0].present());

		m1 = multimap1->m_gridMaps[0].pointer();
		m2 = multimap2->m_gridMaps[0].pointer();
	}
	else if ( IS_CLASS(mm1, COccupancyGridMap2D) && IS_CLASS(mm2, COccupancyGridMap2D) )
	{
		m1 = static_cast<const COccupancyGridMap2D*>(mm1);
		m2 = static_cast<const COccupancyGridMap2D*>(mm2);
	}
	else THROW_EXCEPTION("Metric maps must be of classes COccupancyGridMap2D or CMultiMetricMap")

	ASSERT_( m1->getResolution() == m2->getResolution() );

	// The algorithm output auxiliar info:
	// -------------------------------------------------
	outInfo.goodness		= 1.0f;

	outInfo.landmarks_map1 = lm1;
	outInfo.landmarks_map2 = lm2;


	// The PDF to estimate:
	// ------------------------------------------------------
	CPosePDFSOGPtr pdf_SOG = CPosePDFSOGPtr( new CPosePDFSOG() );

	// Extract features from grid-maps:
	// ------------------------------------------------------
	m1->extractPanoramicFeaturesCached(
		*lm1,
		max(50,round((m1->getSizeX()*m1->getSizeY())*square(m1->getResolution())* options.featsPerSquareMeter)),
		options.feats_DirectionSectors,
		options.feats_RangeSectors,
		options.feats_startDist,
		options.feats_endDist );

	m2->extractPanoramicFeaturesCached(
		*lm2,
		max(50,round((m2->getSizeX()*m2->getSizeY())*square(m2->getResolution())* options.featsPerSquareMeter)),
		options.feats_DirectionSectors,
		options.feats_RangeSectors,
		options.feats_startDist,
		options.feats_endDist );

	if (runningTime)
	{
		tictac = new CTicTac();
		tictac->Tic();
	}

	nLM1 = lm1->size();
	nLM2 = lm2->size();

	//  At least two landmarks at each map!
	// ------------------------------------------------------
	if (nLM1<2 || nLM2<2)
	{
		outInfo.goodness = 0;
	}
	else
	{
//#define METHOD_FFT
//#define DEBUG_SHOW_CORRELATIONS


		// Compute correlation between landmarks:
		// ---------------------------------------------
		CMatrix			CORR(lm1->size(),lm2->size()),auxCorr;
		CMatrix			descriptor1,descriptor2;
		CMRPTImage		im1(1,1,1),im2(1,1,1);		// Grayscale
		vector_float	corr;
		unsigned int		corrsCount = 0;
		std::vector<bool>	hasCorr(nLM1,false);

		bool			descriptorsAreSIFT = lm1->landmarks.get(0)->descriptor1.size()>2;

		if (!descriptorsAreSIFT)
		{
#ifdef METHOD_FFT

	#ifdef DEBUG_SHOW_CORRELATIONS
			CDisplayWindowPlots		auxWin("Individual corr. matrix");
			CDisplayWindow			win1("desc.#1"),win2("desc.#2");
			//auxWin->setPos(10,10);
			//win1.setPos(100,10);
			//win2.setPos(200,10);
	#endif
			for (idx1=0;idx1<nLM1;idx1++)
			{
				lm1->landmarks.get(idx1)->getDescriptorAsMatrix( descriptor1 );
				//im1 = descriptor1;

				for (idx2=0;idx2<nLM2;idx2++)
				{
					lm2->landmarks.get(idx2)->getDescriptorAsMatrix( descriptor2 );
					//im2 = descriptor2;

					/*im1.cross_correlation_FFT(
						im2,
						auxCorr,
						-1,-1,-1,-1,
						0,0		// bias
						);*/
					mrpt::math::cross_correlation_FFT(descriptor1,descriptor2,auxCorr);

					// We will focus on the FIRST row!!!:
					auxCorr.extractRow(0,corr);

					// A goodness measure:
					double	mean,std;
					math::meanAndStd(corr,mean,std);
					float	peak = math::maximum(corr);
					float	goodness = (float)((peak - mean)/std);
					if ( mrpt::system::isNaN(goodness) )
					{
						//printf("[CGridMapAligner::AlignPDF_robustMatch] Warning! Goodness found to be NAN!! Ignoring and assuming it is 0\n");
						goodness = 0;
					}
					CORR(idx1,idx2) = goodness;

	#ifdef DEBUG_SHOW_CORRELATIONS
					{
						printf("[%u - %u] -> goodned: %f peak: %f mean:%f   std:%f \n",idx1,idx2,goodness,peak ,mean,std);
						//auxCorr.adjustRange();
						//CMRPTImageFloat	imgCorr(auxCorr);
						//auxWin->showImage( imgCorr );
						auxWin->plot(corr,"b.","the_points");

						vector_float xs(2);
						xs(0)=0;
						xs(1)=corr.size()+1;
						auxWin->plot(xs,ys,"r","the_mean");

						auxWin->axis_fit();

						win1.showImage(im1);
						win2.showImage(im2);
						auxWin->waitForKey();
					}
	#endif
				} // end for idx2
			} // end for idx1

			//  Establish correspondences between features:
			CORR.findElementsPassingMahalanobisThreshold( 2.0 /*options.thresholdStdTimes*/, idxs1, idxs2 );
#else
			// Non-FFT method:
			// -------------------

			CDisplayWindowPlotsPtr	auxWin;
			if (options.debug_show_corrs)
				auxWin = CDisplayWindowPlotsPtr( new CDisplayWindowPlots("Individual corr.") );


	#endif

			for (idx1=0;idx1<nLM1;idx1++)
			{
				//CMatrixFloat	corr_row(1,nLM2);
				vector_float  	corrs_indiv;
				corrs_indiv.reserve(nLM2);

				for (idx2=0;idx2<nLM2;idx2++)
				{
					float	meanDist, meanAng;
					this->landmarksMatchingCorrelation(
						lm1->landmarks.get(idx1),
						lm2->landmarks.get(idx2),
						meanDist,
						meanAng );

					//corr_row(0,idx2) = meanDist;
					//CORR(idx1,idx2) = meanDist;
					corrs_indiv.push_back(meanDist);

				} // end for idx2



				double corr_mean,corr_std;
				mrpt::math::meanAndStd(corrs_indiv,corr_mean,corr_std);

				const double thres = options.threshold;

				for (idx2=0;idx2<nLM2;idx2++)
				{
					if ( corrs_indiv[idx2] < thres )
					{
						idxs1.push_back(idx1);
						idxs2.push_back(idx2);
					}
					outInfo.correspondences_dists_maha.push_back( TReturnInfo::TPairPlusDistance(idx1,idx2, corrs_indiv[idx2] ) );
				}

				/*
				//CORR.findElementsPassingMahalanobisThreshold( options.thresholdStdTimes, idxs1, idxs2, true );
				vector_size_t	li1,li2;
				corr_row.findElementsPassingMahalanobisThreshold( options.thresholdStdTimes, li1, li2, true );
				for (size_t q=0;q<li1.size();q++)
				{
					idxs1.push_back(idx1);
					idxs2.push_back(li2[q]);
					//printf("%i -> %i\n", (int)idx1, (int)li2[q]);
				}*/


				if (options.debug_show_corrs)
				{
					auxWin->setWindowTitle(format("Corr: %i - rest", int(idx1)));

					auxWin->plot(corrs_indiv,".3","the_corr");


					vector_float xs(2), ys(2);
					xs[0]=0; xs[1]=corrs_indiv.size()+1;

					ys[0]=ys[1] = corr_mean;
					auxWin->plot(xs,ys,"r","the_mean");

					ys[0]=ys[1] = options.threshold;
					auxWin->plot(xs,ys,"g","the_thres");



					if (idx1==0) auxWin->axis_fit();
					auxWin->waitForKey();
				}

			} // end for idx1

			//  Establish correspondences between features:
			//CORR.findElementsPassingMahalanobisThreshold( options.thresholdStdTimes, idxs1, idxs2, true );

			//cout << "Idxs: " << idxs1.size() << " " << idxs2.size() << endl;

			// Save an image with each feature and its matches:
			if (options.save_feat_coors)
			{
				mrpt::system::deleteFilesInDirectory("grid_feats");
				mrpt::system::createDirectory("grid_feats");

				std::map<size_t, std::set<size_t> > corrs_by_idx;
				for (size_t l=0;l<idxs1.size();l++)
					corrs_by_idx[idxs1[l]].insert( idxs2[l] );

				for (std::map<size_t, std::set<size_t> >::iterator it=corrs_by_idx.begin();it!=corrs_by_idx.end();it++)
				{
					const string fil = format("grid_feats/_FEAT_MATCH_%03i.PNG", (int)it->first );

					lm1->landmarks.get(it->first)->getDescriptorAsMatrix( descriptor1 );
					im1 = descriptor1;

					const size_t FEAT_W = im1.getWidth();
					const size_t FEAT_H = im1.getHeight();
					const size_t nF = it->second.size();

					CMRPTImage	img_compose( FEAT_W*2 + 15, 10 + (5+FEAT_H) * (1+nF) );
					img_compose.filledRectangle(0,0,img_compose.getWidth()-1,img_compose.getHeight()-1,0);

					img_compose.drawImage(5,5, im1 );

					size_t j;
					std::set<size_t>::iterator it_j;

					for (j=0,it_j=it->second.begin();j<nF;j++,it_j++)
					{
						lm1->landmarks.get( *it_j )->getDescriptorAsMatrix( descriptor2 );
						im2 = descriptor2;
						img_compose.drawImage(10+FEAT_W,5 + j*(FEAT_H+5), im2 );
					}

					img_compose.saveToFile( fil );
				} // end for map

			}


#if 0
			{
				CDisplayWindow	win1;
				CORR.adjustRange();
				CMRPTImageFloat	imgCorr(CORR);
				imgCorr.saveToFile("__debug_corr_mat.png");
				win1.showImage( imgCorr );
				win1.waitForKey();
			}
#endif
		}
		else
		{
			// Compute distance between SIFT descriptors:
			// ------------------------------------------------
			for (idx1=0;idx1<nLM1;idx1++)
			{
				std::vector<unsigned char>		*sift1 = &lm1->landmarks.get(idx1)->descriptor1;

				vector_float	thisCorr(nLM2,0);

				for (idx2=0;idx2<nLM2;idx2++)
				{
					std::vector<unsigned char>		*sift2 = &lm2->landmarks.get(idx2)->descriptor1;

					// A goodness measure:
					std::vector<unsigned char>		diff = *sift1 - *sift2;

					float	goodness = 0;
					for (std::vector<unsigned char>::iterator it1 = diff.begin();it1!=diff.end();it1++)
						goodness+= square( *it1 );

					thisCorr[idx2] = sqrt(goodness);
				} // end for idx2

				// Establish corrs:
				//system::vectorToTextFile( thisCorr, "__thisCorr.txt" ); system("pause");

				double	d_mean, d_std;
				math::meanAndStd( thisCorr, d_mean,d_std );
				for (idx2=0;idx2<nLM2;idx2++)
				{
					if ( ((d_mean-thisCorr[idx2])/d_std) < options.threshold )
					//if (thisCorr[idx2]<50)
					{
						//printf("[%.02f,%.02f]->[%.02f,%.02f] d=%f\n",lm1.landmarks.get(idx1)->pose_mean_x,lm1.landmarks.get(idx1)->pose_mean_y,lm2.landmarks.get(idx2)->pose_mean_x,lm2.landmarks.get(idx2)->pose_mean_y,thisCorr[idx2]);
						//system("pause");

						idxs1.push_back( idx1 );
						idxs2.push_back( idx2 );
					}
				}

			} // end for idx1
		}

//		CORR.saveToTextFile("__CORR.txt");
//		system("pause");

		correspondences.clear();
		for (it1=idxs1.begin(),it2=idxs2.begin();it1!=idxs1.end();it1++,it2++)
		{
			CMetricMap::TMatchingPair	mp;
			mp.this_idx = *it1;
			mp.this_x   = lm1->landmarks.get(*it1)->pose_mean_x;
			mp.this_y   = lm1->landmarks.get(*it1)->pose_mean_y;
			mp.this_z   = lm1->landmarks.get(*it1)->pose_mean_z;

			mp.other_idx = *it2;
			mp.other_x	 = lm2->landmarks.get(*it2)->pose_mean_x;
			mp.other_y	 = lm2->landmarks.get(*it2)->pose_mean_y;
			mp.other_z	 = lm2->landmarks.get(*it2)->pose_mean_z;
			correspondences.push_back( mp );

			if (!hasCorr[*it1])
			{
				hasCorr[*it1]= true;
				corrsCount++;
			}
		} // end for corres.

		outInfo.goodness = (2.0f * corrsCount) / (nLM1+nLM2);

		// Save all corrs:

		// Compute the estimation using ALL the correspondences (NO ROBUST):
		// ----------------------------------------------------------------------
		if (! scan_matching::leastSquareErrorRigidTransformation(
				correspondences,
				outInfo.noRobustEstimation ) )
		{
			// There's no way to match the maps! e.g. no correspondences
			outInfo.goodness = 0;
			pdf_SOG->m_modes.clear();
			outInfo.sog1 = pdf_SOG;
			outInfo.sog2 = pdf_SOG;
			outInfo.sog3 = pdf_SOG;
		}
		else
		{
			printf_debug("[CGridMapAligner] Debug: Overall estimation(%u corrs, total: %u): (%.03f,%.03f,%.03fdeg)\n",
				corrsCount,(unsigned)correspondences.size(),
				outInfo.noRobustEstimation.x,
				outInfo.noRobustEstimation.y,
				RAD2DEG(outInfo.noRobustEstimation.phi));

			// RANSAC over the found correspondences:
			// -------------------------------------------------
			unsigned int ransac_nSimulations = max(4000u,
				(unsigned int)(20*correspondences.size()));
				//(unsigned int)(0.1*square(correspondences.size())) );
			printf_debug("[CGridMapAligner] Debug: Running RANSAC with %u iterations\n", ransac_nSimulations);
			scan_matching::robustRigidTransformation(
				correspondences,
				*pdf_SOG,
				options.ransac_SOG_sigma_m,
				options.ransac_minSetSize,
				options.ransac_maxSetSize,
				options.ransac_mahalanobisDistanceThreshold,
				ransac_nSimulations,
				&largestConsensusCorrs
				);

			ASSERT_(pdf_SOG);

			// Simplify the SOG by merging close modes:
			// -------------------------------------------------
			size_t nB = pdf_SOG->m_modes.size();
			outInfo.sog1 = pdf_SOG; outInfo.sog1.make_unique();

			CTicTac	merge_clock;
			pdf_SOG->mergeModes( options.maxKLd_for_merge );
			const double merge_clock_tim = merge_clock.Tac();

			outInfo.sog2 = pdf_SOG; outInfo.sog2.make_unique();

			size_t nA = pdf_SOG->m_modes.size();

			printf_debug("[CGridMapAligner] Debug: %u SOG modes merged to %u in %.03fsec\n", (unsigned int)nB, (unsigned int)nA, merge_clock_tim );


			if (options.debug_save_map_pairs)
			{
				static unsigned int NN = 0;
				static const COccupancyGridMap2D	*lastM1 = NULL;

				if (lastM1!=m1)
				{
					lastM1=m1;
					NN=0;
				}

				printf("   Largest consensus: %u\n", static_cast<unsigned>(largestConsensusCorrs.size()) );

				CEnhancedMetaFile::LINUX_IMG_WIDTH = m1->getSizeX() + m2->getSizeX() + 50;
				CEnhancedMetaFile::LINUX_IMG_HEIGHT = max(m1->getSizeY(), m2->getSizeY()) + 50;
				COccupancyGridMap2D::saveAsEMFTwoMapsWithCorrespondences(
					format( "__debug_corrsGrid_%05u.emf",++NN),
					m1,
					m2,
					largestConsensusCorrs );

				COccupancyGridMap2D::saveAsBitmapTwoMapsWithCorrespondences(
					format( "__debug_corrsGrid_%05u.png",++NN),
					m1,
					m2,
					largestConsensusCorrs );
			}

			// --------------------------------------------------------------------
			// At this point:
			//   - "pdf_SOG": has the resulting PDF with the SOG
			//   - "largestConsensusCorrs": The 'best' set of correspondences
			//
			//  Now: If we had a multi-metric map, use the points map to improve
			//        the estimation with ICP.
			// --------------------------------------------------------------------
			if (multimap1 && multimap2 &&
				!multimap1->m_pointsMaps.empty() && !multimap2->m_pointsMaps.empty() &&
				multimap1->m_pointsMaps[0].present() && multimap2->m_pointsMaps[0].present() )
			{
				CSimplePointsMapPtr		pnts1 = multimap1->m_pointsMaps[0];
				CSimplePointsMapPtr		pnts2 = multimap2->m_pointsMaps[0];

				CICP				icp;
				CICP::TReturnInfo	icpInfo;

				icp.options.maxIterations			= 20;
				icp.options.smallestThresholdDist	= 0.20f;
				icp.options.thresholdDist			= 1.0f;

				//cout << "Points: " << pnts1->size() << " " << pnts2->size() << endl;

				// Invoke ICP once for each mode in the SOG:
				size_t cnt = 0;
				outInfo.icp_goodness_all_sog_modes.clear();
				for (std::deque<CPosePDFSOG::TGaussianMode>::iterator i=pdf_SOG->m_modes.begin();i!=pdf_SOG->m_modes.end(); cnt++)
				{
					CPosePDFPtr icp_est = icp.Align(pnts1.pointer(),pnts2.pointer(),i->mean,NULL,&icpInfo);
					cout << "ICP " << cnt << " pos: " << icp_est->getEstimatedPose() << " Goodness: " << 100*icpInfo.goodness << endl;

					outInfo.icp_goodness_all_sog_modes.push_back(icpInfo.goodness);

					// Discard?
					if (icpInfo.goodness > options.min_ICP_goodness)
					{
						i->mean = icp_est->getEstimatedPose();

						i->cov.unit();
						i->cov(0,0) = i->cov(1,1) = square(0.05);
						i->cov(2,2) = square(DEG2RAD(1));

						i++;
					}
					else
					{
						// Delete:
						i = pdf_SOG->m_modes.erase(i);
					}

				} // end for i

				// Merge:
				outInfo.sog3 = pdf_SOG; outInfo.sog3.make_unique();

				pdf_SOG->mergeModes( options.maxKLd_for_merge, false );
				size_t nAA = pdf_SOG->m_modes.size();
				printf_debug("[CGridMapAligner] Debug: %u SOG modes merged after ICP\n", (unsigned int)nAA );

			} // end multimapX
		}

	} // end of: yes, there are landmarks in the grid maps!

	// Copy the output info if requested:
	// -------------------------------------------------
	if (info)
	{
		TReturnInfo* info_ = static_cast<TReturnInfo*>(info);
		ASSERT_( info_->cbSize == sizeof(TReturnInfo) );
		*info_ = outInfo;
	}

	if (runningTime)
	{
		*runningTime = tictac->Tac();
		delete tictac;
	}

	return pdf_SOG;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					landmarksMatchingCorrelation
  ---------------------------------------------------------------*/
void  CGridMapAligner::landmarksMatchingCorrelation(
	mrpt::slam::CLandmark		*lm1,
	mrpt::slam::CLandmark		*lm2,
	float			&minDistance,
	float			&minDistAngle)
{
	MRPT_TRY_START;

	CMatrix		desc1,desc2;

	// Asserts:
	ASSERT_(lm1->type == CLandmark::glPanoramicDescriptor);
	ASSERT_(lm2->type == CLandmark::glPanoramicDescriptor);

	// Get the descriptors:
	lm1->getDescriptorAsMatrix( desc1 );
	lm2->getDescriptorAsMatrix( desc2 );

	ASSERT_(desc1.getColCount() == desc2.getColCount());
	ASSERT_(desc1.getRowCount() == desc2.getRowCount());

	// Find the smallest distance:
	unsigned int	delta,i,j,ii,height = desc1.getRowCount(), width = desc1.getColCount();
	float			dist, minDist=0;
	int				minDistIdx = -1;


#define LM_CORR_METHOD_EUCLID
//#define LM_CORR_METHOD_MANHATTAN

	for (delta=0;delta<width;delta++)
	{
		// Compute the mean distance between desc1[t] and desc2[t-delta]:
		dist = 0;
		for (i=0;i<width;i++)
		{
			ii = (i + delta) % width;	// Shifted index
			for (j=0;j<height;j++)
			{
#ifdef LM_CORR_METHOD_EUCLID
				dist+= square( desc1.get_unsafe(j,i) - desc2.get_unsafe(j,ii) );
#elif defined(LM_CORR_METHOD_MANHATTAN)
				dist+= abs( desc1.get_unsafe(j,i) - desc2.get_unsafe(j,ii) );
#else
#error A LM_CORR_METHOD_XXX method must be selected!
#endif
			}
		}

		// Average:
		dist /= static_cast<float>(width*height);

		if (minDistIdx==-1 || dist<minDist)
		{
			minDistIdx = delta;
			minDist = dist;
		}
	} // end for delta

#ifdef LM_CORR_METHOD_EUCLID
	minDist = sqrt(minDist);
#endif

	// Output:
	minDistance = minDist;
	minDistAngle = minDistIdx * M_2PI / static_cast<float>(width) ;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					AlignPDF_correlation
---------------------------------------------------------------*/
CPosePDFPtr CGridMapAligner::AlignPDF_correlation(
    const CMetricMap		*mm1,
    const CMetricMap		*mm2,
    const CPosePDFGaussian	&initialEstimationPDF,
    float					*runningTime,
    void					*info )
{
	MRPT_UNUSED_PARAM(initialEstimationPDF);	MRPT_UNUSED_PARAM(info);

	MRPT_TRY_START;

//#define	CORRELATION_SHOW_DEBUG

	float						mapsResolution;
	CTicTac						*tictac = NULL;

	// Asserts:
	// -----------------
	ASSERT_(mm1->GetRuntimeClass() == CLASS_ID(COccupancyGridMap2D));
	ASSERT_(mm2->GetRuntimeClass() == CLASS_ID(COccupancyGridMap2D));
	const COccupancyGridMap2D	*m1 = static_cast<const COccupancyGridMap2D*>( mm1 );
	const COccupancyGridMap2D	*m2 = static_cast<const COccupancyGridMap2D*>( mm2 );

	ASSERT_( m1->getResolution() == m2->getResolution() );
	mapsResolution = m1->getResolution();

	if (runningTime)
	{
		tictac = new CTicTac();
		tictac->Tic();
	}

	// The PDF to estimate:
	// ------------------------------------------------------
	CPosePDFGaussianPtr PDF = CPosePDFGaussianPtr( new CPosePDFGaussian() );

	// Determine the extension to compute the correlation into:
	// ----------------------------------------------------------
	float		phiResolution = DEG2RAD( 0.2f );
	float		phiMin = -M_PIf+0.5f*phiResolution;
	float		phiMax = M_PIf;

	phiMax = DEG2RAD(55);
	phiMin = DEG2RAD(45);

	// Compute the difference between maps for each (u,v) pair!
	// --------------------------------------------------------------
	float					phi;
	float					pivotPt_x = 0.5f*(m1->getXMax()+m1->getXMin());
	float					pivotPt_y = 0.5f*(m1->getYMax()+m1->getYMin());
	COccupancyGridMap2D		map2_mod;
	CMRPTImage				map1_img,map2_img;
	float					currentMaxCorr = -1e6f;

	m1->getAsImage(map1_img);

	map2_mod.setSize( m1->getXMin(),m1->getXMax(),m1->getYMin(),m1->getYMax(),m1->getResolution() );
	size_t					map2_lx = map2_mod.getSizeX();
	size_t					map2_ly = map2_mod.getSizeY();
	CMatrix					outCrossCorr,bestCrossCorr;
	float					map2width_2 = 0.5f*(map2_mod.getXMax()-map2_mod.getXMin());

#ifdef CORRELATION_SHOW_DEBUG
	CDisplayWindow			*win  = new CDisplayWindow("corr");
	CDisplayWindow			*win2 = new CDisplayWindow("map2_mod");
#endif

	// --------------------------------------------------------
	// Use FFT-based correlation for each orientation:
	// --------------------------------------------------------
	for (phi=phiMin;phi<phiMax;phi+=phiResolution)
	{
		// Create the displaced map2 grid, for the size of map1:
		// --------------------------------------------------------
		CPose2D		v2(
			pivotPt_x - cos(phi)*map2width_2,
			pivotPt_y - sin(phi)*map2width_2,
			phi);		// Rotation point: the centre of img1:
		CPoint2D	v1,v3;
		v2 = CPose2D(0,0,0) - v2;	// Inverse

		for (unsigned int cy2=0;cy2<map2_ly;cy2++)
		{
			COccupancyGridMap2D::cellType	*row = map2_mod.getRow(cy2);
			for (unsigned int cx2=0;cx2<map2_lx;cx2++)
			{
				v3 = v2 + CPoint2D( map2_mod.idx2x(cx2), map2_mod.idx2y(cy2) );
				*row++ = m2->p2l( m2->getPos( v3.x,v3.y ) );
			}
		}

		map2_mod.getAsImage( map2_img );
//		map2_img.saveToBMP("__debug_map2mod.bmp");

		// Compute the correlation:
		map1_img.cross_correlation_FFT(
			map2_img,
			outCrossCorr,
			-1,-1,-1,-1,
			127,			// Bias to be substracted
			127				// Bias to be substracted
			);

		float	corrPeak = outCrossCorr.maximum();
		printf("phi = %fdeg \tmax corr=%f\n", RAD2DEG(phi), corrPeak );

		// Keep the maximum:
		if (corrPeak>currentMaxCorr)
		{
			currentMaxCorr = corrPeak;
			bestCrossCorr = outCrossCorr;
			PDF->mean.phi = phi;
		}

#ifdef CORRELATION_SHOW_DEBUG
			outCrossCorr.adjustRange(0,1);
			CMRPTImageFloat	aux( outCrossCorr );
			win->showImage(aux);
			win2->showImage(map2_img);
			system::sleep(5);
#endif

	} // end for phi

	if (runningTime)
	{
		*runningTime = tictac->Tac();
		delete tictac;
	}

	bestCrossCorr.adjustRange(0,1);
	CMRPTImageFloat	aux( bestCrossCorr );
	aux.saveToFile("_debug_best_corr.bmp");

#ifdef CORRELATION_SHOW_DEBUG
	delete win;
	delete win2;
#endif

	// Transform the best corr matrix peak into coordinates:
	size_t		uMax,vMax;
	bestCrossCorr.find_index_max_value(uMax,vMax,currentMaxCorr);

	PDF->mean.x = m1->idx2x( uMax );
	PDF->mean.y = m1->idx2y( vMax );

	return PDF;

	// Done!
	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					TConfigParams
  ---------------------------------------------------------------*/
CGridMapAligner::TConfigParams::TConfigParams() :
	methodSelection			( CGridMapAligner::amRobustMatch ),

	ransac_minSetSize		( 3 ),
	ransac_maxSetSize		( 20 ),
	ransac_SOG_sigma_m		( 0.05f ),
	ransac_mahalanobisDistanceThreshold	( 3.0f ),

	featsPerSquareMeter		( 0.05f ),
	feats_DirectionSectors	( 32 ),
	feats_RangeSectors		( 8 ),
	feats_startDist			( 0.10f ),
	feats_endDist			( 2.50f ),
	threshold				( 0.13f ),

	min_ICP_goodness		( 0.25f ),

	maxKLd_for_merge		( 0.7 ),

	save_feat_coors			( false ),
	debug_show_corrs		( false ),
	debug_save_map_pairs	( false )
{
}

/*---------------------------------------------------------------
					dumpToTextStream
  ---------------------------------------------------------------*/
void  CGridMapAligner::TConfigParams::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [CGridMapAligner::TConfigParams] ------------ \n\n");

	LOADABLEOPTS_DUMP_VAR(methodSelection, int)
	LOADABLEOPTS_DUMP_VAR(featsPerSquareMeter, float)
	LOADABLEOPTS_DUMP_VAR(feats_DirectionSectors, int)
	LOADABLEOPTS_DUMP_VAR(feats_RangeSectors, int)
	LOADABLEOPTS_DUMP_VAR(feats_startDist, float)
	LOADABLEOPTS_DUMP_VAR(feats_endDist, float)
	LOADABLEOPTS_DUMP_VAR(threshold, float)
	LOADABLEOPTS_DUMP_VAR(min_ICP_goodness, float)
	LOADABLEOPTS_DUMP_VAR(maxKLd_for_merge,float)
	LOADABLEOPTS_DUMP_VAR(ransac_minSetSize,int)
	LOADABLEOPTS_DUMP_VAR(ransac_maxSetSize, int)
	LOADABLEOPTS_DUMP_VAR(ransac_mahalanobisDistanceThreshold,float)
	LOADABLEOPTS_DUMP_VAR(ransac_SOG_sigma_m,float)
	LOADABLEOPTS_DUMP_VAR(save_feat_coors,bool)
	LOADABLEOPTS_DUMP_VAR(debug_show_corrs, bool)
	LOADABLEOPTS_DUMP_VAR(debug_save_map_pairs, bool)

	out.printf("\n");
}

/*---------------------------------------------------------------
					loadFromConfigFile
  ---------------------------------------------------------------*/
void  CGridMapAligner::TConfigParams::loadFromConfigFile(
	const mrpt::utils::CConfigFileBase  &iniFile,
	const std::string &section)
{
	MRPT_LOAD_CONFIG_VAR_CAST(methodSelection, int, TAlignerMethod, 		iniFile, "methodSelection");

	featsPerSquareMeter		= iniFile.read_float(section.c_str(),"featsPerSquareMeter",featsPerSquareMeter);
	feats_DirectionSectors	= iniFile.read_int(section.c_str(),"feats_DirectionSectors",feats_DirectionSectors);
	feats_RangeSectors		= iniFile.read_int(section.c_str(),"feats_RangeSectors",feats_RangeSectors);
	feats_startDist			= iniFile.read_float(section.c_str(),"feats_startDist",feats_startDist);
	feats_endDist			= iniFile.read_float(section.c_str(),"feats_endDist",feats_endDist);

	ransac_SOG_sigma_m		= iniFile.read_float(section.c_str(),"ransac_SOG_sigma_m",ransac_SOG_sigma_m);

	threshold		= iniFile.read_float(section.c_str(),"threshold",threshold);

	MRPT_LOAD_CONFIG_VAR(min_ICP_goodness, float,   iniFile, section)

	maxKLd_for_merge        = iniFile.read_double(section.c_str(),"maxKLd_for_merge",maxKLd_for_merge);

	ransac_minSetSize		= iniFile.read_int(section.c_str(),"ransac_minSetSize",ransac_minSetSize);
	ransac_maxSetSize		= iniFile.read_int(section.c_str(),"ransac_maxSetSize",ransac_maxSetSize);

	ransac_mahalanobisDistanceThreshold		= iniFile.read_float(section.c_str(),"ransac_mahalanobisDistanceThreshold",ransac_mahalanobisDistanceThreshold);

	MRPT_LOAD_CONFIG_VAR(save_feat_coors, bool,   iniFile,section )
	MRPT_LOAD_CONFIG_VAR(debug_show_corrs, bool,   iniFile,section )
	MRPT_LOAD_CONFIG_VAR(debug_save_map_pairs, bool,   iniFile,section )
}
