/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.
#include <mrpt/opengl/CPolyhedron.h>
#include <math.h>
#include <algorithm>
#include <mrpt/math/CMatrix.h>
#include <mrpt/poses/CPoint2D.h>
#include <mrpt/poses/CPose3D.h>

#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace std;

IMPLEMENTS_SERIALIZABLE(CPolyhedron,CRenderizable,mrpt::opengl)

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void CPolyhedron::render() const	{
#if MRPT_HAS_OPENGL_GLUT
	if (mWireframe)	{
		glLineWidth(mLineWidth);
		checkOpenGLError();
		glColor4f(m_color_R,m_color_G,m_color_B,m_color_A);
		glBegin(GL_LINES);
		for (vector<TPolyhedronEdge>::const_iterator it=mEdges.begin();it!=mEdges.end();it++)	{
			CPoint3D p=mVertices[it->v1];
			glVertex3f(p.x,p.y,p.z);
			p=mVertices[it->v2];
			glVertex3f(p.x,p.y,p.z);
		}
		glEnd();
	}	else	{
		checkOpenGLError();
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
		/*glEnable(GL_LIGHTING);
		glEnable(GL_LIGHT0);*/
		glEnable(GL_COLOR_MATERIAL);
		glShadeModel(GL_SMOOTH);
		for (vector<TPolyhedronFace>::const_iterator it=mFaces.begin();it!=mFaces.end();it++)	{
			glBegin(GL_POLYGON);
			glNormal3f(it->normal[0],it->normal[1],it->normal[2]);
			glColor4f(m_color_R,m_color_G,m_color_B,m_color_A);
			for (vector<uint32_t>::const_iterator it2=it->vertices.begin();it2!=it->vertices.end();it2++)	{
				const CPoint3D &p=mVertices[*it2];
				glVertex3f(p.x,p.y,p.z);
			}
			glEnd();
		}
		glDisable(GL_BLEND);
		glDisable(GL_LIGHTING);
	}
#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void CPolyhedron::writeToStream(CStream &out,int *version) const	{
	if (version) *version=0;
	else	{
		writeToStreamRender(out);
		//version 0
		out<<mVertices<<mFaces<<mWireframe<<mLineWidth;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void CPolyhedron::readFromStream(CStream &in,int version)	{
	switch (version)	{
		case 0:
			readFromStreamRender(in);
			in>>mVertices>>mFaces>>mWireframe>>mLineWidth;
			if (!checkConsistence(mVertices,mFaces)) throw std::logic_error("Inconsistent data read from stream");
			for (vector<TPolyhedronFace>::iterator it=mFaces.begin();it!=mFaces.end();it++)	{
				if (!setNormal(*it)) throw std::logic_error("Bad face specification");
				addEdges(*it);
			}
			break;
		default:
			MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)
	};
}

void CPolyhedron::addEdges(const TPolyhedronFace &f)	{
	TPolyhedronEdge e;
	vector<uint32_t>::const_iterator it=f.vertices.begin();
	e.v1=*it;
	it++;
	while (it!=f.vertices.end())	{
		e.v2=*it;
		if (find(mEdges.begin(),mEdges.end(),e)==mEdges.end()) mEdges.push_back(e);
		e.v1=e.v2;
		it++;
	}
	e.v2=*(f.vertices.begin());
	if (find(mEdges.begin(),mEdges.end(),e)==mEdges.end()) mEdges.push_back(e);
}

bool CPolyhedron::setNormal(TPolyhedronFace &f,bool doCheck)	{
	uint32_t nVertices=f.vertices.size();
	if (nVertices<3) return false;
	CMatrix m(0,0);
	if (doCheck)	{
		m.setSize(nVertices-1,3);
		const CPoint3D& pB=mVertices[f.vertices[0]];
		for (uint32_t i=0;i<nVertices-1;i++)	{
			CPoint3D p=mVertices[f.vertices[i+1]];
			m(i,0)=p.x-pB.x;
			m(i,1)=p.y-pB.y;
			m(i,2)=p.z-pB.z;
		}
		if (m.rank(0.01)!=2) return false;
	}
	CPoint3D p1=f.vertices[0];
	CPoint3D p2=f.vertices[1];
	m.setSize(2,3);
	m(0,0)=p2.x-p1.x;
	m(0,1)=p2.y-p1.y;
	m(0,2)=p2.z-p1.z;
	CPoint3D p3;
	if (doCheck)	{
		uint32_t i=2;
		do	{
			p3=f.vertices[i];
			m(1,0)=p3.x-p1.x;
			m(1,1)=p3.y-p1.y;
			m(1,2)=p3.z-p1.z;
		}	while (m.rank()!=2);
		i++;
	}	else	{
		p3=f.vertices[2];
		m(1,0)=p3.x-p1.x;
		m(1,1)=p3.y-p1.y;
		m(1,2)=p3.z-p1.z;
	}
	f.normal[0]=m(0,1)*m(1,2)-m(0,2)*m(1,1);
	f.normal[1]=m(0,2)*m(1,0)-m(0,0)*m(1,2);
	f.normal[2]=m(0,0)*m(1,1)-m(0,1)*m(1,0);
	return true;
}

inline bool lineCoefs(const float &y1,const float &z1,const float &y2,const float &z2,float coefs[3])	{
	if ((y1==y2)&(z1==z2)) return false;	//Both points are the same
	if (y1==y2)	{
		//Equation is y-y1=0
		coefs[0]=1;
		coefs[1]=0;
		coefs[2]=-y1;
		return true;
	}	else	{
		//Equation is:
		// z1 - z2        /z2 - z1       \ .
		// -------y + z + |-------y1 - z1| = 0
		// y2 - y1        \y2 - y1       /
		coefs[0]=(z1-z2)/(y2-y1);
		coefs[1]=1;
		coefs[2]=((z2-z1)/(y2-y1))*y1-z1;
		return true;
	}
}

class FTraceRayPolygon	{
	//This functor will only work if the polygon is CONVEX.
public:
	float &dist;
	const vector<CPoint3D> &vertices;
	bool res;
	FTraceRayPolygon(float &d,const vector<CPoint3D> &v):dist(d),vertices(v),res(false)	{}
	void operator()(const CPolyhedron::TPolyhedronFace &f)	{
		//DOESN'T WORK. REDO
		size_t N=f.vertices.size();
		//Basic sign check
		bool px=false,ny=false,py=false,nz=false,pz=false;
		for (size_t i=0;i<N;i++)	{
			CPoint3D p=vertices[f.vertices[i]];
			if (p.x>=0) px=true;
			if (p.y>0) py=true;
			else if (p.y<0) ny=true;
			else py=ny=true;
			if (p.z>0) pz=true;
			else if (p.z<0) nz=true;
			else pz=nz=true;
			if (px&&py&&ny&&pz&&nz) break;
		}
		if (!px||!py||!ny||!pz||!nz) return;
		//distance calculation
		float mat[9];
		for (int i=0;i<3;i++)	{
			mat[3*i]=vertices[i].x;
			mat[3*i+1]=vertices[i].y;
			mat[3*i+2]=vertices[i].z;
		}
		CMatrixTemplateNumeric<float> M=CMatrixTemplateNumeric<float>(3,3,mat);
		float pDist;
		switch (M.rank())	{
			case 3:
				{
					for (int i=0;i<9;i+=3) mat[i]=1;
					float d2=CMatrixTemplateNumeric<float>(3,3,mat).det();
					if (d2==0) return;
					else pDist=M.det()/d2;
				}
				break;
			case 2:
				float mat2[6];
				for (int i=0;i<2;i++)	{
					mat[3*i]=vertices[i+1].x-vertices[0].x;
					mat[3*i+1]=vertices[i+1].y-vertices[0].x;
					mat[3*i+2]=vertices[i+1].z-vertices[0].x;
				}
				if (CMatrixTemplateNumeric<float>(2,3,mat2).rank()==2) pDist=0;
				else return;
			default:
				return;
		}
		if (pDist<0) return;
		if (res&&(pDist>=dist)) return;
		//Check if (pDist,0,0) is inside the polygon
		CPoint3D p0=vertices[f.vertices[0]];
		CPoint3D p1=vertices[f.vertices[1]];
		CPoint3D p2=vertices[f.vertices[2]];
		float coefs[3];
		for (size_t i=0;i<N;i++)	{
			if (!lineCoefs(p0.y,p0.z,p1.y,p1.z,coefs)) return;
			if (coefs[2]==0) break;
			else if (((coefs[0]*p2.y+coefs[1]*p2.z+coefs[2])>0)!=(coefs[2]>0)) return;
			p0=p1;
			p1=p2;
			p2=vertices[f.vertices[(i+3)%N]];
		}
		dist=pDist;
		res=true;
	}
};
bool CPolyhedron::traceRay(const mrpt::poses::CPose3D &o,float &dist) const	{
	vector<CPoint3D> tVerts(mVertices.size());
	for (size_t i=0;i<mVertices.size();i++) tVerts[i]=o+mVertices[i];
	FTraceRayPolygon functor=FTraceRayPolygon(dist,tVerts);
	for_each(mFaces.begin(),mFaces.end(),functor);
	return functor.res;
}

CPolyhedronPtr CPolyhedron::CreateTetrahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts(4);
	vector<TPolyhedronFace> faces(4);
	verts[0]=CPoint3D(0,0,radius);
	float z1=-radius/3.0;
	float d1=radius*sqrt(8.0)/3.0;
	for (int i=1;i<4;i++) verts[i]=CPoint3D(d1*cos(M_PI*i*2.0/3.0),d1*sin(M_PI*i*2.0/3.0),z1);
	TPolyhedronFace f;
	f.vertices.resize(3);
	for (size_t i=0;i<4;i++)	{
		f.vertices[0]=(i>=3)?1:0;
		f.vertices[1]=(i>=2)?2:1;
		f.vertices[2]=(i>=1)?3:2;
		faces[i]=f;
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateHexahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	float d=radius/sqrt(3.0);
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	for (int i=0;i<8;i++) verts.push_back(CPoint3D((i&1)?-d:d,(i&2)?-d:d,(i&4)?-d:d));
	static uint32_t faceVertices[]={0,1,5,4, 2,3,7,6, 0,2,6,4, 1,3,7,5, 0,1,3,2, 4,5,7,6};
	TPolyhedronFace f;
	for (uint32_t *p=reinterpret_cast<uint32_t *>(&faceVertices);p<24+reinterpret_cast<uint32_t *>(&faceVertices);p+=4)	{
		f.vertices.insert(f.vertices.begin(),p,p+4);
		faces.push_back(f);
		f.vertices.clear();
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateOctahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	float d=radius/sqrt(2.0);
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	verts.push_back(CPoint3D(0,0,radius));
	verts.push_back(CPoint3D(0,0,-radius));
	for (int i=0;i<4;i++) verts.push_back(CPoint3D((i&1)?-d:d,(i&2)?-d:d,0));
	static uint32_t faceVertices[]={0,2,3, 0,3,5, 0,5,4, 0,4,2, 1,2,3, 1,3,5, 1,5,4, 1,4,2};
	TPolyhedronFace f;
	for (uint32_t *p=reinterpret_cast<uint32_t *>(&faceVertices);p<24+reinterpret_cast<uint32_t *>(&faceVertices);p+=3)	{
		f.vertices.insert(f.vertices.begin(),p,p+3);
		faces.push_back(f);
		f.vertices.clear();
	}
	return CreateNoCheck(verts,faces);
}

//TODO: create a member function for CPoint3D.
inline CPoint3D Minus(const CPoint3D &p)	{
	return CPoint3D(-p.x,-p.y,-p.z);
}

CPolyhedronPtr CPolyhedron::CreateDodecahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	const static float den=sqrt(15.0+6*sqrt(5.0));
	float z1=radius*(2.0+sqrt(5.0))/den;
	float z2=radius/den;
	float d1=radius*(1+sqrt(5.0))/den;
	float d2=radius*(3+sqrt(5.0))/den;
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	for (int i=0;i<5;i++) verts.push_back(CPoint3D(d1*cos(2*M_PI*i/5),d1*sin(2*M_PI*i/5),z1));
	for (int i=0;i<5;i++) verts.push_back(CPoint3D(d2*cos(2*M_PI*i/5),d2*sin(2*M_PI*i/5),z2));
	for (uint32_t i=0;i<5;i++) verts.push_back(Minus(verts[i+5]));
	for (uint32_t i=0;i<5;i++) verts.push_back(Minus(verts[i]));
	//const uint32_t v1[]={0,1,2,3,4,0};
	const uint32_t v3[]={13,14,10,11,12};
	//const uint32_t v2[]={15,16,17,18,19,15};
	TPolyhedronFace f,g;
	f.vertices.reserve(5);
	g.vertices.reserve(5);
	for (uint32_t i=0;i<5;i++)	{
		uint32_t ii=(i+1)%5;
		f.vertices.push_back(i);
		f.vertices.push_back(i+5);
		f.vertices.push_back(v3[i]);
		f.vertices.push_back(ii+5);
		f.vertices.push_back(ii);
		g.vertices.push_back(i+15);
		g.vertices.push_back(i+10);
		g.vertices.push_back(v3[i]-5);
		g.vertices.push_back(ii+10);
		g.vertices.push_back(ii+15);
		faces.push_back(f);
		faces.push_back(g);
		f.vertices.clear();
		g.vertices.clear();
	}
	for (uint32_t i=0;i<5;i++)	{
		f.vertices.push_back(i);
		g.vertices.push_back(i+15);
	}
	faces.push_back(f);
	faces.push_back(g);
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateIcosahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	float z=radius/sqrt(5.0);
	float d=z+z;
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	for (int i=0;i<5;i++) verts.push_back(CPoint3D(d*cos(2*M_PI*i/5),d*sin(2*M_PI*i/5),z));
	for (int i=0;i<5;i++) verts.push_back(Minus(verts[i]));
	verts.push_back(CPoint3D(0,0,radius));
	verts.push_back(CPoint3D(0,0,-radius));
	TPolyhedronFace f;
	f.vertices.resize(3);
	f.vertices[0]=10;
	for (uint32_t i=0;i<5;i++)	{
		f.vertices[1]=i;
		f.vertices[2]=(i+1)%5;
		faces.push_back(f);
	}
	f.vertices[0]=11;
	for (uint32_t i=0;i<5;i++)	{
		f.vertices[1]=i+5;
		f.vertices[2]=5+((i+1)%5);
		faces.push_back(f);
	}
	f.vertices.resize(3);
	//The following code creates the "center" triangles.
	//It's efficient, and makes perfect sense if you look at the icosahedron from above.
	f.vertices[0]=0;
	f.vertices[1]=8;
	f.vertices[2]=1;
	faces.push_back(f);
	static const uint32_t faceVertices[]={9,2,5,3,6,4,7,0,8};
	for (uint32_t i=0;i<9;i++)	{
		f.vertices[0]=f.vertices[1];
		f.vertices[1]=f.vertices[2];
		f.vertices[2]=faceVertices[i];
		faces.push_back(f);
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateCubicPrism(const float x1,const float x2,const float y1,const float y2,const float z1,const float z2)	{
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	for (int i=0;i<8;i++) verts.push_back(CPoint3D((i&1)?x2:x1,(i&2)?y2:y1,(i&4)?z2:z1));
	static uint32_t faceVertices[]={0,1,5,4, 2,3,7,6, 0,2,6,4, 1,3,7,5, 0,1,3,2, 4,5,7,6};
	TPolyhedronFace f;
	for (uint32_t *p=reinterpret_cast<uint32_t *>(&faceVertices);p<24+reinterpret_cast<uint32_t *>(&faceVertices);p+=4)	{
		f.vertices.insert(f.vertices.begin(),p,p+4);
		faces.push_back(f);
		f.vertices.clear();
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreatePyramid(const vector<CPoint2D> &baseVertices,const float height)	{
	uint32_t n=baseVertices.size();
	if (baseVertices.size()<3) throw std::logic_error("Not enought vertices");
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	verts.push_back(CPoint3D(0,0,height));
	for (vector<CPoint2D>::const_iterator it=baseVertices.begin();it!=baseVertices.end();it++) verts.push_back(CPoint3D(it->x,it->y,0));
	TPolyhedronFace f,g;
	f.vertices.push_back(0);
	f.vertices.push_back(n);
	f.vertices.push_back(1);
	g.vertices.push_back(1);
	faces.push_back(f);
	for (uint32_t i=2;i<=n;i++)	{
		f.vertices.erase(f.vertices.begin()+1);
		f.vertices.push_back(i);
		faces.push_back(f);
		g.vertices.push_back(i);
	}
	faces.push_back(g);
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateDoublePyramid(const vector<CPoint2D> &baseVertices,const float height1,const float height2)	{
	uint32_t n=baseVertices.size();
	if (baseVertices.size()<3) throw std::logic_error("Not enought vertices");
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	verts.push_back(CPoint3D(0,0,height1));
	for (vector<CPoint2D>::const_iterator it=baseVertices.begin();it!=baseVertices.end();it++) verts.push_back(CPoint3D(it->x,it->y,0));
	verts.push_back(CPoint3D(0,0,-height2));
	TPolyhedronFace f,g;
	f.vertices.push_back(0);
	f.vertices.push_back(n);
	f.vertices.push_back(1);
	g.vertices.push_back(n+1);
	g.vertices.push_back(n);
	g.vertices.push_back(1);
	faces.push_back(f);
	for (uint32_t i=2;i<=n;i++)	{
		f.vertices.erase(f.vertices.begin()+1);
		f.vertices.push_back(i);
		faces.push_back(f);
		g.vertices.erase(g.vertices.begin()+1);
		g.vertices.push_back(i);
		faces.push_back(g);
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateTruncatedTetrahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	float d=radius/sqrt(66.0);
	float r12=sqrt(12.0)*d;
	float _r18=-sqrt(18.0)*d;
	float r2=sqrt(2.0)*d;
	verts.push_back(CPoint3D(0,4.0*d,5.0*r2));
	verts.push_back(CPoint3D(r12,-d-d,5.0*r2));
	verts.push_back(CPoint3D(-r12,-d-d,5.0*r2));
	verts.push_back(CPoint3D(0,8.0*d,r2));
	verts.push_back(CPoint3D(r12,6.0*d,_r18));
	verts.push_back(CPoint3D(-r12,6.0*d,_r18));
	verts.push_back(CPoint3D(r12+r12,-4.0*d,r2));
	verts.push_back(CPoint3D(r12+r12,0.0,_r18));
	verts.push_back(CPoint3D(r12,-6.0*d,_r18));
	verts.push_back(CPoint3D(-r12-r12,-4.0*d,r2));
	verts.push_back(CPoint3D(-r12-r12,0,_r18));
	verts.push_back(CPoint3D(-r12,-6.0*d,_r18));
	TPolyhedronFace f;
	for (uint32_t i=0;i<4;i++)	{
		for (uint32_t j=0;j<3;j++) f.vertices.push_back(3*i+j);
		faces.push_back(f);
		f.vertices.clear();
	}
	static uint32_t faces6[]={5,10,11,8,7,4, 2,9,11,8,6,1, 2,9,10,5,3,0, 1,6,7,4,3,0};
	f.vertices.insert(f.vertices.begin(),reinterpret_cast<uint32_t *>(&faces6),6+reinterpret_cast<uint32_t *>(&faces6));
	faces.push_back(f);
	for (uint32_t *p=6+reinterpret_cast<uint32_t *>(&faces6);p<24+reinterpret_cast<uint32_t *>(&faces6);p+=6)	{
		f.vertices.clear();
		f.vertices.insert(f.vertices.begin(),p,p+6);
		faces.push_back(f);
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateCuboctahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts;
	vector<TPolyhedronFace> faces;
	float d=radius/sqrt(2.0);
	verts.push_back(CPoint3D(d,0,d));
	verts.push_back(CPoint3D(0,d,d));
	verts.push_back(CPoint3D(0,-d,d));
	verts.push_back(CPoint3D(-d,0,d));
	verts.push_back(CPoint3D(d,d,0));
	verts.push_back(CPoint3D(d,-d,0));
	verts.push_back(CPoint3D(-d,d,0));
	verts.push_back(CPoint3D(-d,-d,0));
	verts.push_back(CPoint3D(d,0,-d));
	verts.push_back(CPoint3D(0,d,-d));
	verts.push_back(CPoint3D(0,-d,-d));
	verts.push_back(CPoint3D(-d,0,-d));
	TPolyhedronFace f;
	static uint32_t faces3[]={0,1,4, 0,2,5, 1,3,6, 2,3,7, 8,9,4, 8,10,5, 9,11,6, 10,11,7};
	static uint32_t faces4[]={0,1,3,2, 8,9,11,10, 0,4,8,5, 1,4,9,6, 2,5,10,7, 3,6,11,7};
	for (uint32_t *p=reinterpret_cast<uint32_t *>(&faces3);p<24+reinterpret_cast<uint32_t *>(&faces3);p+=3)	{
		f.vertices.insert(f.vertices.begin(),p,p+3);
		faces.push_back(f);
		f.vertices.clear();
	}
	for (uint32_t *p=reinterpret_cast<uint32_t *>(&faces4);p<24+reinterpret_cast<uint32_t *>(&faces4);p+=4)	{
		f.vertices.insert(f.vertices.begin(),p,p+4);
		faces.push_back(f);
		f.vertices.clear();
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateTruncatedHexahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts(24);
	vector<TPolyhedronFace> faces;
	faces.reserve(14);
	float a=sqrt(2.0)/(2.0+sqrt(2.0));
	float R=radius*sqrt(a*a+1.0)/2.0;
	float d=radius/2.0;
	float d2=radius/sqrt(2.0);
	for (size_t i=0;i<8;i++)	{
		verts[i]=CPoint3D(R*cos((2*i+1)*M_PI/8),R*sin((2*i+1)*M_PI/8),d);
		verts[i+16]=CPoint3D(R*cos((2*i+1)*M_PI/8),R*sin((2*i+1)*M_PI/8),-d);
	}
	for (size_t i=0;i<4;i++)	{
		verts[i+8]=CPoint3D(cos((2*i+1)*M_PI/4)*d2,sin((2*i+1)*M_PI/4)*d2,radius*a/2);
		verts[i+12]=CPoint3D(cos((2*i+1)*M_PI/4)*d2,sin((2*i+1)*M_PI/4)*d2,-radius*a/2);
	}
	TPolyhedronFace f;
	static uint32_t faces8[]={0,7,11,15,23,16,12,8, 5,6,11,15,22,21,14,10, 3,4,10,14,20,19,13,9, 1,2,9,13,18,17,12,8};
	for (uint32_t *p=reinterpret_cast<uint32_t *>(&faces8);p<32+reinterpret_cast<uint32_t *>(&faces8);p+=8)	{
		f.vertices.insert(f.vertices.begin(),p,p+8);
		faces.push_back(f);
		f.vertices.clear();
	}
	for (uint32_t i=0;i<=16;i+=8)	{
		for (int j=0;j<8;j++) f.vertices.push_back(i++);
		faces.push_back(f);
		f.vertices.clear();
	}
	for (uint32_t i=0;i<4;i++)	{
		f.vertices.push_back(i+8);
		f.vertices.push_back(i<<1);
		f.vertices.push_back(1+(i<<1));
		faces.push_back(f);
		f.vertices.clear();
	}
	for (uint32_t i=0;i<4;i++)	{
		f.vertices.push_back(i+12);
		f.vertices.push_back(16+(i<<1));
		f.vertices.push_back(17+(i<<1));
		faces.push_back(f);
		f.vertices.clear();
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateTruncatedOctahedron(const float radius)	{
	//MAL. Rehacer
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts(24);
	vector<TPolyhedronFace> faces(14);
/*	float R1=radius/sqrt(5.0);
	float R2=R1+R1;
	for (uint32_t i=0;i<4;i++)	{
		float c=cos((2*i+1)*M_PI/4);
		float s=sin((2*i+1)*M_PI/4);
		verts[i]=CPoint3D(c*R1,s*R1,R2);
		verts[i+4]=CPoint3D(c*R2,s*R2,R1);
		verts[i+16]=CPoint3D(c*R2,s*R2,-R1);
		verts[i+20]=CPoint3D(c*R1,s*R1,-R2);
	}*/
	float R1=radius/sqrt(5.0);
	float R2=R1+R1;
	float s2=sqrt(0.5);
	for (uint32_t i=0;i<4;i++)	{
		float c=(i==0||i==3)?s2:-s2;
		float s=(i>=2)?-s2:s2;
		verts[i]=CPoint3D(c*R1,s*R1,R2);
		verts[i+4]=CPoint3D(c*R2,s*R2,R1);
		verts[i+16]=CPoint3D(c*R2,s*R2,-R1);
		verts[i+20]=CPoint3D(c*R1,s*R1,-R2);
	}
	float R3=radius*3.0/sqrt(10.0);
	verts[8].x=verts[15].x=verts[9].y=verts[10].y=R3;
	verts[11].x=verts[12].x=verts[13].y=verts[14].y=-R3;
	for (uint32_t i=0;i<2;i++)	{
		verts[i+9].x=verts[i].x;
		verts[i+13].x=verts[i+2].x;
		verts[7*i+8].y=verts[3*i].y;
		verts[i+11].y=verts[i+1].y;
	}
	for (uint32_t i=8;i<16;i++) verts[i].z=0;
	TPolyhedronFace f,g,h;
	g.vertices.resize(4);
	h.vertices.resize(4);
	for (uint32_t i=0;i<4;i++)	{
		f.vertices.resize(4);
		f.vertices[0]=i+4;
		f.vertices[1]=i+i+8;
		f.vertices[2]=i+16;
		f.vertices[3]=i+i+9;
		faces[i]=f;
		g.vertices[i]=i;
		h.vertices[i]=i+20;
		f.vertices.resize(6);
		uint32_t ii=(i+1)%4;
		f.vertices[0]=i;
		f.vertices[1]=i+4;
		f.vertices[2]=i+i+9;
		f.vertices[3]=((i+i+2)%8)+8;
		f.vertices[4]=ii+4;
		f.vertices[5]=ii;
		faces[i+4]=f;
		f.vertices[0]=i+20;
		f.vertices[1]=i+16;
		f.vertices[4]=ii+16;
		f.vertices[5]=ii+20;
		faces[i+8]=f;
		f.vertices.clear();
	}
	faces[12]=g;
	faces[13]=h;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateRhombicuboctahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts(24);
	vector<TPolyhedronFace> faces(26);
	float C1=radius*sqrt((2.0+sqrt(2.0))/(6.0-sqrt(2.0)));
	float S1=sqrt(radius*radius-C1*C1);
	float Z1=S1/sqrt(2.0);
	float R1=sqrt(radius*radius-Z1*Z1);
	for (uint32_t i=0;i<4;i++)	{
		float c4=cos((2*i+1)*M_PI/4);
		float s4=sin((2*i+1)*M_PI/4);
		verts[i]=CPoint3D(S1*c4,S1*s4,C1);
		verts[i+20]=CPoint3D(S1*c4,S1*s4,-C1);
	}
	for (uint32_t i=0;i<8;i++)	{
		float c8=cos((2*i+1)*M_PI/8);
		float s8=sin((2*i+1)*M_PI/8);
		verts[i+4]=CPoint3D(R1*c8,R1*s8,Z1);
		verts[i+12]=CPoint3D(R1*c8,R1*s8,-Z1);
	}
	TPolyhedronFace f;
	f.vertices.resize(3);
	for (uint32_t i=0;i<4;i++)	{
		f.vertices[0]=i;
		f.vertices[1]=2*i+4;
		f.vertices[2]=2*i+5;
		faces[i]=f;
		f.vertices[0]+=20;
		f.vertices[1]+=8;
		f.vertices[2]+=8;
		faces[i+4]=f;
	}
	f.vertices.resize(4);
	TPolyhedronFace g,h;
	g.vertices.resize(4);
	h.vertices.resize(4);
	for (uint32_t i=0;i<4;i++)	{
		uint32_t ii=(i+1)%4;
		f.vertices[0]=i;
		f.vertices[1]=2*i+5;
		f.vertices[2]=2*ii+4;
		f.vertices[3]=ii;
		faces[i+8]=f;
		f.vertices[0]+=20;
		f.vertices[1]+=8;
		f.vertices[2]+=8;
		f.vertices[3]+=20;
		faces[i+12]=f;
		g.vertices[i]=i;
		h.vertices[i]=20+i;
	}
	faces[24]=g;
	faces[25]=h;
	for (uint32_t i=0;i<8;i++)	{
		uint32_t ii=(i+1)%8;
		f.vertices[0]=i+4;
		f.vertices[1]=ii+4;
		f.vertices[2]=ii+12;
		f.vertices[3]=i+12;
		faces[i+16]=f;
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateIcosidodecahedron(const float radius)	{
	if (radius==0) return CreateEmpty();
	vector<CPoint3D> verts(30);
	vector<TPolyhedronFace> faces(32);
	float R1=radius*sqrt((5.0-sqrt(5.0))/10.0);
	float R2=radius*sqrt((5.0+sqrt(5.0))/10.0);
	for (int i=0;i<5;i++)	{
		float ca=cos((2*i+1.0)*M_PI/5.0);
		float sa=sin((2*i+1.0)*M_PI/5.0);
		float cb=cos(2.0*i*M_PI/5.0);
		float sb=sin(2.0*i*M_PI/5.0);
		verts[i]=CPoint3D(R1*ca,R1*sa,R2);
		verts[i+5]=CPoint3D(R2*cb,R2*sb,R1);
		verts[i+20]=CPoint3D(R2*ca,R2*sa,-R1);
		verts[i+25]=CPoint3D(R1*cb,R1*sb,-R2);
	}
	for (int i=0;i<10;i++) verts[i+10]=CPoint3D(radius*cos((2*i+1)*M_PI/10.0),radius*sin((2*i+1)*M_PI/10.0),0);
	TPolyhedronFace f;
	f.vertices.resize(3);
	for (uint32_t i=0;i<5;i++)	{
		uint32_t ii=(i+1)%5;
		f.vertices[0]=i;
		f.vertices[1]=ii;
		f.vertices[2]=ii+5;
		faces[i]=f;
		f.vertices[0]=i+5;
		f.vertices[1]=2*i+10;
		f.vertices[2]=((i+i+8)%10)+11;
		faces[i+5]=f;
		f.vertices[0]=2*i+10;
		f.vertices[1]=2*i+11;
		f.vertices[2]=i+20;
		faces[i+10]=f;
		f.vertices[0]=i+20;
		f.vertices[1]=i+25;
		f.vertices[2]=ii+25;
		faces[i+15]=f;
	}
	f.vertices.resize(5);
	TPolyhedronFace g,h;
	g.vertices.resize(5);
	h.vertices.resize(5);
	for (uint32_t i=0;i<5;i++)	{
		f.vertices[0]=i;
		f.vertices[1]=i+5;
		f.vertices[2]=2*i+10;
		f.vertices[3]=2*i+11;
		f.vertices[4]=((i+1)%5)+5;
		faces[i+20]=f;
		f.vertices[0]=i+25;
		f.vertices[1]=i+20;
		f.vertices[2]=2*i+10;
		f.vertices[3]=((i+i+8)%10)+11;
		f.vertices[4]=((i+4)%5)+20;
		faces[i+25]=f;
		g.vertices[i]=i;
		h.vertices[i]=i+25;
	}
	faces[30]=g;
	faces[31]=h;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateTruncatedPyramid(const vector<CPoint2D> &baseVertices,const float height,const float ratio)	{
	uint32_t n=baseVertices.size();
	if (n<3) throw std::logic_error("Not enough vertices");
	vector<CPoint3D> verts(n+n);
	vector<TPolyhedronFace> faces(n+2);
	TPolyhedronFace f,g,h;
	f.vertices.resize(4);
	g.vertices.resize(n);
	h.vertices.resize(n);
	for (uint32_t i=0;i<n;i++)	{
		verts[i]=CPoint3D(baseVertices[i].x,baseVertices[i].y,0);
		verts[i+n]=CPoint3D(baseVertices[i].x*ratio,baseVertices[i].y*ratio,height);
		uint32_t ii=(i+1)%n;
		f.vertices[0]=i;
		f.vertices[1]=ii;
		f.vertices[2]=ii+n;
		f.vertices[3]=i+n;
		faces[i]=f;
		g.vertices[i]=i;
		h.vertices[i]=i+n;
	}
	faces[n]=g;
	faces[n+1]=h;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateCustomPrism(const vector<CPoint2D> &baseVertices,const float height)	{
	return CreateTruncatedPyramid(baseVertices,height,1.0);
}

CPolyhedronPtr CPolyhedron::CreateRegularAntiprism(const uint32_t numBaseEdges,const float baseRadius,const float height)	{
	if (numBaseEdges<3) throw std::logic_error("Not enough vertices");
	vector<CPoint3D> verts(numBaseEdges<<1);
	vector<TPolyhedronFace> faces((numBaseEdges<<1)+2);
	float incr=2.0*M_PI/numBaseEdges;
	float base0=0;
	float base1=incr/2;
	TPolyhedronFace f,g,h;
	f.vertices.resize(3);
	g.vertices.resize(numBaseEdges);
	h.vertices.resize(numBaseEdges);
	for (uint32_t i=0;i<numBaseEdges;i++)	{
		verts[i]=CPoint3D(baseRadius*cos(base0),baseRadius*sin(base0),0);
		verts[i+numBaseEdges]=CPoint3D(baseRadius*cos(base1),baseRadius*sin(base1),height);
		base0+=incr;
		base1+=incr;
		uint32_t ii=(i+1)%numBaseEdges;
		f.vertices[0]=i;
		f.vertices[1]=ii;
		f.vertices[2]=i+numBaseEdges;
		faces[i]=f;
		f.vertices[0]=i+numBaseEdges;
		f.vertices[1]=ii+numBaseEdges;
		f.vertices[2]=ii;
		faces[i+numBaseEdges]=f;
		g.vertices[i]=i;
		h.vertices[i]=i+numBaseEdges;
	}
	faces[numBaseEdges<<1]=g;
	faces[(numBaseEdges<<1)+1]=h;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateRegularPrism(const uint32_t numBaseEdges,const float baseRadius,const float height)	{
	if (numBaseEdges<3) throw std::logic_error("Not enough vertices");
	vector<CPoint3D> verts(numBaseEdges<<1);
	vector<TPolyhedronFace> faces(numBaseEdges+2);
	float base=0;
	TPolyhedronFace f,g,h;
	f.vertices.resize(4);
	g.vertices.resize(numBaseEdges);
	h.vertices.resize(numBaseEdges);
	for (uint32_t i=0;i<numBaseEdges;i++)	{
		verts[i]=CPoint3D(baseRadius*cos(base),baseRadius*sin(base),0);
		verts[i+numBaseEdges]=CPoint3D(baseRadius*cos(base),baseRadius*sin(base),height);
		base+=2.0*M_PI/numBaseEdges;
		uint32_t ii=(i+1)%numBaseEdges;
		f.vertices[0]=i;
		f.vertices[1]=ii;
		f.vertices[2]=ii+numBaseEdges;
		f.vertices[3]=i+numBaseEdges;
		faces[i]=f;
		g.vertices[i]=i;
		h.vertices[i]=i+numBaseEdges;
	}
	faces[numBaseEdges]=g;
	faces[numBaseEdges+1]=h;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateCustomAntiprism(const vector<CPoint2D> &bottomBase,const vector<CPoint2D> &topBase,const float height)	{
	uint32_t n=bottomBase.size();
	if (n<3) throw std::logic_error("Not enough vertices");
	if (n!=topBase.size()) throw std::logic_error("Bases' number of vertices do not match");
	vector<CPoint3D> verts(n+n);
	vector<TPolyhedronFace> faces(n+n+2);
	TPolyhedronFace f,g,h;
	f.vertices.resize(3);
	g.vertices.resize(n);
	h.vertices.resize(n);
	for (uint32_t i=0;i<n;i++)	{
		verts[i]=CPoint3D(bottomBase[i].x,bottomBase[i].y,0);
		verts[n+i]=CPoint3D(topBase[i].x,topBase[i].y,height);
		uint32_t ii=(i+1)%n;
		f.vertices[0]=i;
		f.vertices[1]=ii;
		f.vertices[2]=i+n;
		faces[i]=f;
		f.vertices[0]=i+n;
		f.vertices[1]=ii+n;
		f.vertices[2]=ii;
		faces[n+i]=f;
		g.vertices[i]=i;
		h.vertices[i]=n+i;
	}
	faces[n+n]=g;
	faces[n+n+1]=h;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateParallelepiped(const CPoint3D &base,const CPoint3D &v1,const CPoint3D &v2,const CPoint3D &v3)	{
	vector<CPoint3D> verts(8);
	vector<TPolyhedronFace> faces(6);
	for (uint32_t i=0;i<8;i++)	{
		verts[i]=base;
		if (i&1) verts[i]=verts[i]+v1;
		if (i&2) verts[i]=verts[i]+v2;
		if (i&4) verts[i]=verts[i]+v3;
	}
	TPolyhedronFace f;
	f.vertices.resize(4);
	f.vertices[0]=0;
	f.vertices[1]=1;
	f.vertices[2]=3;
	f.vertices[3]=2;
	faces[0]=f;
	//f.vertices[0]=0;
	//f.vertices[1]=1;
	f.vertices[2]=5;
	f.vertices[3]=4;
	faces[1]=f;
	//f.vertices[0]=0;
	f.vertices[1]=2;
	f.vertices[2]=6;
	//f.vertices[3]=4;
	faces[2]=f;
	for (uint32_t i=0;i<3;i++)	{
		uint32_t valueAdd=4>>i;
		faces[i+3].vertices.resize(4);
		for (uint32_t j=0;j<4;j++) faces[i+3].vertices[j]=faces[i].vertices[j]+valueAdd;
	}
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateRegularPyramid(const uint32_t numBaseEdges,const float baseRadius,const float height)	{
	if (baseRadius==0) return CreateEmpty();
	if (numBaseEdges<3) throw std::logic_error("Not enough vertices");
	vector<CPoint3D> verts(numBaseEdges+1);
	vector<TPolyhedronFace> faces(numBaseEdges+1);
	TPolyhedronFace f,g;
	f.vertices.resize(3);
	f.vertices[0]=numBaseEdges;
	g.vertices.resize(numBaseEdges);
	for (uint32_t i=0;i<numBaseEdges;i++)	{
		verts[i]=CPoint3D(cos(2*M_PI*i/numBaseEdges),sin(2*M_PI*i/numBaseEdges),0);
		f.vertices[1]=i;
		f.vertices[2]=(i+1)%numBaseEdges;
		faces[i]=f;
		g.vertices[i]=i;
	}
	verts[numBaseEdges]=CPoint3D(0,0,height);
	faces[numBaseEdges]=g;
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateRegularDoublePyramid(const uint32_t numBaseEdges,const float baseRadius,const float height1,const float height2)	{
	if (baseRadius==0) return CreateEmpty();
	if (numBaseEdges<3) throw std::logic_error("Not enough vertices");
	vector<CPoint3D> verts(numBaseEdges+2);
	vector<TPolyhedronFace> faces(numBaseEdges<<1);
	TPolyhedronFace f,g;
	f.vertices.resize(3);
	g.vertices.resize(3);
	f.vertices[0]=numBaseEdges;
	g.vertices[0]=numBaseEdges+1;
	g.vertices.resize(numBaseEdges);
	for (uint32_t i=0;i<numBaseEdges;i++)	{
		verts[i]=CPoint3D(cos(2*M_PI*i/numBaseEdges),sin(2*M_PI*i/numBaseEdges),0);
		g.vertices[1]=f.vertices[1]=i;
		g.vertices[2]=f.vertices[2]=(i+1)%numBaseEdges;
		faces[i]=f;
		faces[i+numBaseEdges]=g;
	}
	verts[numBaseEdges]=CPoint3D(0,0,height1);
	verts[numBaseEdges+1]=CPoint3D(0,0,-height2);
	return CreateNoCheck(verts,faces);
}

CPolyhedronPtr CPolyhedron::CreateArchimedeanRegularPrism(const uint32_t numBaseEdges,const float baseRadius)	{
	if (baseRadius==0) return CreateEmpty();
	if (numBaseEdges<3) throw std::logic_error("Not enough vertices");
	//for the prism to be archimedean, the height must equal each base's edge length.
	return CreateRegularPrism(numBaseEdges,baseRadius,baseRadius*2*cos((M_PI*(numBaseEdges-2))/(numBaseEdges<<1)));
}

CPolyhedronPtr CPolyhedron::CreateArchimedeanRegularAntiprism(const uint32_t numBaseEdges,const float baseRadius)	{
	if (baseRadius==0) return CreateEmpty();
	if (numBaseEdges<3) throw std::logic_error("Not enough vertices");
	//Edge length=radius*2*sin((M_PI*(numBaseEdges-2))/(numBaseEdges<<1))
	//Apothem=radius*cos((M_PI*(numBaseEdges-2))/(numBaseEdges<<1))
	//The vertical edges' length is equal to sqrt(x^2+y^2+z^2)=sqrt((radius-apothem)^2+height^2)
	//So, height=sqrt((edge length)^2-(radius-apothem)^2)
	float angle=M_PI/numBaseEdges;
	float tmp=cos(angle)-cos(angle+angle);
	return CreateRegularAntiprism(numBaseEdges,baseRadius,baseRadius*sqrt(tmp+tmp));
}

CPolyhedronPtr CPolyhedron::CreateRegularTruncatedPyramid(const uint32_t numBaseEdges,const float baseRadius,const float height,const float ratio)	{
	//TODO
	return CreateEmpty();
}

bool CPolyhedron::checkConsistence(const vector<CPoint3D> &vertices,const vector<TPolyhedronFace> &faces)	{
	//REDO
	/*if (vertices.size()>0)	{
		const CPoint3D &p=vertices[0];
		for (uint32_t i=1;i<vertices.size();i++) if (vertices[i]==p) return false;
	}
	uint32_t nVertices=vertices.size();
	for (vector<TPolyhedronEdge>::const_iterator it=faces.begin();it!=faces.end();it++) for (vector<uint32_t>::const_iterator it2=it->vertices.begin();it2!=it->vertices.end();it2++) if ((*it2<0)||(*it2)>=nVertices)) return false;*/
	return true;
}

float CPolyhedron::TPolyhedronEdge::length(const vector_serializable<CPoint3D> &vertices) const	{
	//TODO. Calculate as Euclidean distance
	return 0.0;
}

float CPolyhedron::TPolyhedronFace::area(const vector_serializable<CPoint3D> &vertices) const	{
	//TODO. Calculate as set of triangles with a vertex in the center of the polygon (will work only with convex polygons).
	//Use this formule to compute each triangle's area:
	//Let A,B,C be the vertices: let V2=B-A and V3=C-A.
	//Let V2=(V21,V22,V23) and V3=(V31,V32,V33).
	//area=1/2*sqrt((V22V33-V23V32)^2+(V23V31-V21V33)^2+(V21V32+V22V31)^2)
	return 0.0;
}

void CPolyhedron::getEdgesLength(std::vector<float> &lengths) const	{
	lengths.resize(mEdges.size());
	std::vector<float>::iterator it2=lengths.begin();
	for (std::vector<TPolyhedronEdge>::const_iterator it=mEdges.begin();it!=mEdges.end();it++,it2++) *it2=it->length(mVertices);
}

void CPolyhedron::getFacesArea(std::vector<float> &areas) const	{
	areas.resize(mFaces.size());
	std::vector<float>::iterator it2=areas.begin();
	for (std::vector<TPolyhedronFace>::const_iterator it=mFaces.begin();it!=mFaces.end();it++,it2++) *it2=it->area(mVertices);
}

float CPolyhedron::getVolume() const	{
	//TODO. Calculate as set of pyramids whoses apices are situated in the center of the polyhedron (will work only with convex polyhedrons).
	//NOTE: try to cache results from facesArea using a private mutable member, since polygons are non-mutable objects.
	//Pyramid volume=V=1/3*base area*height. Height=abs((A-V)·N), where A is the apex, V is any other vertex, N is the base's normal vector and (·) is the scalar product.
	return 0.0;
}

CStream &mrpt::opengl::operator>>(CStream& in,CPolyhedron::TPolyhedronEdge &o)	{
	in>>o.v1>>o.v2;
	return in;
}

CStream &mrpt::opengl::operator<<(CStream &out,const CPolyhedron::TPolyhedronEdge &o)	{
	out<<o.v1<<o.v2;
	return out;
}

CStream &mrpt::opengl::operator>>(CStream& in,CPolyhedron::TPolyhedronFace &o)	{
	in>>o.vertices>>o.normal[0]>>o.normal[1]>>o.normal[2];
	return in;
}

CStream &mrpt::opengl::operator<<(CStream& out,const CPolyhedron::TPolyhedronFace &o)	{
	out<<o.vertices<<o.normal[0]<<o.normal[1]<<o.normal[2];
	return out;
}

