/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/math/CPolygon.h>
#include <mrpt/math/utils.h>

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace std;

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CPolygon, CSerializable, mrpt::math)


/*---------------------------------------------------------------
						Constructor
  ---------------------------------------------------------------*/
CPolygon::CPolygon(double cx,double cy)
{
	min_x=min_y=1e6;
	max_x=max_y=-1e6;

	this->cx=cx;
	this->cy=cy;
	vx.reserve( 10 );
	vy.reserve( 10 );
}

/*---------------------------------------------------------------
							isLeft

	Return "true" if P is on the left side of P0-P1:
  ---------------------------------------------------------------*/
bool CPolygon::isLeft( double P0x,double P0y,double P1x,double P1y,double Px,double Py  ) const
{
	return ( ( (P1x - P0x) * (Py - P0y)
			- (Px - P0x) * (P1y - P0y) ) > 0);
}

/*---------------------------------------------------------------
							AddVertex

	 Add a new vertex to polygon:
  ---------------------------------------------------------------*/
void CPolygon::AddVertex(double x,double y)
{
	vx.push_back(x);
	vy.push_back(y);

	min_x=min(min_x,x);
	min_y=min(min_y,y);
	max_x=max(max_x,x);
	max_y=max(max_y,y);
}

/*---------------------------------------------------------------
							PointIntoPolygon

	 Check if a point is inside the polygon:
  ---------------------------------------------------------------*/
bool  CPolygon::PointIntoPolygon(double x,double y) const
{
	// If out of rectangle, there is no need of more checks:
	if ( x<min_x || x>max_x || y<min_y || y>max_y )
		return false;

	// http://softsurfer.com/Archive/algorithm_0103/algorithm_0103.htm
	// ------------------------------------------------------------------
	unsigned int    wn = 0;    // the winding number counter

	double	vi_x,vi_y;
	double	vi1_x,vi1_y;

	// loop through all edges of the polygon
	for (unsigned int i=0; i<vx.size(); i++)
	{   // edge from V[i] to V[i+1]
		vi_x = vx[i];
		vi_y = vy[i];

		if (i==(vx.size()-1))
		{
			vi1_x = vx[0];
			vi1_y = vy[0];
		}
		else
		{
			vi1_x = vx[i+1];
			vi1_y = vy[i+1];
		}


		if (vi_y <= y) {         // start y <= P.y
			if (vi1_y> y)      // an upward crossing
				if (isLeft( vi_x,vi_y, vi1_x,vi1_y, x,y) )  // P left of edge
					wn++;            // have a valid up intersect
		}
		else {                       // start y > P.y (no test needed)
			if (vi1_y<= y)     // a downward crossing
				if (!isLeft( vi_x,vi_y, vi1_x,vi1_y, x,y) )  // P right of edge
					wn--;            // have a valid down intersect
		}
	}

	return wn!=0;
}


/*---------------------------------------------------------------
					ComputeTangentAngNearPoint

  Computed the angle between the viewing angle and the
   closest edge in the polygon to the given point.
  ---------------------------------------------------------------*/
double CPolygon::ComputeTangentAngNearPoint(
		double x,
		double y,
		double view_phi,
		double min_dist)
{
	MRPT_TRY_START;

	int		idx_nearest = -1;
	double	d_min = 1e10;

	for (unsigned int i=0;i<vx.size();i++)
	{
		double Ax = x - vx[i];
		double Ay = y - vy[i];
		double d = Ax*Ax + Ay*Ay;

		if (d<d_min)
		{
			d_min = d;
			idx_nearest = i;
		}
	}

	if (d_min > min_dist) return M_PI/2;

	double	bigest_tangent = 0;
	double	edge_ang,tang_ang;
	double	p1x,p1y,p2x,p2y;


	// Compute edge_1 tangent:
	// ------------------------------
	p1x = vx[idx_nearest];
	p1y = vy[idx_nearest];
	if (((unsigned int)idx_nearest+1)<vx.size() )
	{
		p2x = vx[idx_nearest+1];
		p2y = vy[idx_nearest+1];
		if (p1x==p2x && p1y==p2y) return 0; // Don't delete it.
		edge_ang= atan2( p1y-p2y,p1x-p2x );
		tang_ang = fabs(edge_ang-view_phi);

		tang_ang=math::wrapToPi(tang_ang);

		bigest_tangent = max (bigest_tangent, tang_ang);
	}

	// Compute edge_2 tangent:
	// ------------------------------
	if ((unsigned int)(idx_nearest+1)<vx.size() )
	{
		p2x = vx[idx_nearest-1];
		p2y = vy[idx_nearest-1];
		if (p1x==p2x && p1y==p2y) return 0; // Don't delete it.
		edge_ang= atan2( p1y-p2y,p1x-p2x );
		tang_ang = fabs(edge_ang-view_phi);
		tang_ang = math::wrapToPi(tang_ang);

		bigest_tangent = max (bigest_tangent, tang_ang);
	}

	if (bigest_tangent>M_PI/2)
		bigest_tangent= M_PI - fabs(bigest_tangent);

	return bigest_tangent;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
	Implements the writing to a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPolygon::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 1;
	else
	{
		// The number of vertexs:
		uint32_t	n = (uint32_t)vx.size();

		// All elemental typed variables:
		out << n << max_x << max_y << min_x << min_y << cx << cy;

		// The vertexs arrays:
		if (n>0)
		{
			out.WriteBuffer( (void*)&vx[0],sizeof(vx[0])*n);
			out.WriteBuffer( (void*)&vy[0],sizeof(vy[0])*n);
		}
	}

}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPolygon::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:  // floats
		{
			// The number of vertexs:
			uint32_t	i,n;
			float		f;

			// All elemental typed variables:
			in >> n;
			in >> f; max_x=f;
			in >> f; max_y=f;
			in >> f; min_x=f;
			in >> f; min_y=f;
			in >> f; cx=f;
			in >> f; cy=f;

			vx.resize(n);
			vy.resize(n);

			// The vertexs arrays:
			for (i=0;i<n;i++) { in >> f; vx[i] = f;	}
			for (i=0;i<n;i++) { in >> f; vy[i] = f;	}
		} break;

	case 1:
		{
			// The number of vertexs:
			uint32_t	n;

			// All elemental typed variables:
			in >> n >> max_x >> max_y >> min_x >> min_y >> cx >> cy;

			vx.resize(n);
			vy.resize(n);

			// The vertexs arrays:
			if (n>0)
			{
				in.ReadBuffer( (void*)&vx[0],sizeof(vx[0])*n);
				in.ReadBuffer( (void*)&vy[0],sizeof(vy[0])*n);
			}
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
	Set all vertices at once, not to be used normally.
  ---------------------------------------------------------------*/
void  CPolygon::setAllVertices( const std::vector<double> &x, const std::vector<double> &y )
{
	ASSERT_(x.size()==y.size() && !x.empty());

	setAllVertices( x.size() ,&x[0],&y[0] );
}

/*---------------------------------------------------------------
	Set all vertices at once, not to be used normally.
  ---------------------------------------------------------------*/
void  CPolygon::setAllVertices( size_t nVertices, const double *xs, const double *ys )
{
	// Reserve:
	if (vx.capacity()<nVertices)
	{
		vx.reserve( nVertices + 20 );
		vy.reserve( nVertices + 20 );
	}

	// Resize:
	vx.resize(nVertices);
	vy.resize(nVertices);

	// Copy and update mins/maxs:
	min_x=min_y=1e6;
	max_x=max_y=-1e6;

	for (size_t i=0;i<nVertices;i++)
	{
		double x = xs[i];
		double y = ys[i];
		min_x=min(min_x,x);
		min_y=min(min_y,y);
		max_x=max(max_x,x);
		max_y=max(max_y,y);
		vx[i]=x;
		vy[i]=y;
	}

}


/*---------------------------------------------------------------
	Set all vertices at once, not to be used normally.
  ---------------------------------------------------------------*/
void  CPolygon::setAllVertices( size_t nVertices, const float *xs, const float *ys )
{
	// Reserve:
	if (vx.capacity()<nVertices)
	{
		vx.reserve( nVertices + 20 );
		vy.reserve( nVertices + 20 );
	}

	// Resize:
	vx.resize(nVertices);
	vy.resize(nVertices);

	// Copy and update mins/maxs:
	min_x=min_y=1e6;
	max_x=max_y=-1e6;

	for (size_t i=0;i<nVertices;i++)
	{
		double x = xs[i];
		double y = ys[i];
		min_x=min(min_x,x);
		min_y=min(min_y,y);
		max_x=max(max_x,x);
		max_y=max(max_y,y);
		vx[i]=x;
		vy[i]=y;
	}

}
