/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/core.h>

using namespace mrpt::utils;
using namespace mrpt::gui;
using namespace mrpt::vision;
using namespace std;

#include <mrpt/examples_config.h>
string   myDataDir( MRPT_EXAMPLES_BASE_DIRECTORY + string("imageConvolutionFFT/") );

void TestRectifyImages()
{
    CMRPTImage im;
    CMatrix cam_matrix(3,3), dist_coeff(1,4);

    im.loadFromFile("/home/paco/Documents/Images/calib/L01.bmp");
    cam_matrix(0,0) = 938.8868; cam_matrix(0,1) = 0;            cam_matrix(0,2) = 367.8682;
    cam_matrix(1,0) = 0;        cam_matrix(1,1) = 938.8868;     cam_matrix(1,2) = 303.2578;
    cam_matrix(2,0) = 0;        cam_matrix(2,1) = 0;            cam_matrix(2,2) = 1;

    dist_coeff(0,0) = -0.3202480;
    dist_coeff(0,1) = -0.3470451;
    dist_coeff(0,2) = 0;
    dist_coeff(0,3) = 0;

    im.rectifyImageInPlace( cam_matrix, dist_coeff );

    im.saveToFile("/home/paco/Documents/Images/calib/L01_REC.bmp");
} // end TestRectifyImages

// ------------------------------------------------------
//				TestCapture
// ------------------------------------------------------
void TestExtractFeatures()
{
	CFeatureExtraction	fExt;
	CFeatureList		featsHarris, featsKLT, featsSIFT_Hess, featsSIFT_Lowe, featsSIFT_Vedaldi;
	CMRPTImage			img;

	string the_img = myDataDir+string("test_image.jpg");

	if (!img.loadFromFile(the_img ))
	{
		cerr << "Cannot load " << the_img  << endl;
		return;
	}
	cout << "Loaded test image: " << the_img << endl;

	TImageROI ROI( 200, 400, 0, img.getHeight()-1 );
	CTicTac	tictac;

	cout << "Extracting Harris features... [f_harris.txt]";
	tictac.Tic();
	fExt.options.method = vision::CFeatureExtraction::extMethodHarris;
	fExt.detectFeatures( img, featsHarris, 0, 0, ROI );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsHarris.saveToTextFile("f_harris.txt");

	cout << "Computing SIFT descriptors only ... [f_harris+sift.txt]";
	tictac.Tic();
	fExt.options.SIFTOptions.implementation = vision::CFeatureExtraction::Hess;
	fExt.computeSiftDescriptors( img, featsHarris );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsHarris.saveToTextFile("f_harris+sift.txt");

	cout << "Extracting KLT features... [f_klt.txt]";
	tictac.Tic();
	fExt.options.method = vision::CFeatureExtraction::extMethodKLT;
	fExt.detectFeatures( img, featsKLT, 0, 0, ROI );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsKLT.saveToTextFile("f_klt.txt");

	cout << "Extracting SIFT features... [f_sift_hess.txt]";
	tictac.Tic();
	fExt.options.method = vision::CFeatureExtraction::extMethodSIFT;
	fExt.options.SIFTOptions.implementation = vision::CFeatureExtraction::Hess;
	fExt.detectFeatures( img, featsSIFT_Hess, 0, 0, ROI );
	cout << format("  %.03fms",tictac.Tac()*1000) << endl;
	featsSIFT_Hess.saveToTextFile("f_sift_hess.txt");
}

int main(int argc, char **argv)
{
	try
	{
		//TestExtractFeatures();
		TestRectifyImages();

		return 0;
	} catch (std::exception &e)
	{
		std::cout << "MRPT exception caught: " << e.what() << std::endl;
		return -1;
	}
	catch (...)
	{
		printf("Another exception!!");
		return -1;
	}

}
