/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */


#include <mrpt/hwdrivers/CHokuyoURG.h>
#include <mrpt/hwdrivers/CSerialPort.h>

IMPLEMENTS_GENERIC_SENSOR(CHokuyoURG,mrpt::hwdrivers)

using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::hwdrivers;
using namespace mrpt::system;
using namespace std;


/*-------------------------------------------------------------
						Constructor
-------------------------------------------------------------*/
CHokuyoURG::CHokuyoURG() :
	m_firstRange(44),
	m_lastRange(725),
	m_motorSpeed_rpm(0),
    m_sensorPose(0,0,0),
    m_rx_buffer(0),
    m_rx_idx(0),
    m_wr_idx(0),
	m_com_port(""),
	m_I_am_owner_serial_port(false),
	m_timeStartUI( 0 ),
	m_sensorLabel("Hokuyo")
{
    m_rx_buffer.reserve( 15000 );
}

/*-------------------------------------------------------------
					~CHokuyoURG
-------------------------------------------------------------*/
CHokuyoURG::~CHokuyoURG()
{
	if (m_stream)
	{
		turnOff();

		if (m_I_am_owner_serial_port)
			delete m_stream;
		m_stream = NULL;
	}
}

/*-------------------------------------------------------------
						doProcessSimple
-------------------------------------------------------------*/
void  CHokuyoURG::doProcessSimple(
	bool							&outThereIsObservation,
	mrpt::slam::CObservation2DRangeScan	&outObservation,
	bool							&hardwareError )
{
	outThereIsObservation	= false;
	hardwareError			= false;

	// Bound?
	if (!checkCOMisOpen())
	{
		m_timeStartUI = 0;
		hardwareError = true;
		return;
	}

	// Wait for a message:
	char			rcv_status0,rcv_status1;
	vector_byte		rcv_data(10000);
	int				rcv_dataLength;
	int				nRanges = m_lastRange-m_firstRange+1;
	int				expectedSize = nRanges*3 + 4;

    m_state = ssWorking;
	if (!receiveResponse( m_lastSentMeasCmd.c_str(), rcv_status0,rcv_status1, (char*)&rcv_data[0], rcv_dataLength ) )
	{
		// No new data
		return ;
	}

	// DEBUG
//	cout << "sent: " << m_lastSentMeasCmd << endl;
//	mrpt::system::vectorToBinaryFile( rcv_data, "rcv_data.txt");

	// DECODE:
	if (rcv_status0!='0' && rcv_status0!='9')
	{
		hardwareError = true;
		return;
	}

	// -----------------------------------------------
	//   Extract the observation:
	// -----------------------------------------------
	outObservation.timestamp = system::getCurrentTime();

	if ( expectedSize!=rcv_dataLength )
	{
		printf_debug("[CHokuyoURG::doProcess] ERROR: Expecting %u data bytes, received %u instead!\n",expectedSize,rcv_dataLength);
		hardwareError = true;
		return;
	}

	// Extract the timestamp of the sensor:
	uint32_t nowUI	=
		((rcv_data[0]-0x30) << 18) +
		((rcv_data[1]-0x30) << 12) +
		((rcv_data[2]-0x30) << 6) +
		(rcv_data[3]-0x30);

	uint32_t AtUI = 0;
	if( m_timeStartUI == 0 )
	{
		m_timeStartUI = nowUI;
		m_timeStartTT = mrpt::system::now();
	}
	else	AtUI	= nowUI - m_timeStartUI;

	mrpt::system::TTimeStamp AtDO	=  mrpt::system::secondsToTimestamp( AtUI * 1e-3 /* Board time is ms */ );
	outObservation.timestamp = m_timeStartTT + AtDO;

	// And the scan ranges:
	outObservation.rightToLeft = true;

	outObservation.aperture = nRanges *2*M_PI/m_sensor_info.scans_per_360deg;

	outObservation.maxRange	= m_sensor_info.d_max;
	outObservation.stdError = 0.010f;
	outObservation.sensorPose = m_sensorPose;
	outObservation.sensorLabel = m_sensorLabel;

	outObservation.scan.resize(nRanges);
	outObservation.validRange.resize(nRanges);
	char		*ptr = (char*) &rcv_data[4];
	for (int i=0;i<nRanges;i++)
	{
		int		b1 = (*ptr++)-0x30;
		int		b2 = (*ptr++)-0x30;
		int		b3 = (*ptr++)-0x30;

		int		range_mm = ( (b1 << 12) | (b2 << 6) | b3);

		outObservation.scan[i]			= range_mm * 0.001f;
		outObservation.validRange[i]	= range_mm>=20;
	}

	outThereIsObservation = true;
}

/*-------------------------------------------------------------
						loadConfig
-------------------------------------------------------------*/
void  CHokuyoURG::loadConfig(
	const mrpt::utils::CConfigFileBase &configSource,
	const std::string	  &iniSection )
{
	m_firstRange		= configSource.read_int(iniSection,"HOKUYO_firstRange",m_firstRange);
	m_lastRange			= configSource.read_int(iniSection,"HOKUYO_lastRange",m_lastRange);
	m_motorSpeed_rpm	= configSource.read_int(iniSection,"HOKUYO_motorSpeed_rpm",600);
	m_sensorLabel		= configSource.read_string(iniSection,"sensorLabel",m_sensorLabel);
	m_sensorPose.setFromValues(
		configSource.read_float(iniSection,"pose_x",0),
		configSource.read_float(iniSection,"pose_y",0),
		configSource.read_float(iniSection,"pose_z",0),
		DEG2RAD( configSource.read_float(iniSection,"pose_yaw",0) ),
		DEG2RAD( configSource.read_float(iniSection,"pose_pitch",0) ),
		DEG2RAD( configSource.read_float(iniSection,"pose_roll",0) )
		);

#ifdef MRPT_OS_WINDOWS
	m_com_port = configSource.read_string(iniSection, "COM_port_WIN", m_com_port, true );
#else
	m_com_port = configSource.read_string(iniSection, "COM_port_LIN", m_com_port, true );
#endif
}

/*-------------------------------------------------------------
						turnOn
-------------------------------------------------------------*/
bool  CHokuyoURG::turnOn()
{
	MRPT_TRY_START;

	// Bound?
	if (!checkCOMisOpen()) return false;

	// If we are over a serial link, set it up:
	CSerialPort* COM = dynamic_cast<CSerialPort*>(m_stream);
	if (COM!=NULL)
	{
		// It is a COM:
		COM->setConfig( 19200 );
		COM->setTimeouts(50,0,200,0,0);

		// Assure the laser is off and quiet:
		switchLaserOff();
		system::sleep(10);

		COM->purgeBuffers();
		system::sleep(10);

		COM->setConfig( 115200 );
		switchLaserOff();
		system::sleep(10);
		COM->purgeBuffers();
		system::sleep(10);
		COM->setConfig( 19200 );
	}

	// Enable SCIP 2.0
	enableSCIP20();

	if (COM!=NULL)
	{
		// Set 115200 baud rate:
		setHighBaudrate();
		enableSCIP20();
		COM->setConfig( 115200 );
	}

	if (!enableSCIP20()) return false;

	// Turn on the laser:
	if (!switchLaserOn()) return false;

	// Set the motor speed:
	if (m_motorSpeed_rpm)
		if (!setMotorSpeed( m_motorSpeed_rpm )) return false;

	// Display sensor information:
	if (!displaySensorInfo(&m_sensor_info )) return false;

	// Set for scanning angles:
	m_firstRange = m_sensor_info.scan_first;
	m_lastRange  = m_sensor_info.scan_last;

	if (!displayVersionInfo( )) return false;

	// Start!
	if (!startScanningMode()) return false;

	return true;

	MRPT_TRY_END;
}

/*-------------------------------------------------------------
						turnOff
-------------------------------------------------------------*/
bool  CHokuyoURG::turnOff()
{
	// Turn off the laser:
	if (!switchLaserOff()) return false;

	return true;
}

/*-------------------------------------------------------------
						setHighBaudrate
-------------------------------------------------------------*/
bool  CHokuyoURG::setHighBaudrate()
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[100];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::setHighBaudrate] Changing baudrate to 115200...");

	// Send command:
	os::strcpy(cmd,20, "SS115200\x0A");
	toWrite = 9;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}


	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");
	return true;
}


/*-------------------------------------------------------------
						assureBufferHasBytes
-------------------------------------------------------------*/
bool CHokuyoURG::assureBufferHasBytes(const int &nDesiredBytes)
{
    // Reset pointers?
    ASSERT_(m_wr_idx>=m_rx_idx);

    if (m_wr_idx==m_rx_idx)
    {
        m_wr_idx =
        m_rx_idx = 0;
    }

    if ( m_wr_idx-m_rx_idx < nDesiredBytes )  // Is it enough with bytes in the current buffer?
    {
        // NO: read more bytes:
        size_t nToRead = 1000;

        if ( m_rx_buffer.size()>40000 )
        {
            // Something bad is going on...
            std::cerr << "[CHokuyoURG] Clearing input buffer since it's growing too much..." << std::endl;
            m_wr_idx =
            m_rx_idx = 0;
        }

        // Assure buffer has room:
        m_rx_buffer.resize(m_wr_idx+nToRead);


        size_t nRead = m_stream->ReadBuffer(&m_rx_buffer[m_wr_idx],nToRead);
        m_wr_idx+=nRead;
        return (m_wr_idx-m_rx_idx)>=nDesiredBytes;
    }
    else return true;
}

/*-------------------------------------------------------------
						receiveResponse
-------------------------------------------------------------*/
bool  CHokuyoURG::receiveResponse(
		const char	*sentCmd_forEchoVerification,
		char	&rcv_status0,
		char	&rcv_status1,
		char	*rcv_data,
		int		&rcv_dataLength)
{
	if (!checkCOMisOpen()) return false;

    ASSERT_(sentCmd_forEchoVerification!=NULL);

	try
	{
		// Process response:
		// ---------------------------------

		// COMMAND ECHO ---------
		int i=0;
		int verifLen = strlen(sentCmd_forEchoVerification);

		do
		{
            if (!assureBufferHasBytes( verifLen-i ))
                return false;

            // If matches the echo, go on:
            if ( m_rx_buffer[m_rx_idx++]==sentCmd_forEchoVerification[i] )
                i++;
            else
                i=0;
		} while ( i<verifLen );

		//printf("VERIF.CMD OK!: %s", sentCmd_forEchoVerification);


		// Now, the status bytes:
        if (!assureBufferHasBytes( 2 ))
            return false;

		rcv_status0 = m_rx_buffer[m_rx_idx++];
		rcv_status1 = m_rx_buffer[m_rx_idx++];
        //printf("STATUS: %c%c\n", rcv_status0,rcv_status1);

        // In SCIP2.0, there is an additional sum char:
        if (rcv_status1!=0x0A)
		{
			// Yes, it is SCIP2.0
            if (!assureBufferHasBytes( 1 )) return false;
            m_rx_idx++;  // Ignore this byte
			//char sumStatus = m_rx_buffer[m_rx_idx++];
            //printf("STATUS SUM: %c\n",sumStatus);
		}
		else
		{
			// Continue, it seems a SCIP1.1 response...
		}

        // After the status bytes, there is a LF:
        if (!assureBufferHasBytes( 1 )) return false;
        char nextChar = m_rx_buffer[m_rx_idx++];
        if (nextChar!=0x0A)   return false;

        // -----------------------------------------------------------------------------
        // Now the data:
        // There's a problem here, we don't know in advance how many bytes to read,
        //  so rely on the serial port class implemented buffer and call many times
        //  the read method with only 1 byte each time:
        // -----------------------------------------------------------------------------
        bool lastWasLF=false;
        i=0;
        for (;;)
        {
            if (!assureBufferHasBytes(1)) return false;
            rcv_data[i] = m_rx_buffer[m_rx_idx++];

            i++;    // One more byte in the buffer

            // No data?
            if (i==1 && rcv_data[0]==0x0A)
            {
                rcv_dataLength = 0;
                return true;
            }

            // Is a LF?
            if (rcv_data[i-1]==0x0A)
            {
                if (!lastWasLF)
                {
                    // Discard SUM+LF
                    ASSERT_(i>=2);
                    i-=2;
                }
                else
                {
                    // Discard this last LF:
                    i--;

                    // Done!
                    rcv_data[i]=0;
                    //printf("RX %u:\n'%s'\n",i,rcv_data);

                    rcv_dataLength = i;
                    return true;
                }
                lastWasLF = true;
            }
            else lastWasLF = false;
        }

	}
	catch(std::exception &)
	{
		//cerr << e.what() << endl;
		return false;
	}
	catch(...)
	{

		return false;	// Serial port timeout,...
	}
}

/*-------------------------------------------------------------
						enableSCIP20
-------------------------------------------------------------*/
bool  CHokuyoURG::enableSCIP20()
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[100];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::enableSCIP20] Changing protocol to SCIP2.0...");

	// Send command:
	os::strcpy(cmd,20, "SCIP2.0\x0A");
	toWrite = 8;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}


	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}


	printf_debug("OK\n");
	return true;
}

/*-------------------------------------------------------------
						switchLaserOn
-------------------------------------------------------------*/
bool  CHokuyoURG::switchLaserOn()
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[100];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::switchLaserOn] Switching laser ON...");

	// Send command:
	os::strcpy(cmd,20, "BM\x0A");
	toWrite = 3;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}

	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");

	return true;
}

/*-------------------------------------------------------------
						switchLaserOff
-------------------------------------------------------------*/
bool  CHokuyoURG::switchLaserOff()
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[100];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::switchLaserOff] Switching laser OFF...");

	// Send command:
	os::strcpy(cmd,20, "QT\x0A");
	toWrite = 3;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}

	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");
	return true;
}

/*-------------------------------------------------------------
						setMotorSpeed
-------------------------------------------------------------*/
bool  CHokuyoURG::setMotorSpeed(int motoSpeed_rpm)
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[100];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::setMotorSpeed] Setting to %i rpm...",motoSpeed_rpm);

	// Send command:
	int		motorSpeedCode = (600 - motoSpeed_rpm) / 6;
	if (motorSpeedCode<0 || motorSpeedCode>10)
	{
		printf("ERROR! Motorspeed must be in the range 540-600 rpm\n");
		return false;
	}

	os::sprintf(cmd,20, "CR%02i\x0A",motorSpeedCode);
	toWrite = 5;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}

	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");
	return true;
}

/*-------------------------------------------------------------
						displayVersionInfo
-------------------------------------------------------------*/
bool  CHokuyoURG::displayVersionInfo( )
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[2000];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::displayVersionInfo] Asking info...");

	// Send command:
	os::sprintf(cmd,20, "VV\x0A");
	toWrite = 3;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}

	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");

	// PRINT:
	for (int i=0;i<rcv_dataLength;i++)
	{
		if (rcv_data[i]==';')
			rcv_data[i]='\n';
	}
	rcv_data[rcv_dataLength]=0;

	printf_debug("\n------------- HOKUYO Scanner: Version Information ------\n");
	printf_debug(rcv_data);
	printf_debug("-------------------------------------------------------\n\n");

	return true;
}

/*-------------------------------------------------------------
						displaySensorInfo
-------------------------------------------------------------*/
bool  CHokuyoURG::displaySensorInfo( TSensorInfo * out_data)
{
	char			cmd[20];
	char			rcv_status0,rcv_status1;
	char			rcv_data[1000];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::displaySensorInfo] Asking for info...");

	// Send command:
	os::sprintf(cmd,20, "PP\x0A");
	toWrite = 3;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}

	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");

	// PRINT:
	for (int i=0;i<rcv_dataLength;i++)
	{
		if (rcv_data[i]==';')
			rcv_data[i]='\n';
	}
	rcv_data[rcv_dataLength]=0;

	printf_debug("\n------------- HOKUYO Scanner: Product Information ------\n");
	printf_debug(rcv_data);
	printf_debug("-------------------------------------------------------\n\n");

	// Parse the data:
	if (out_data)
	{
		const char *ptr;

		if ( NULL != (ptr=strstr(rcv_data,"DMAX:")) )
				out_data->d_max = 0.001 * atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find DMAX" << endl;

		if ( NULL != (ptr=strstr(rcv_data,"DMIN:")) )
				out_data->d_min= 0.001 * atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find DMIN" << endl;

		if ( NULL != (ptr=strstr(rcv_data,"ARES:")) )
				out_data->scans_per_360deg= atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find ARES" << endl;

		if ( NULL != (ptr=strstr(rcv_data,"SCAN:")) )
				out_data->motor_speed_rpm= atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find SCAN" << endl;

		if ( NULL != (ptr=strstr(rcv_data,"AMIN:")) )
				out_data->scan_first= atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find AMIN" << endl;
		if ( NULL != (ptr=strstr(rcv_data,"AMAX:")) )
				out_data->scan_last= atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find AMAX" << endl;
		if ( NULL != (ptr=strstr(rcv_data,"AFRT:")) )
				out_data->scan_front= atoi( ptr+5 );
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find AFRT" << endl;

		if ( NULL != (ptr=strstr(rcv_data,"MODL:")) )
		{
			char aux[30];
			memcpy( aux, ptr+5, 8 );
			aux[8]='\0';
			out_data->model= aux;
		}
		else	cerr << "[CHokuyoURG::displayVersionInfo] Parse error: didn't find AFRT" << endl;

	}

	return true;
}

/*-------------------------------------------------------------
						startScanningMode
-------------------------------------------------------------*/
bool  CHokuyoURG::startScanningMode()
{
	char			cmd[51];
	char			rcv_status0,rcv_status1;
	char			rcv_data[3000];
	size_t			toWrite;
	int				rcv_dataLength;

	if (!checkCOMisOpen()) return false;

	printf_debug("[CHokuyoURG::startScanningMode] Starting scanning mode...");

	// Send command:
	os::sprintf(cmd,50, "MD%04u%04u01000\x0A", m_firstRange,m_lastRange);
	toWrite = 16;

	m_lastSentMeasCmd = cmd;

	m_stream->WriteBuffer(cmd,toWrite);

	// Receive response:
	if (!receiveResponse( cmd, rcv_status0,rcv_status1, rcv_data, rcv_dataLength ) )
	{
		printf("ERROR!\n");
		return false;
	}

	// DECODE:
	if (rcv_status0!='0')
	{
		printf("ERROR!\n");
		return false;
	}

	printf_debug("OK\n");
	return true;
}

/*-------------------------------------------------------------
						checkCOMisOpen
-------------------------------------------------------------*/
bool  CHokuyoURG::checkCOMisOpen()
{
	MRPT_TRY_START

	if (m_stream)
	{
		// Has the port been disconected (USB serial ports)??
		CSerialPort* COM = dynamic_cast<CSerialPort*>(m_stream);
		if (COM!=NULL)
		{
			if (COM->isOpen())
				return true;

			// It has been disconnected... try to reconnect:
			cerr << "[CHokuyoURG] Serial port connection lost! Trying to reconnect..." << endl;

			try
			{
				COM->open();
				// OK, reconfigure the laser:
				turnOn();
				return true;
			}
			catch (...)
			{
				// Not yet..
				return false;
			}
		}
		else
		{
			return true;		// Assume OK
		}
	}
	else
	{
		if (m_com_port.empty())
		{
			THROW_EXCEPTION("No stream bound to the laser nor COM serial port name provided in 'm_com_port'");
		}

		// Try to open the serial port:
		CSerialPort		*theCOM = new CSerialPort(m_com_port, true);

		if (!theCOM->isOpen())
		{
			cerr << "[CHokuyoURG] Cannot open serial port '" << m_com_port << "'" << endl;
			delete theCOM;
			return false;
		}

		// Bind:
		bindIO( theCOM );

		m_I_am_owner_serial_port=true;

		return true;
	}
	MRPT_TRY_END
}

/*-------------------------------------------------------------
						initialize
-------------------------------------------------------------*/
void CHokuyoURG::initialize()
{
	if (!checkCOMisOpen()) return;

	if (!turnOn())
	{
		cerr << "[CHokuyoURG::initialize] Error initializing HOKUYO scanner" << endl;
		return;
	}

}


/*-------------------------------------------------------------
						purgeBuffers
-------------------------------------------------------------*/
void CHokuyoURG::purgeBuffers()
{
	if (!checkCOMisOpen()) return;

	CSerialPort* COM = dynamic_cast<CSerialPort*>(m_stream);
	if (COM!=NULL)   
	{
		COM->purgeBuffers();
	}
}


