/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/hwdrivers/CGenericSensor.h>

#include <mrpt/slam/CAction.h>
#include <mrpt/slam/CObservation.h>

using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::system;
using namespace mrpt::hwdrivers;
using namespace std;

map< std::string , const TSensorClassId *>	CGenericSensor::m_knownClasses;


/*-------------------------------------------------------------
						Constructor
-------------------------------------------------------------*/
CGenericSensor::CGenericSensor() :
    m_state( ssInitializing )
{
}

/*-------------------------------------------------------------
						Destructor
-------------------------------------------------------------*/
CGenericSensor::~CGenericSensor()
{
	// Free objects in list, if any:
	m_objList.clear();
}

/*-------------------------------------------------------------
						appendObservation
-------------------------------------------------------------*/
void CGenericSensor::appendObservation( const mrpt::utils::CSerializablePtr &obj)
{
	// It must be a CObservation or a CAction!
	TTimeStamp	timestamp;

	if ( obj->GetRuntimeClass()->derivedFrom( CLASS_ID(CAction) ) )
	{
		timestamp = CActionPtr(obj)->timestamp;
	}
	else
	if ( obj->GetRuntimeClass()->derivedFrom( CLASS_ID(CObservation) ) )
	{
		timestamp = CObservationPtr(obj)->timestamp;
	}
	else THROW_EXCEPTION("Passed object must be CObservation.");

	// OK, add:
	synch::CCriticalSectionLocker	lock( & m_csObjList );
	m_objList.insert( TListObsPair(timestamp, obj) );
}


/*-------------------------------------------------------------
						getObservations
-------------------------------------------------------------*/
void CGenericSensor::getObservations( TListObservations	&lstObjects )
{
	synch::CCriticalSectionLocker	lock( & m_csObjList );
	lstObjects = m_objList;
	m_objList.clear();			// Memory of objects will be freed by invoker.
}

/*-------------------------------------------------------------

						createSensor

-------------------------------------------------------------*/
CGenericSensor* CGenericSensor::createSensor(std::string &className)
{
	std::map< std::string , const TSensorClassId *>::iterator it=m_knownClasses.find(className);
	return it==m_knownClasses.end() ? NULL : it->second->ptrCreateObject();
}


/*-------------------------------------------------------------
						registerClass
-------------------------------------------------------------*/
void CGenericSensor::registerClass(const TSensorClassId* pNewClass)
{
	m_knownClasses[ pNewClass->className ] = pNewClass;
}

