/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 

#include <mrpt/vision/CGaussianConvolutionKernel.h>
#include <mrpt/math/CVectorTemplate.h>

using namespace mrpt::utils;
using namespace mrpt::vision;

// The value of the Gaussian/DOF response such as it can be truncated:
#define	GAUSS_TAIL_TRUNCATION	0.01f


/*-------------------------------------------------------------
						Constructor
-------------------------------------------------------------*/
CGaussianConvolutionKernel::CGaussianConvolutionKernel(float sigma, bool isDOG )
{
	buildPrecomputedKernels( sigma, isDOG );
}

/*-------------------------------------------------------------
					buildPrecomputedKernels
-------------------------------------------------------------*/
void  CGaussianConvolutionKernel::buildPrecomputedKernels( float sigma, bool isDOG )
{
	int								i,len;
	float							v;
	std::vector<float>::iterator	it;
	float							max_DOG	  = (float) (sigma*exp(-0.5f));
	float							accum;

	// Already computed?
	if ( fabs(sigma - m_sigma)<0.10f )
		return;

	// Re-compute the tables:
	ASSERT_(sigma>=0);

	m_sigma = sigma;
	m_isDOG = isDOG;

	if (!m_isDOG)
	{
		// Compute the Gaussian kernel:
		// --------------------------------------
		// First, find out the required size of the window:
		len = -1;
		v = 0;
		do
		{
			len++;
			v = exp( - 0.5f * square( ((float)len) / sigma) );
		} while (v>GAUSS_TAIL_TRUNCATION);
		m_kernel.resize(2*len+1);	// [-len,...,0,...,+len]

		// Fill out the vector:
		accum = 0;		// Accumulator
		for (i=-len,it=m_kernel.begin();i<=len;i++,it++)
		{
			*it = exp( - 0.5f * square( ((float)i) / sigma) );
			accum += *it;
		}

		// Normalize:
		accum = 1.0f / accum;
		for (i=-len,it=m_kernel.begin();i<=len;i++,it++)
			*it *= accum;
	}
	else
	{
		// Compute the DOG kernel:
		// --------------------------------------
		// First, find out the required size of the window:
		len = -1;
		v = 0;
		do
		{
			len++;
			v = -len * exp( - 0.5f * square( ((float)len) / sigma) );
		} while ( len<1 || fabs(v/max_DOG) > GAUSS_TAIL_TRUNCATION );
		m_kernel.resize(2*len+1);	// [-len,...,0,...,+len]

		// Fill out the vector:
		accum = 0;		// Accumulator
		for (i=-len,it=m_kernel.begin();i<=len;i++,it++)
		{
			*it = -i * exp( - 0.5f * square( ((float)i) / sigma) );	// Normalized;
			accum -= i * *it;
		}

		// Normalize:
		accum = 1.0f / accum;
		for (i=-len,it=m_kernel.begin();i<=len;i++,it++)
			*it *= accum;
	}

	// system::vectorToTextFile( m_kernel, "debug_gaussKernel.txt");
}


/*-------------------------------------------------------------
						getKernelSize
-------------------------------------------------------------*/
unsigned int  CGaussianConvolutionKernel::getKernelSize()
{
	return (unsigned int )m_kernel.size();
}

/*-------------------------------------------------------------
						changeSigma
-------------------------------------------------------------*/
void  CGaussianConvolutionKernel::changeSigma( float sigma )
{
	buildPrecomputedKernels( sigma, m_isDOG );
}
