/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/vision/CFeature.h>
#include <mrpt/math/CVectorTemplate.h>

using namespace mrpt;
using namespace mrpt::vision;
using namespace mrpt::system;

IMPLEMENTS_SERIALIZABLE(CFeature, CSerializable, mrpt::vision)
 
void  CFeature::writeToStream(CStream &out,int *version) const
{
	THROW_EXCEPTION("Not implemented yet");
}
 
void  CFeature::readFromStream(CStream &in,int version)
{
	THROW_EXCEPTION("Not implemented yet");
}

/****************************************************
				Class CFEATURE
*****************************************************/
// CONSTRUCTOR
CFeature::CFeature(): x(0.0f), y(0.0f), ID(0), patchSize(21), type(featNotDefined), 
	KLT_status(statusKLT_IDLE), KLT_val(0.0), 
	orientation(0.0), scale(0.0), descriptorSIFT(0), descriptorSURF(0), hasDescriptorSIFT(false), hasDescriptorSURF(false)
{}

// --------------------------------------------------
// descriptorSIFTDistanceTo
// --------------------------------------------------
float CFeature::descriptorSIFTDistanceTo( const CFeature &oFeature ) const
{
	ASSERT_( this->descriptorSIFT.size() == oFeature.descriptorSIFT.size() );

	float dist = 0.0f;
	std::vector<unsigned char>::const_iterator itDesc1, itDesc2;
	for( itDesc1 = this->descriptorSIFT.begin(), itDesc2 = oFeature.descriptorSIFT.begin();
		 itDesc1 != this->descriptorSIFT.end();
		 itDesc1++, itDesc2++ )
	{
		dist += square(*itDesc1 - *itDesc2);
	}
	return sqrt(dist);
} // end descriptorSIFTDistanceTo

// --------------------------------------------------
// descriptorSURFDistanceTo
// --------------------------------------------------
float CFeature::descriptorSURFDistanceTo( const CFeature &oFeature ) const
{
	ASSERT_( this->descriptorSURF.size() == oFeature.descriptorSURF.size() );

	float dist = 0.0f;
	std::vector<unsigned char>::const_iterator itDesc1, itDesc2;
	for( itDesc1 = this->descriptorSURF.begin(), itDesc2 = oFeature.descriptorSURF.begin();
		 itDesc1 != this->descriptorSURF.end();
		 itDesc1++, itDesc2++ )
	{
		dist += square(*itDesc1 - *itDesc2);
	}
	return sqrt(dist);
} // end descriptorSURFDistanceTo

/****************************************************
			   Class CFEATURELIST
*****************************************************/
// --------------------------------------------------
// CONSTRUCTOR
// --------------------------------------------------
CFeatureList::CFeatureList()
{} //end constructor

CFeatureList::CFeatureList(const CFeatureList &o)
{
	THROW_EXCEPTION("Call to copy constructor in CFeatureList");
} //end constructor
CFeatureList& CFeatureList::operator=(const CFeatureList &o)
{
	THROW_EXCEPTION("Call to copy operator in CFeatureList");
}

// --------------------------------------------------
// DESTRUCTOR
// --------------------------------------------------
CFeatureList::~CFeatureList()
{
	/*CFeatureList::iterator it;
	for(it=begin(); it!=end(); it++)
	{
		if( (*it)->numRefs == 1)
			delete *it;
		else
			(*it)->numRefs--;
	}*/
} // end destructor

// --------------------------------------------------
// insert
// --------------------------------------------------
void CFeatureList::insert( const CFeaturePtr &feat )
{
	//feat->numRefs++;
	push_back( feat );
} // end insert( CFeature *feat )

// --------------------------------------------------
// saveToTextFile
// --------------------------------------------------
// FORMAT: ID type x y orientation scale [descriptorSIFT] [descriptorSURF] KLT_status KLT_val
void CFeatureList::saveToTextFile( const std::string &filename, bool APPEND )
{
	FILE *f;
	
	if(!APPEND)
		f = os::fopen( filename.c_str(), "wt" );
	else
		f = os::fopen( filename.c_str(), "at" );

	if(!f) return;

	CFeatureList::iterator it;
	for( it = this->begin(); it != this->end(); it++ )
	{
		os::fprintf( f, "%d %d %.3f %.3f ", (unsigned int)(*it)->ID, (int)(*it)->get_type(), (*it)->x, (*it)->y );
		os::fprintf( f, "%.2f %.2f ", (*it)->orientation, (*it)->scale ); 

		if( (*it)->hasDescriptorSIFT )
			for( unsigned int k = 0; k < (*it)->descriptorSIFT.size(); k++ )
				os::fprintf( f, "%d ", (*it)->descriptorSIFT[k]);

		if( (*it)->hasDescriptorSURF )
			for( unsigned int k = 0; k < (*it)->descriptorSURF.size(); k++ )
				os::fprintf( f, "%d ", (*it)->descriptorSURF[k]);

		os::fprintf( f, "%d %.3f\n", (int)(*it)->KLT_status, (*it)->KLT_val );
	} // end for
	os::fclose( f );

} // end saveToTextFile

// --------------------------------------------------
// getByID()
// --------------------------------------------------
CFeaturePtr CFeatureList::getByID( TFeatureID ID ) const
{
	for( CFeatureList::const_iterator it = begin(); it != end(); it++ )
	{
		if( (*it)->ID == ID )
			return (*it);
	}
	return CFeaturePtr();
} // end getByID

// --------------------------------------------------
// getMaxID()
// --------------------------------------------------
TFeatureID CFeatureList::getMaxID()
{
	MRPT_TRY_START;
	ASSERT_( size() > 0 );

	vision::TFeatureID maxID	= (*begin())->ID;
	CFeatureList::iterator itList;

	for(itList = begin(); itList != end(); itList++)
	{
		if( (*itList)->ID > maxID )
			maxID = (*itList)->ID;
	} // end for

	return maxID;

	MRPT_TRY_END;

} // end getMaxID()

// --------------------------------------------------
// remove
// --------------------------------------------------
CFeatureList::iterator CFeatureList::remove( CFeatureList::iterator itFeat )
{
	//if( (*itFeat)->numRefs == 1 )
	//	delete (*itFeat);
	//else
	//	(*itFeat)->numRefs--;

	return erase( itFeat );

} // end remove


/****************************************************
		  Class CMATCHEDFEATUREKLT
*****************************************************/
// --------------------------------------------------
// CONSTRUCTOR
// --------------------------------------------------
CMatchedFeatureList::CMatchedFeatureList(){}

CMatchedFeatureList::CMatchedFeatureList(const CMatchedFeatureList &o)
{
	THROW_EXCEPTION("Call to copy constructor in CMatchedFeatureList");
} //end constructor
CMatchedFeatureList& CMatchedFeatureList::operator=(const CMatchedFeatureList &o)
{
	THROW_EXCEPTION("Call to copy operator in CMatchedFeatureList");
}


// --------------------------------------------------
// DESTRUCTOR
// --------------------------------------------------
CMatchedFeatureList::~CMatchedFeatureList()
{
	//CMatchedFeatureList::iterator it;
	//for(it=begin(); it!=end(); it++)
	//{
	//	if((*it->first).numRefs == 1)
	//		delete it->first;
	//	else
	//		(*it->first).numRefs--;

	//	if((*it->second).numRefs == 1)
	//		delete it->second;
	//	else
	//		(*it->second).numRefs--;
	//}
} // end destructor

// --------------------------------------------------
// insert
// --------------------------------------------------
void CMatchedFeatureList::insert( std::pair<CFeaturePtr,CFeaturePtr> mFeats )
{
	//mFeats.first->numRefs++;
	//mFeats.second->numRefs++;

	push_back( mFeats );
} // end insert( std::pair<CFeature*,CFeature*> mFeats )

// --------------------------------------------------
// remove
// --------------------------------------------------
CMatchedFeatureList::iterator CMatchedFeatureList::remove( CMatchedFeatureList::iterator pMatchedFeats )
{
	//if( (pMatchedFeats->first)->numRefs == 1 )
	//	delete pMatchedFeats->first;
	//else
	//	(pMatchedFeats->first)->numRefs--;

	//if( (pMatchedFeats->second)->numRefs == 1 )
	//	delete pMatchedFeats->second;
	//else
	//	(pMatchedFeats->second)->numRefs--;

	return erase( pMatchedFeats );						// Erase the match
} // end remove( std::pair<CFeature*,CFeature*> mFeats )

// --------------------------------------------------
// saveToTextFile
// --------------------------------------------------
void CMatchedFeatureList::saveToTextFile(const std::string &filename)
{
	// OUTPUT FORMAT: ID_1 x_1 y_1 ID_2 x_2 y_2

	FILE *f = os::fopen( filename.c_str(), "wt" );
	if(!f) return;

	CMatchedFeatureList::iterator it;
	for( it = this->begin(); it != this->end(); it++ )
	{
		os::fprintf( f, "%d %.3f %.3f %d %.3f %.3f\n",
			(unsigned int)(*it->first).ID, (*it->first).x, (*it->first).y,
			(unsigned int)(*it->second).ID, (*it->second).x, (*it->second).y);

	} // end for
	os::fclose( f );
}

