/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/vision/CCamModel.h>

using namespace mrpt;
using namespace mrpt::vision;


/**	Constructor
*/
CCamModel::CCamModel(const std::string &file) :
	cam()
{
	utils::CConfigFile fich(file);

#define INVALID 666666
    //Lectura de los parametros de la camara
	cam.d			=	fich.read_float("CAMERA","d",INVALID);
	if (cam.d==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.nrows		=	fich.read_int("CAMERA","nrows",INVALID);
	if (cam.nrows==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.ncols		=	fich.read_int("CAMERA","ncols",INVALID);
	if (cam.ncols==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.cx			=	fich.read_float("CAMERA","cx",INVALID);
	if (cam.cx==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.cy			=	fich.read_float("CAMERA","cy",INVALID);
	if (cam.cy==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.k1			=	fich.read_float("CAMERA","k1",INVALID);
	if (cam.k1==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.k2			=	fich.read_float("CAMERA","k2",INVALID);
	if (cam.k2==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
	cam.f			=	fich.read_float("CAMERA","f",INVALID);
	if (cam.f==INVALID) THROW_EXCEPTION("MonoSlamMotionModel constructor: Initial value not found");
}

/**********************************************************************************************************************/
void  CCamModel::jacob_undistor_fm(const math::CVectorFloat &uvd, math::CMatrixFloat &J_undist)
{

	float xd=(uvd[0]-cam.cx)*cam.d;
    float yd=(uvd[1]-cam.cy)*cam.d;
	float rd2=xd*xd+yd*yd;
	float rd4=rd2*rd2;

	float uu_ud=(1+cam.k1*rd2+cam.k2*rd4)+(uvd[0]-cam.cx)*(cam.k1+2*cam.k2*rd2)*(2*(uvd[0]-cam.cx)*cam.d*cam.d);
	float vu_vd=(1+cam.k1*rd2+cam.k2*rd4)+(uvd[1]-cam.cy)*(cam.k1+2*cam.k2*rd2)*(2*(uvd[1]-cam.cy)*cam.d*cam.d);

	float uu_vd=(uvd[0]-cam.cx)*(cam.k1+2*cam.k2*rd2)*(2*(uvd[1]-cam.cy)*cam.d*cam.d);
	float vu_ud=(uvd[1]-cam.cy)*(cam.k1+2*cam.k2*rd2)*(2*(uvd[0]-cam.cx)*cam.d*cam.d);

	J_undist.setSize(2,2);
	J_undist(0,0)=uu_ud;	J_undist(0,1)=uu_vd;
	J_undist(1,0)=vu_ud;	J_undist(1,1)=vu_vd;

}
/******************************************************************************************************************************/
math::CMatrixFloat  CCamModel::jacob_undistor(float &col , float &row )
{
	float Cx = cam.cx;
	float Cy = cam.cy;
	float k1 = cam.k1;
	float k2 = cam.k2;
	float dx = cam.d;
	float dy = cam.d;

	float ud = col;
	float vd = row;
	float xd = (ud-Cx)*dx;
	float yd = (vd-Cy)*dy;

	float rd2 = xd*xd + yd*yd;
	float rd4 = rd2 * rd2;

	math::CMatrixFloat J_undistor(2,2);

	J_undistor(0,0) = (1+k1*rd2+k2*rd4) + (ud-Cx) * (k1+2*k2*rd2) * (2*(ud-Cx)*dx*dx);
	J_undistor(1,1) = (1+k1*rd2+k2*rd4) + (vd-Cy) * (k1+2*k2*rd2) * (2*(vd-Cy)*dy*dy);
	J_undistor(0,1) = (ud-Cx) * (k1+2*k2*rd2) * (2*(vd-Cy)*dy*dy);
	J_undistor(1,0) = (vd-Cy) * (k1+2*k2*rd2) * (2*(ud-Cx)*dx*dx);

	return J_undistor;
}
/**********************************************************************************************************************/

void  CCamModel::distort_a_point(const math::CVectorFloat &uvu, math::CVectorFloat &uvd)
{
	float xu = (uvu[0] - cam.cx) * cam.d;
	float yu = (uvu[1] - cam.cy) * cam.d;

    float ru = sqrt(xu*xu + yu*yu);
	float rd = ru / (1 + cam.k1 * ru*ru + cam.k2 * ru*ru*ru*ru);	//initial value for iteration

  //Newton-Rapson. 100 iterations
	float f=0.0, f_p=1.0;;
	for (int k=0 ; k<100 ; k++)
	{
        f = rd + cam.k1 * rd*rd*rd + cam.k2 * rd*rd*rd*rd*rd - ru;
		f_p= 1+ 3 * cam.k1 * rd*rd + 5 * cam.k2 * rd*rd*rd*rd;
		rd = rd - f / f_p;
	}

    float D = 1 + cam.k1 * rd*rd + cam.k2*rd*rd*rd*rd;

	uvd[0] = ( cam.cx + (xu/D) / cam.d );
	uvd[1] = ( cam.cy + (yu/D) / cam.d );
}
/*************************************************************************************************************************/
void  CCamModel::undistort_point(float &col,float &row)
{
	//Elimina la distorsion de un punto en pxl mediante los paramentros de calibracion de la camara (pag.52 del libro de Javier G.)
	//Viene el modelo de distorsion mejor en pag.3 "A visual compass based on SLAM"
	float Cx = cam.cx;
	float Cy = cam.cy;
	float k1 = cam.k1;
	float k2 = cam.k2;
	float dx = cam.d;
	float dy = cam.d;

	float ud = col;
	float vd = row;
	float rd = sqrt( (dx*(ud-Cx))*(dx*(ud-Cx)) + (dy*(vd-Cy))*(dy*(vd-Cy)) );

	float uu = Cx + ( ud - Cx )*( 1 + k1*rd*rd + k2*rd*rd*rd*rd );
	float vu = Cy + ( vd - Cy )*( 1 + k1*rd*rd + k2*rd*rd*rd*rd );

	col = uu;
	row = vu;
}
/*************************************************************************************************************************/
