/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/slam/CSensFrameProbSequence.h>

using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::poses;
using namespace std;

IMPLEMENTS_SERIALIZABLE(CSensFrameProbSequence, CSerializable,mrpt::slam)

/*---------------------------------------------------------------
						Constructor
  ---------------------------------------------------------------*/
CSensFrameProbSequence::CSensFrameProbSequence() : m_posesObsPairs()
{
}

/*---------------------------------------------------------------
					Copy
  ---------------------------------------------------------------*/
CSensFrameProbSequence::CSensFrameProbSequence( const CSensFrameProbSequence &o )  :
	m_posesObsPairs( o.m_posesObsPairs )
{
	for_each( m_posesObsPairs.begin(), m_posesObsPairs.end(), ObjectPairMakeUnique() );
}

/*---------------------------------------------------------------
					Copy
  ---------------------------------------------------------------*/
CSensFrameProbSequence & CSensFrameProbSequence::operator = ( const CSensFrameProbSequence& o)
{
	MRPT_TRY_START;

	//TPosePDFSensFramePair	pair;

	if (this == &o) return *this;		// It may be used sometimes

	m_posesObsPairs = o.m_posesObsPairs;
	for_each( m_posesObsPairs.begin(), m_posesObsPairs.end(), ObjectPairMakeUnique() );

	return *this;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						size
  ---------------------------------------------------------------*/
size_t CSensFrameProbSequence::size() const
{
	return m_posesObsPairs.size();
}

/*---------------------------------------------------------------
						clear
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::clear()
{
	m_posesObsPairs.clear();
}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
CSensFrameProbSequence::~CSensFrameProbSequence()
{
	clear();
}

/*---------------------------------------------------------------
							get const
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::get(
		size_t	        index,
		CPose3DPDFPtr &out_posePDF,
		CSensoryFramePtr &out_SF ) const
{
	if (index>=m_posesObsPairs.size())
		THROW_EXCEPTION("Index out of bounds");

	out_posePDF	= m_posesObsPairs[index].first;
	out_SF		= m_posesObsPairs[index].second;
}

/*---------------------------------------------------------------
						remove
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::remove(size_t index)
{
	MRPT_TRY_START;

	if (index>=m_posesObsPairs.size())
		THROW_EXCEPTION("Index out of bounds");

	m_posesObsPairs.erase( m_posesObsPairs.begin() + index );

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						set
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::set(
	size_t	index,
	const CPose3DPDFPtr &in_posePDF,
	const CSensoryFramePtr & in_SF )
{
	MRPT_TRY_START;

	if (index>=m_posesObsPairs.size())
		THROW_EXCEPTION("Index out of bounds");

	if (in_posePDF) m_posesObsPairs[index].first = in_posePDF;
	if (in_SF) 		m_posesObsPairs[index].second = in_SF;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						set 2D
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::set(
	size_t	index,
	const CPosePDFPtr &in_posePDF,
	const CSensoryFramePtr &in_SF )
{
	MRPT_TRY_START;

	if (index>=m_posesObsPairs.size())
		THROW_EXCEPTION("Index out of bounds");

	if (in_posePDF) 	m_posesObsPairs[index].first = CPose3DPDFPtr( CPose3DPDF::createFrom2D( *in_posePDF ) );
	if (in_SF) 			m_posesObsPairs[index].second = in_SF;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						insert
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::insert(
	const CPose3DPDFPtr &in_posePDF,
	const CSensoryFramePtr &in_SF )
{
	MRPT_TRY_START;

	TPosePDFSensFramePair	pair;

	pair.second  = in_SF;
	pair.first	 = in_posePDF;

	m_posesObsPairs.push_back( pair );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						insert
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::insert( const CPose3DPDF *in_posePDF, const CSensoryFrame &in_SF )
{
	MRPT_TRY_START;

	TPosePDFSensFramePair	pair;

	pair.second  = CSensoryFramePtr( new CSensoryFrame(in_SF) );
	pair.first	 = CPose3DPDFPtr( static_cast<CPose3DPDF*>(in_posePDF->duplicate()) );

	m_posesObsPairs.push_back( pair );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						insert
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::insert( const CPosePDF *in_posePDF, const CSensoryFrame &in_SF )
{
	MRPT_TRY_START;

	TPosePDFSensFramePair	pair;

	pair.second  = CSensoryFramePtr( new CSensoryFrame(in_SF) );
	pair.first	 = CPose3DPDFPtr( static_cast<CPose3DPDF*>(in_posePDF->duplicate()) );

	m_posesObsPairs.push_back( pair );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						insert
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::insert( const CPosePDF *in_posePDF, const CSensoryFramePtr &in_SF )
{
	MRPT_TRY_START;

	TPosePDFSensFramePair	pair;

	pair.second  = in_SF;
	pair.first	 = CPose3DPDFPtr( static_cast<CPose3DPDF*>(in_posePDF->duplicate()) );

	m_posesObsPairs.push_back( pair );

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						insert  2D
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::insert(
	const CPosePDFPtr &in_posePDF,
	const CSensoryFramePtr &in_SF )
{
	insert( CPose3DPDFPtr( CPose3DPDF::createFrom2D( *in_posePDF ) ) ,in_SF);
}

/*---------------------------------------------------------------
					writeToStream
	Implements the writing to a CStream capability of
	  CSerializable objects
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		uint32_t		i,n;
		n = m_posesObsPairs.size();
		out << n;
		for (i=0;i<n;i++)
			out << *m_posesObsPairs[i].first << *m_posesObsPairs[i].second;
	}
}

/*---------------------------------------------------------------
					readFromStream
  ---------------------------------------------------------------*/
void  CSensFrameProbSequence::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 1:
		{
			uint32_t	i,n;
			clear();
			in >> n;
			m_posesObsPairs.resize(n);
			for (i=0;i<n;i++)
				in >> m_posesObsPairs[i].first >> m_posesObsPairs[i].second;
		} break;
	case 0:
		{
			// There are 2D poses PDF instead of 3D: transform them:
			uint32_t	i,n;
			clear();
			in >> n;
			m_posesObsPairs.resize(n);
			for (i=0;i<n;i++)
			{
				CPosePDFPtr aux2Dpose;
				in >> aux2Dpose >> m_posesObsPairs[i].second;
				m_posesObsPairs[i].first = CPose3DPDFPtr( CPose3DPDF::createFrom2D( *aux2Dpose ) );
			}
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}


/*---------------------------------------------------------------
					changeCoordinatesOrigin
  ---------------------------------------------------------------*/
void CSensFrameProbSequence::changeCoordinatesOrigin( const CPose3D  &newOrigin )
{
	for (TPosePDFSensFramePairList::iterator it=m_posesObsPairs.begin(); it!=m_posesObsPairs.end(); it++)
		it->first->changeCoordinatesReference(newOrigin);
}
