/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/slam/CRawlogXXL.h>
#include <mrpt/utils/CFileInputStream.h>

using namespace mrpt;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::utils;

IMPLEMENTS_SERIALIZABLE(CRawlogXXL, CSerializable,mrpt::slam)

// The parentheses are to force VC to detect this is NOT a macro!!
const size_t CRawlogXXL::INVALID_POS =  (std::numeric_limits<size_t>::max) ();


/*---------------------------------------------------------------
					Default constructor
  ---------------------------------------------------------------*/
CRawlogXXL::CRawlogXXL() :
	m_seqOfActObs(),
	m_tempFile(),
	m_objectsInMemory(0)
{
	m_tempFile.open( mrpt::system::getTempFileName() );
	ASSERT_(m_tempFile.fileOpenCorrectly());
}

/*---------------------------------------------------------------
					isOk
  ---------------------------------------------------------------*/
bool CRawlogXXL::isOK()
{
	return m_tempFile.fileOpenCorrectly();
}

/*---------------------------------------------------------------
					Destructor
  ---------------------------------------------------------------*/
CRawlogXXL::~CRawlogXXL()
{
	clear();
}

/*---------------------------------------------------------------
						clear
  ---------------------------------------------------------------*/
void  CRawlogXXL::clear()
{
	// Since we use smart pointers, there's no need to manually delete object...
	m_seqOfActObs.clear();
	m_objectsInMemory = 0;
}

/*---------------------------------------------------------------
						clearWithoutDelete
  ---------------------------------------------------------------*/
void  CRawlogXXL::clearWithoutDelete()
{
	m_seqOfActObs.clear();
	m_objectsInMemory = 0;
}


/*---------------------------------------------------------------
						addObservation
  ---------------------------------------------------------------*/
void  CRawlogXXL::addObservations(
		CSensoryFrame		&observations )
{
	TObjectElementXXL	d;
	d.obj			= CSerializablePtr( observations.duplicate() );
	d.lastAccess	= mrpt::system::now();

	m_seqOfActObs.push_back( d );
	m_objectsInMemory++;

	swapEntriesToDiskIfRequired( mrpt::system::now() );
}

/*---------------------------------------------------------------
					addActions
  ---------------------------------------------------------------*/
void  CRawlogXXL::addActions(
		CActionCollection		&actions )
{
	TObjectElementXXL	d;
	d.obj			= CSerializablePtr( actions.duplicate() );
	d.lastAccess	= mrpt::system::now();

	m_seqOfActObs.push_back( d );
	m_objectsInMemory++;

	swapEntriesToDiskIfRequired( mrpt::system::now() );
}

/*---------------------------------------------------------------
					addActionsMemoryReference
  ---------------------------------------------------------------*/
void  CRawlogXXL::addActionsMemoryReference( const CActionCollectionPtr &action )
{
	TObjectElementXXL	d;
	d.obj			= action;
	d.lastAccess	= mrpt::system::now();

	m_seqOfActObs.push_back( d );
	m_objectsInMemory++;

	swapEntriesToDiskIfRequired( mrpt::system::now() );
}

/*---------------------------------------------------------------
					addObservationsMemoryReference
  ---------------------------------------------------------------*/
void  CRawlogXXL::addObservationsMemoryReference( const CSensoryFramePtr &observations )
{
	TObjectElementXXL	d;
	d.obj			= observations;
	d.lastAccess	= mrpt::system::now();

	m_seqOfActObs.push_back( d );
	m_objectsInMemory++;

	swapEntriesToDiskIfRequired( mrpt::system::now() );
}

/*---------------------------------------------------------------
					addObservationMemoryReference
  ---------------------------------------------------------------*/
void  CRawlogXXL::addObservationMemoryReference( const CObservationPtr &observation )
{
	TObjectElementXXL	d;
	d.obj			= observation;
	d.lastAccess	= mrpt::system::now();

	m_seqOfActObs.push_back( d );
	m_objectsInMemory++;

	swapEntriesToDiskIfRequired( mrpt::system::now() );
}


/*---------------------------------------------------------------

  ---------------------------------------------------------------*/
void  CRawlogXXL::addAction( CAction &action )
{
	CActionCollection	*temp = new CActionCollection();
	temp->insert( action );

	TObjectElementXXL	d;
	d.obj			= CSerializablePtr(temp);
	d.lastAccess	= mrpt::system::now();

	m_seqOfActObs.push_back( d );
	m_objectsInMemory++;

	swapEntriesToDiskIfRequired( mrpt::system::now() );
}

/*---------------------------------------------------------------
						size
  ---------------------------------------------------------------*/
size_t  CRawlogXXL::size()
{
	return m_seqOfActObs.size();
}

/*---------------------------------------------------------------
						getAsAction
  ---------------------------------------------------------------*/
CActionCollectionPtr  CRawlogXXL::getAsAction( size_t index )
{
	MRPT_TRY_START

	assureElementIsLoaded(index);

	CSerializablePtr obj = m_seqOfActObs[index].obj;

	if ( obj->GetRuntimeClass() == CLASS_ID(CActionCollection) )
			return CActionCollectionPtr( obj );
	else	THROW_EXCEPTION_CUSTOM_MSG1("Element at index %i is not a CActionCollection",(int)index);
	MRPT_TRY_END
}

/*---------------------------------------------------------------
						getAsObservation
  ---------------------------------------------------------------*/
CObservationPtr  CRawlogXXL::getAsObservation( size_t index )
{
	MRPT_TRY_START

	assureElementIsLoaded(index);

	CSerializablePtr obj = m_seqOfActObs[index].obj;

	if ( obj->GetRuntimeClass()->derivedFrom( CLASS_ID(CObservation) ) )
			return CObservationPtr( obj );
	else	THROW_EXCEPTION_CUSTOM_MSG1("Element at index %i is not a CObservation",(int)index);
	MRPT_TRY_END
}


/*---------------------------------------------------------------
						getAsGeneric
  ---------------------------------------------------------------*/
CSerializablePtr CRawlogXXL::getAsGeneric( size_t index )
{
	MRPT_TRY_START
	assureElementIsLoaded(index);

	return m_seqOfActObs[index].obj;
	MRPT_TRY_END
}

/*---------------------------------------------------------------
						isAction
  ---------------------------------------------------------------*/
bool  CRawlogXXL::isAction( size_t index )
{
	MRPT_TRY_START
	assureElementIsLoaded(index);

	CSerializablePtr obj = m_seqOfActObs[index].obj;

	return ( obj->GetRuntimeClass() == CLASS_ID(CActionCollection) );
	MRPT_TRY_END
}

/*---------------------------------------------------------------
						getType
  ---------------------------------------------------------------*/
CRawlogXXL::TEntryType CRawlogXXL::getType( size_t index ) const
{
	MRPT_TRY_START
	const_cast<CRawlogXXL*>(this)->assureElementIsLoaded(index);

	const CSerializablePtr &obj = m_seqOfActObs[index].obj;

	if( obj->GetRuntimeClass()->derivedFrom( CLASS_ID(CObservation) ) )
		return etObservation;

	if( obj->GetRuntimeClass() == CLASS_ID(CActionCollection) )
		return etActionCollection;

	if( obj->GetRuntimeClass() == CLASS_ID(CSensoryFrame) )
		return etSensoryFrame;

	THROW_EXCEPTION("Object is not of any of the 3 allowed classes.");

	MRPT_TRY_END
}

/*---------------------------------------------------------------
						getAsObservations
  ---------------------------------------------------------------*/
CSensoryFramePtr  CRawlogXXL::getAsObservations( size_t index )
{
	MRPT_TRY_START
	assureElementIsLoaded(index);

	CSerializablePtr obj = m_seqOfActObs[index].obj;

	if ( obj->GetRuntimeClass()->derivedFrom( CLASS_ID(CSensoryFrame) ))
			return CSensoryFramePtr( obj );
	else	THROW_EXCEPTION_CUSTOM_MSG1("Element at index %i is not a CSensoryFrame",(int)index);
	MRPT_TRY_END
}

/*---------------------------------------------------------------
					writeToStream
	Implements the writing to a CStream capability of
	  CSerializable objects
  ---------------------------------------------------------------*/
void  CRawlogXXL::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 0;
	else
	{
		uint32_t	i,n;
		n = static_cast<uint32_t>( m_seqOfActObs.size() );
		out << n;
		for (i=0;i<n;i++)
		{
			const_cast<CRawlogXXL*>(this)->assureElementIsLoaded(i);
			out << m_seqOfActObs[i].obj;
		}
	}
}

/*---------------------------------------------------------------
					readFromStream
  ---------------------------------------------------------------*/
void  CRawlogXXL::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			uint32_t	i,n;

			clear();
			m_objectsInMemory = 0;

			in >> n;
			m_seqOfActObs.resize(n);
			for (i=0;i<n;i++)
			{
				m_seqOfActObs[i].obj = CSerializablePtr( in.ReadObject() );
				m_objectsInMemory++;
			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)
	};
}

/*---------------------------------------------------------------
						loadFromRawLogFile
  ---------------------------------------------------------------*/
bool  CRawlogXXL::loadFromRawLogFile( const std::string &fileName )
{
	bool		keepReading = true;
	// Open for read.
	CFileInputStream		fs(fileName);

	if (!fs.fileOpenCorrectly()) return false;

	// Clear first:
	clear();

	// OK: read objects:
	while (keepReading)
	{
		CSerializablePtr newObj;
		try
		{
			fs >> newObj;
            // Check type:
			if ( newObj->GetRuntimeClass() == CLASS_ID(CRawlogXXL))
        	{
				// It is an entire object: Copy and finish:
				CRawlogXXLPtr ao = CRawlogXXLPtr( newObj );
				this->swap(*ao);
				return true;
            }
            else if ( newObj->GetRuntimeClass() == CLASS_ID(CSensoryFrame))
            {
				TObjectElementXXL	d;
				d.obj			= newObj;
	        	m_seqOfActObs.push_back( d );
				m_objectsInMemory++;
            }
            else if ( newObj->GetRuntimeClass() == CLASS_ID(CActionCollection))
            {
				TObjectElementXXL	d;
				d.obj			= newObj;
	        	m_seqOfActObs.push_back( d );
				m_objectsInMemory++;
            }
			/** FOR BACKWARD COMPATIBILITY: CPose2D was used previously intead of an "ActionCollection" object
																				26-JAN-2006	*/
            else if ( newObj->GetRuntimeClass() == CLASS_ID(CPose2D))
            {
				CPose2DPtr					poseChange = CPose2DPtr( newObj );
				CActionCollectionPtr	temp = CActionCollectionPtr( new CActionCollection() );
				CActionRobotMovement2D		action;
				CActionRobotMovement2D::TMotionModelOptions	options;
				action.computeFromOdometry( *poseChange, options);
				temp->insert( action );

				TObjectElementXXL	d;
				d.obj			= temp;
	        	m_seqOfActObs.push_back( d );
				m_objectsInMemory++;
            }
			else
			{       // Unknown class:
				keepReading = false;
			}
		}
		catch (...)
		{
			keepReading = false;
		}
	}

	return true;
}

/*---------------------------------------------------------------
					remove
  ---------------------------------------------------------------*/
void  CRawlogXXL::remove( size_t index )
{
	MRPT_TRY_START;

	if (index >=m_seqOfActObs.size())
		THROW_EXCEPTION("Index out of bounds")

	m_seqOfActObs.erase( m_seqOfActObs.begin()+index );
	m_objectsInMemory--;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					remove
  ---------------------------------------------------------------*/
CRawlogXXL::iterator CRawlogXXL::remove(const CRawlogXXL::iterator &it)
{
	this->remove(it.m_index);
	return it;	// Same index, now is the next element
}

/*---------------------------------------------------------------
					moveFrom
  ---------------------------------------------------------------*/
void CRawlogXXL::moveFrom( CRawlogXXL &obj)
{
	MRPT_TRY_START
	if (this == &obj) return;
	clear();

	for (size_t i=0;i<obj.size();i++)
		obj.assureElementIsLoaded(i,false);

	m_seqOfActObs = obj.m_seqOfActObs;
	m_objectsInMemory = m_seqOfActObs.size();

	obj.m_seqOfActObs.clear();
	MRPT_TRY_END
}

/*---------------------------------------------------------------
					swap
  ---------------------------------------------------------------*/
void CRawlogXXL::swap( CRawlogXXL &obj)
{
	if (this == &obj) return;
	m_seqOfActObs.swap(obj.m_seqOfActObs);
}



/*---------------------------------------------------------------
						swapEntryToDisk
  ---------------------------------------------------------------*/
void CRawlogXXL::swapEntryToDisk( size_t index )
{
	MRPT_TRY_START;
	if (index >=m_seqOfActObs.size()) THROW_EXCEPTION("Index out of bounds");

	if (!m_seqOfActObs[index].obj.present()) return;	// Already swapped

	// Already swapped sometime in the past?
	if ( m_seqOfActObs[index].positionInTheTempFile != INVALID_POS)
	{
		// It's on disk, so just free the memory:
		m_seqOfActObs[index].obj.clear();
		return;
	}

	// It's on memory and not swapped in disk: Save to disk, then free:
	m_tempFile.Seek(0,CStream::sFromEnd);
	m_seqOfActObs[index].positionInTheTempFile = m_tempFile.getPositionO();

	m_tempFile << *m_seqOfActObs[index].obj;

	m_seqOfActObs[index].obj.clear();	// Free memory
	m_objectsInMemory--;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						assureElementIsLoaded
  ---------------------------------------------------------------*/
void CRawlogXXL::assureElementIsLoaded( size_t index, bool swapOutOldEntries )
{
	MRPT_TRY_START;
	if (index >=m_seqOfActObs.size()) THROW_EXCEPTION("Index out of bounds");

	const TTimeStamp	tnow = mrpt::system::now();

	m_seqOfActObs[index].lastAccess = tnow;

	// in memory yet?
	if (m_seqOfActObs[index].obj.present())
	{
		return;	// OK
	}

	ASSERT_(m_seqOfActObs[index].positionInTheTempFile!=INVALID_POS);

	// Nope: load it, but first unload some...
	// Swap out?
	if (swapOutOldEntries)
		swapEntriesToDiskIfRequired( mrpt::system::now() );


	// Load the element:
	m_tempFile.Seek( m_seqOfActObs[index].positionInTheTempFile );
	m_tempFile >> m_seqOfActObs[index].obj;
	m_objectsInMemory++;

	MRPT_TRY_END;
}


void CRawlogXXL::swapEntriesToDiskIfRequired(const TTimeStamp	tnow)
{
	if (m_objectsInMemory>200000 )
	{
		// Look for entries that have not been visited in a while:
		size_t	nRemoved = 0;
		size_t	j = 0;
		static const TTimeStamp THRE = mrpt::system::secondsToTimestamp(5.0);

		while (j<m_seqOfActObs.size() && nRemoved<1000)
		{
			if ( tnow - m_seqOfActObs[j].lastAccess > THRE )
			{
				swapEntryToDisk(j);
				nRemoved++;
			}
			j++;
		}
	} // end swap out old entries
}

