/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/poses/CPosePDFSOG.h>
#include <mrpt/poses/CPosePDFParticles.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/math/CMatrixD.h>
#include <mrpt/math_mrpt.h>
#include <mrpt/opengl/CEllipsoid.h>


using namespace mrpt;
using namespace mrpt::poses;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CPosePDFSOG, CPosePDF, mrpt::poses )


/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPosePDFSOG::CPosePDFSOG( size_t nModes ) :
	m_modes(nModes)
{

}

/*---------------------------------------------------------------
						getEstimatedPose
  Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
 ---------------------------------------------------------------*/
CPose2D  CPosePDFSOG::getEstimatedPose() const
{
	size_t		N = m_modes.size();

	if (N)
	{
		// Use an auxiliary parts. set to manage the problematic computation of the mean "PHI":
		//  See CPosePDFParticles::getEstimatedPose
		CPosePDFParticles										auxParts( N );
		CPosePDFParticles::CParticleList::iterator	itPart;
		std::deque<TGaussianMode>::const_iterator				it;

		for (it=m_modes.begin(),itPart=auxParts.m_particles.begin();it!=m_modes.end();it++,itPart++)
		{
			itPart->log_w = it->log_w;
			*itPart->d = it->mean;
		}

		return auxParts.getEstimatedPose();
	}
	else
	{
		return CPose2D(0,0,0);
	}
}

/*---------------------------------------------------------------
						getEstimatedCovariance
 ---------------------------------------------------------------*/
CMatrixD  CPosePDFSOG::getEstimatedCovariance() const
{
	size_t		N = m_modes.size();

	if (N)
	{
		// 1) Get the mean:
		//CPose3D		estMean_ = getEstimatedPose();
		CPose2D		estMean2D = getEstimatedPose(); //(estMean_);
		double		w,sumW = 0;
		CMatrixD	estCov(3,3);
		CMatrixD	estMeanMat( estMean2D );
		estCov.zeros();

		std::deque<TGaussianMode>::const_iterator	it;

		CMatrixD temp;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			sumW += w = exp(it->log_w);
			CMatrixD estMean_i ( it->mean );

			estMean_i -= estMeanMat;

			temp.multiply_AAt(estMean_i);
			temp+= it->cov;
			temp*=w;

			estCov += temp;
		}

		if (sumW==0)
				return CMatrixD(3,3);		// Zeros!
		else	return estCov * (1.0/sumW);
	}
	else
	{
		return CMatrixD(3,3);
	}
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CPosePDFSOG::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 2;
	else
	{
		uint32_t	N = m_modes.size();
		std::deque<TGaussianMode>::const_iterator		it;

		out << N;

		for (it=m_modes.begin();it!=m_modes.end();it++)
		{
			out << it->log_w;
			out << it->mean;
			out << it->cov(0,0) << it->cov(1,1) << it->cov(2,2);
			out << it->cov(0,1) << it->cov(0,2) << it->cov(1,2);
		}
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CPosePDFSOG::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
	case 1:
	case 2:
		{
			uint32_t								N;
			std::deque<TGaussianMode>::iterator		it;
			float									x;
			double									x0;

			in >> N;

			m_modes.resize(N);

			for (it=m_modes.begin();it!=m_modes.end();it++)
			{
				in >> it->log_w;

				// In version 0, weights were linear!!
				if (version==0) it->log_w = log(max(1e-300,it->log_w));

				in >> it->mean;

				it->cov.setSize(3,3);

				if (version==1)  // float's
				{
					in >> x; it->cov(0,0) = x;
					in >> x; it->cov(1,1) = x;
					in >> x; it->cov(2,2) = x;

					in >> x; it->cov(1,0) = x; it->cov(0,1) = x;
					in >> x; it->cov(2,0) = x; it->cov(0,2) = x;
					in >> x; it->cov(1,2) = x; it->cov(2,1) = x;
				}
				else
				{
					in >> x0; it->cov(0,0) = x0;
					in >> x0; it->cov(1,1) = x0;
					in >> x0; it->cov(2,2) = x0;

					in >> x0; it->cov(1,0) = x0; it->cov(0,1) = x0;
					in >> x0; it->cov(2,0) = x0; it->cov(0,2) = x0;
					in >> x0; it->cov(1,2) = x0; it->cov(2,1) = x0;
				}

			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}



/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
void  CPosePDFSOG::copyFrom(const CPosePDF &o)
{
	MRPT_TRY_START;

	if (this == &o) return;		// It may be used sometimes

	if (o.GetRuntimeClass()==CLASS_ID(CPosePDFSOG))
	{
		m_modes = static_cast<const CPosePDFSOG*>(&o)->m_modes;
	}
	else
	{
		// Approximate as a mono-modal gaussian pdf:
		// TODO: More elaborate!!
		m_modes.resize(1);
		m_modes[0].log_w = 0;
		m_modes[0].mean = o.getEstimatedPose();
		m_modes[0].cov = o.getEstimatedCovariance();
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						saveToTextFile
  ---------------------------------------------------------------*/
void  CPosePDFSOG::saveToTextFile(const std::string &file) const
{
	FILE	*f=os::fopen(file.c_str(),"wt");
	if (!f) return;


	for (std::deque<TGaussianMode>::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		os::fprintf(f,"%e %e %e %e %e %e %e %e %e %e\n",
			exp(it->log_w),
			it->mean.x, it->mean.y, it->mean.phi,
			it->cov(0,0),it->cov(1,1),it->cov(2,2),
			it->cov(0,1),it->cov(0,2),it->cov(1,2) );
	os::fclose(f);
}

/*---------------------------------------------------------------
						changeCoordinatesReference
 ---------------------------------------------------------------*/
void  CPosePDFSOG::changeCoordinatesReference(const CPose2D &newReferenceBase )
{
	CMatrixD		M = newReferenceBase.getHomogeneousMatrix();
	M.setSize(3,3);	// Clip the 4x4 matrix

	// The variance in phi is unmodified:
	M(0,2) = 0; M(1,2) = 0;
	M(2,0) = 0; M(2,1) = 0;
	M(2,2) = 1;

	for (std::deque<TGaussianMode>::iterator it=m_modes.begin();it!=m_modes.end();it++)
	{
		// The mean:
		it->mean = newReferenceBase + it->mean;

		// The covariance:
		M.multiplyByMatrixAndByTranspose( CMatrixD(it->cov), it->cov );  // * it->cov * (~M);
	}

	assureSymmetry();
}

/*---------------------------------------------------------------
						rotateAllCovariances
 ---------------------------------------------------------------*/
void  CPosePDFSOG::rotateAllCovariances(const double & ang)
{
	CMatrixD		rot(3,3);
	rot.zeros();
	rot(0,0)=rot(1,1)=cos(ang);
	rot(0,1)=-sin(ang);
	rot(1,0)=sin(ang);
	rot(2,2)=1;

	for (std::deque<TGaussianMode>::iterator it=m_modes.begin();it!=m_modes.end();it++)
		rot.multiplyByMatrixAndByTranspose( CMatrixD(it->cov), it->cov );
}

/*---------------------------------------------------------------
					drawSingleSample
 ---------------------------------------------------------------*/
void  CPosePDFSOG::drawSingleSample( CPose2D &outPart ) const
{
	MRPT_TRY_START;
	MRPT_UNUSED_PARAM(outPart);

	THROW_EXCEPTION("Not implemented yet!!");

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					drawManySamples
 ---------------------------------------------------------------*/
void  CPosePDFSOG::drawManySamples(
	size_t						N,
	std::vector<vector_double>	&outSamples )  const
{
	MRPT_TRY_START;
	MRPT_UNUSED_PARAM(N);
	MRPT_UNUSED_PARAM(outSamples);
//	std::vector<vector_float>	rndSamples;

	THROW_EXCEPTION("Not implemented yet!!");

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					bayesianFusion
 ---------------------------------------------------------------*/
void  CPosePDFSOG::bayesianFusion( CPosePDF &p1_, CPosePDF &p2_, const double &minMahalanobisDistToDrop )
{
	MRPT_TRY_START

	MRPT_UNUSED_PARAM(minMahalanobisDistToDrop);

	// p1: CPosePDFSOG, p2: CPosePDFGaussian:

	ASSERT_( p1_.GetRuntimeClass() == CLASS_ID(CPosePDFSOG) );
	ASSERT_( p2_.GetRuntimeClass() == CLASS_ID(CPosePDFGaussian) );

	CPosePDFSOG			*p1 = static_cast<CPosePDFSOG*>(&p1_);
	CPosePDFGaussian	*p2 = static_cast<CPosePDFGaussian*>(&p2_);

	// Compute the new kernel means, covariances, and weights after multiplying to the Gaussian "p2":
	CPosePDFGaussian	auxGaussianProduct,auxSOG_Kernel_i;
	TGaussianMode		newKernel;

	CMatrixD				covInv( p2->cov.inv() );
	CMatrixD				eta(3,1);
	eta(0,0) = p2->mean.x;
	eta(1,0) = p2->mean.y;
	eta(2,0) = p2->mean.phi;
	eta = covInv * eta;

	// Normal distribution canonical form constant:
	// See: http://www-static.cc.gatech.edu/~wujx/paper/Gaussian.pdf
	double				a = -0.5*( 3*log(M_2PI) - log( covInv.det() ) + (~eta * p2->cov * eta)(0,0) );

	this->m_modes.clear();
	for (std::deque<TGaussianMode>::iterator it =p1->m_modes.begin();it!=p1->m_modes.end();it++)
	{
		auxSOG_Kernel_i.mean = it->mean;
		auxSOG_Kernel_i.cov  = it->cov;
		auxGaussianProduct.bayesianFusion( auxSOG_Kernel_i, *p2 );

		// ----------------------------------------------------------------------
		// The new weight is given by:
		//
		//   w'_i = w_i * exp( a + a_i - a' )
		//
		//      a = -1/2 ( dimensionality * log(2pi) - log(det(Cov^-1)) + (Cov^-1 * mu)^t * Cov^-1 * (Cov^-1 * mu) )
		//
		// ----------------------------------------------------------------------
		newKernel.mean = auxGaussianProduct.mean;
		newKernel.cov  = auxGaussianProduct.cov;

		CMatrixD		covInv_i( auxSOG_Kernel_i.cov.inv() );
		CMatrixD		eta_i(3,1);
		eta_i(0,0) = auxSOG_Kernel_i.mean.x;
		eta_i(1,0) = auxSOG_Kernel_i.mean.y;
		eta_i(2,0) = auxSOG_Kernel_i.mean.phi;
		eta_i = covInv_i * eta_i;

		CMatrixD		new_covInv_i( newKernel.cov.inv() );
		CMatrixD		new_eta_i(3,1);
		new_eta_i(0,0) = newKernel.mean.x;
		new_eta_i(1,0) = newKernel.mean.y;
		new_eta_i(2,0) = newKernel.mean.phi;
		new_eta_i = new_covInv_i * new_eta_i;

		double		a_i	    = -0.5*( 3*log(M_2PI) - log( new_covInv_i.det() ) + (~eta_i * auxSOG_Kernel_i.cov * eta_i)(0,0) );
		double		new_a_i = -0.5*( 3*log(M_2PI) - log( new_covInv_i.det() ) + (~new_eta_i * newKernel.cov * new_eta_i)(0,0) );

		//newKernel.w	   = it->w * exp( a + a_i - new_a_i );
		newKernel.log_w	   = it->log_w + a + a_i - new_a_i;

		// Add to the results (in "this") the new kernel:
		this->m_modes.push_back( newKernel );
	}

	normalizeWeights();

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					inverse
 ---------------------------------------------------------------*/
void	 CPosePDFSOG::inverse(CPosePDF &o)  const
{
	ASSERT_(o.GetRuntimeClass() == CLASS_ID(CPosePDFSOG));
	CPosePDFSOG	*out = static_cast<CPosePDFSOG*>( &o );

	std::deque<TGaussianMode>::const_iterator	itSrc;
	std::deque<TGaussianMode>::iterator			itDest;

	out->m_modes.resize(m_modes.size());

	for (itSrc=m_modes.begin(),itDest=out->m_modes.begin();itSrc!=m_modes.end();itSrc++,itDest++)
	{
		// The mean:
		itDest->mean = CPose2D(0,0,0) - itSrc->mean;

		// The covariance: Is the same:
		itDest->cov = itSrc->cov;
	}
}


/*---------------------------------------------------------------
							+=
 ---------------------------------------------------------------*/
void  CPosePDFSOG::operator += ( CPose2D Ap)
{
	for (std::deque<TGaussianMode>::iterator it=m_modes.begin();it!=m_modes.end();it++)
		it->mean = it->mean + Ap;

	this->rotateAllCovariances( Ap.phi );
}

/*---------------------------------------------------------------
						evaluatePDF
 ---------------------------------------------------------------*/
double  CPosePDFSOG::evaluatePDF(
	const	CPose2D &x,
	bool	sumOverAllPhis ) const
{
	if (!sumOverAllPhis)
	{
		// Normal evaluation:
		CMatrixD	X(3,1), MU(3,1);
		double	ret = 0;

		X(0,0) = x.x;
		X(1,0) = x.y;
		X(2,0) = x.phi;

		for (std::deque<TGaussianMode>::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		{
			MU(0,0) = it->mean.x;
			MU(1,0) = it->mean.y;
			MU(2,0) = it->mean.phi;

			ret+= exp(it->log_w) * math::normalPDF( X, MU, it->cov );
		}

		return ret;
	}
	else
	{
		// Only X,Y:
		CMatrixD	X(2,1), MU(2,1),COV(2,2);
		double	ret = 0;

		X(0,0) = x.x;
		X(1,0) = x.y;

		for (std::deque<TGaussianMode>::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
		{
			MU(0,0) = it->mean.x;
			MU(1,0) = it->mean.y;

			COV(0,0) = it->cov(0,0);
			COV(1,1) = it->cov(1,1);
			COV(0,1) = COV(1,0) = it->cov(0,1);

			ret+= exp(it->log_w) * math::normalPDF( X, MU, COV );
		}

		return ret;
	}
}

/*---------------------------------------------------------------
						evaluateNormalizedPDF
 ---------------------------------------------------------------*/
double  CPosePDFSOG::evaluateNormalizedPDF( const CPose2D &x ) const
{
	CMatrixD	X(3,1), MU(3,1);
	double	ret = 0;

	X(0,0) = x.x;
	X(1,0) = x.y;
	X(2,0) = x.phi;

	for (std::deque<TGaussianMode>::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
	{
		MU(0,0) = it->mean.x;
		MU(1,0) = it->mean.y;
		MU(2,0) = it->mean.phi;

		ret+= exp(it->log_w) * math::normalPDF( X, MU, it->cov ) / math::normalPDF( MU, MU, it->cov );
	}

	return ret;
}

/*---------------------------------------------------------------
						assureSymmetry
 ---------------------------------------------------------------*/
void  CPosePDFSOG::assureSymmetry()
{
	// Differences, when they exist, appear in the ~15'th significant
	//  digit, so... just take one of them arbitrarily!
	for (std::deque<TGaussianMode>::iterator it=m_modes.begin();it!=m_modes.end();it++)
	{
		it->cov(0,1) = it->cov(1,0);
		it->cov(0,2) = it->cov(2,0);
		it->cov(1,2) = it->cov(2,1);
	}
}

/*---------------------------------------------------------------
						normalizeWeights
 ---------------------------------------------------------------*/
void  CPosePDFSOG::normalizeWeights()
{
	MRPT_TRY_START;

	if (!m_modes.size()) return;

	std::deque<TGaussianMode>::iterator		it;
	double									maxW = m_modes[0].log_w;
	for (it=m_modes.begin();it!=m_modes.end();it++)
		maxW = max(maxW,it->log_w);

	for (it=m_modes.begin();it!=m_modes.end();it++)
		it->log_w -= maxW;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						normalizeWeights
 ---------------------------------------------------------------*/
void  CPosePDFSOG::evaluatePDFInArea(
	const double &	x_min,
	const double &		x_max,
	const double &		y_min,
	const double &		y_max,
	const double &		resolutionXY,
	const double &		phi,
	CMatrixD	&outMatrix,
	bool		sumOverAllPhis )
{
	MRPT_TRY_START;

	ASSERT_(x_max>x_min);
	ASSERT_(y_max>y_min);
	ASSERT_(resolutionXY>0);

	size_t		Nx = (size_t)ceil((x_max-x_min)/resolutionXY);
	size_t		Ny = (size_t)ceil((y_max-y_min)/resolutionXY);
	size_t		i,j;
	double		x,y;

	outMatrix.setSize(Ny,Nx);

	for (i=0;i<Ny;i++)
	{
		y = y_min + i*resolutionXY;
		for (j=0;j<Nx;j++)
		{
			x = x_min + j*resolutionXY;
			outMatrix(i,j) = evaluatePDF(CPose2D(x,y,phi),sumOverAllPhis);
		}
	}


	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						getAs3DObject
 ---------------------------------------------------------------*/
void  CPosePDFSOG::getAs3DObject( mrpt::opengl::CSetOfObjectsPtr	&outObj ) const
{
	outObj->clear();

	for (std::deque<TGaussianMode>::const_iterator it=m_modes.begin();it!=m_modes.end();it++)
	{
		opengl::CEllipsoidPtr ellip = opengl::CEllipsoid::Create();
		ellip->m_x	 = it->mean.x;
		ellip->m_y	 = it->mean.y;
		ellip->m_z	 = it->mean.phi;

		ellip->m_cov = it->cov;

		ellip->m_color_R = 0;
		ellip->m_color_G = 0;
		ellip->m_color_B = 1;
		ellip->m_color_A = 0.6f;

		outObj->insert(ellip);
	}

}

/*---------------------------------------------------------------
						mergeModes
 ---------------------------------------------------------------*/
void CPosePDFSOG::mergeModes( double max_KLd, bool verbose  )
{
	MRPT_TRY_START

	normalizeWeights();

	size_t N = m_modes.size();
	if (N<2) return; // Nothing to do

	// Method described in:
	// "Kullback-Leibler Approach to Gaussian Mixture Reduction", A.R. Runnalls.
	// IEEE Transactions on Aerospace and Electronic Systems, 2007.

	size_t Nold = 0; // force recompute the first time

	for (size_t i=0;i<(N-1); )
	{
		N = m_modes.size(); // It might have changed.
		double sumW=0;

		//if (Nold!=N)
		{
			// For getting linear weights:
			sumW = 0;
			for (size_t i=0;i<N;i++)
				sumW += exp(m_modes[i].log_w);
			ASSERT_(sumW);

			Nold = N;
		}

		const double Wi = exp(m_modes[i].log_w) / sumW;

		double 	min_Bij = std::numeric_limits<double>::max();
		CMatrixDouble  min_Bij_COV;
		size_t  best_j = 0;

		CMatrixD MUi( m_modes[i].mean );

		// Compute B(i,j), j=[i+1,N-1]  (the discriminant)
		//for (size_t j=i+1;j<N;j++)
		for (size_t j=0;j<N;j++)
		if (i!=j)
		{
			const double Wj = exp(m_modes[j].log_w) / sumW;
			const double Wij_ = 1.0/(Wi+Wj);

			CMatrixDouble  Pij = m_modes[i].cov * Wi*Wij_;
			Pij += m_modes[j].cov * Wj*Wij_;

			CMatrixD 	MUij(m_modes[j].mean );
			MUij-=MUi;
			// Account for circular dimensions:
			mrpt::math::wrapToPiInPlace( MUij(2,0) );

			CMatrixDouble AUX;
			AUX.multiply_AAt( MUij ); // AUX = MUij * MUij^T

			AUX *= Wi*Wj*Wij_*Wij_;
			Pij += AUX;

			double Bij = (Wi+Wj)*log( Pij.det() ) - Wi*log(m_modes[i].cov.det()) - Wj*log(m_modes[j].cov.det());
			if (verbose)
			{
				cout << "try merge[" << i << ", " << j << "] -> Bij: " << Bij << endl;
				//cout << "AUX: " << endl << AUX; 
				//cout << "Wi: " << Wi << " Wj:" << Wj << " Wij_: " << Wij_ << endl;
				//cout << "Pij: " << Pij.det() << " Pi: " << m_modes[i].cov.det() << " Pj: " << m_modes[j].cov.det() << endl;
			}

			if (Bij<min_Bij)
			{
				min_Bij = Bij;
				best_j = j;
				min_Bij_COV = Pij;
			}
		}

		// Is a good move to merge (i,j)??
		if (verbose)
			cout << "merge[" << i << ", " << best_j << "] Tempting merge: KLd = " << min_Bij;

		if (min_Bij<max_KLd)
		{
			if (verbose)
				cout << " Accepted." << endl;

			// Do the merge (i,j):
			TGaussianMode	Mij;
			TGaussianMode	&Mi = m_modes[i];
			TGaussianMode	&Mj = m_modes[best_j];

			// Weight:
			Mij.log_w = log( exp(Mi.log_w) + exp(Mj.log_w) );

			// Mean:
			const double Wj = exp(Mj.log_w) / sumW;
			const double Wij_ = 1.0/(Wi+Wj);
			const double Wi_ = Wi*Wij_;
			const double Wj_ = Wj*Wij_;

			Mij.mean = CPose2D(
				Wi_ * Mi.mean.x + Wj_ * Mj.mean.x,
				Wi_ * Mi.mean.y + Wj_ * Mj.mean.y,
				Wi_ * Mi.mean.phi + Wj_ * Mj.mean.phi );

			// Cov:
			Mij.cov = min_Bij_COV;

			// Replace Mi <- Mij:
			m_modes[i] = Mij;
			m_modes.erase( m_modes.begin() + best_j );	// erase Mj
		} // end merge
		else 
		{
			if (verbose)
				cout << " Nope." << endl;

			i++;
		}
	} // for i

	normalizeWeights();

	MRPT_TRY_END
}
