/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/poses/CPointPDFGaussian.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/math/CMatrixD.h>
#include <mrpt/random/RandomGenerators.h>

using namespace mrpt::poses;
using namespace mrpt::utils;

IMPLEMENTS_SERIALIZABLE( CPointPDFGaussian, CPointPDF, mrpt::poses )


/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPointPDFGaussian::CPointPDFGaussian() : mean(0,0,0), cov(3,3)
{

}

/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPointPDFGaussian::CPointPDFGaussian(
	const CPoint3D	&init_Mean,
	const CMatrix	&init_Cov ) : mean(init_Mean), cov(init_Cov)
{

}

/*---------------------------------------------------------------
	Constructor
  ---------------------------------------------------------------*/
CPointPDFGaussian::CPointPDFGaussian(
	const CPoint3D	&init_Mean ) : mean(init_Mean), cov(3,3)
{
	cov.zeros();
}


/*---------------------------------------------------------------
						getEstimatedPose
  Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
 ---------------------------------------------------------------*/
CPoint3D  CPointPDFGaussian::getEstimatedPoint() const
{
	return mean;
}

/*---------------------------------------------------------------
						getEstimatedCovariance
 ---------------------------------------------------------------*/
CMatrixD  CPointPDFGaussian::getEstimatedCovariance() const
{
	return cov;
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CPointPDFGaussian::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		out << mean << cov;
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CPointPDFGaussian::readFromStream(CStream &in,int version)
{
	switch(version)
	{
	case 0:
		{
			in >> mean;

			CMatrix c;
			in >> c; cov = c;
		} break;
	case 1:
		{
			in >> mean >> cov;
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}


/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
void  CPointPDFGaussian::copyFrom(const CPointPDF &o)
{
	if (this == &o) return;		// It may be used sometimes

	// Convert to gaussian pdf:
	mean = o.getEstimatedPoint();
	cov  = o.getEstimatedCovariance();
}

/*---------------------------------------------------------------

  ---------------------------------------------------------------*/
void  CPointPDFGaussian::saveToTextFile(const std::string &file) const
{
	MRPT_TRY_START;

	FILE	*f=os::fopen(file.c_str(),"wt");
	if (!f) return;

	os::fprintf(f,"%f %f %f\n", mean.x, mean.y, mean.z );

	os::fprintf(f,"%f %f %f\n", cov(0,0),cov(0,1),cov(0,2) );
	os::fprintf(f,"%f %f %f\n", cov(1,0),cov(1,1),cov(1,2) );
	os::fprintf(f,"%f %f %f\n", cov(2,0),cov(2,1),cov(2,2) );


	os::fclose(f);

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
						changeCoordinatesReference
 ---------------------------------------------------------------*/
void  CPointPDFGaussian::changeCoordinatesReference(const CPose3D &newReferenceBase )
{
	CMatrixD		M( newReferenceBase.getHomogeneousMatrix() );
	M.setSize(3,3);	// Clip the 4x4 matrix

	// The mean:
	mean = newReferenceBase + mean;

	// The covariance:
	CMatrixD		temp(cov);
	M.multiplyByMatrixAndByTranspose(temp, cov); // save in cov
}

/*---------------------------------------------------------------
					bayesianFusion
 ---------------------------------------------------------------*/
void  CPointPDFGaussian::bayesianFusion( CPointPDFGaussian &p1, CPointPDFGaussian &p2 )
{
	MRPT_TRY_START;

	CMatrixD	x1(3,1),x2(3,1),x(3,1);
	CMatrixD	C1( p1.cov );
	CMatrixD	C2( p2.cov );
	CMatrixD	C1_inv( !C1 );
	CMatrixD	C2_inv( !C2 );
	CMatrixD	C;

	x1(0,0) = p1.mean.x; x1(1,0) = p1.mean.y; x1(2,0) = p1.mean.z;
	x2(0,0) = p2.mean.x; x2(1,0) = p2.mean.y; x2(2,0) = p2.mean.z;


	C = !(C1_inv + C2_inv);
	cov = C;

	x = C * ( C1_inv*x1 + C2_inv*x2 );

	mean.x = x(0,0);
	mean.y = x(1,0);
	mean.z = x(2,0);

//	std::cout << "IN1: " << x1 << "\n" << C1 << "\n";
//	std::cout << "IN2: " << x2 << "\n" << C2 << "\n";
//	std::cout << "OUT: " << x << "\n" << C << "\n";

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					productIntegralWith
 ---------------------------------------------------------------*/
double  CPointPDFGaussian::productIntegralWith( CPointPDFGaussian &p)
{
	MRPT_TRY_START;

	double	s,x;

	// ---------------------------------------------
	// Derivation in notebook, date 13-JUN-2006
	//  Jose Luis Blanco Claraco
	// ---------------------------------------------
	// Inverses of covariance matrixs:
	CMatrixD	C1(cov);
	CMatrixD	C2(p.cov);
	CMatrixD	L1(!C1);
	CMatrixD	L2(!C2);
	CMatrixD	L = L1+L2;
	CMatrixD	C = !L;

	// The mean values as matrixes:
	CMatrixD	MU1(3,1); MU1(0,0)=mean.x; MU1(1,0)=mean.y; MU1(2,0)=mean.z;
	CMatrixD	MU2(3,1); MU2(0,0)=p.mean.x; MU2(1,0)=p.mean.y; MU2(2,0)=p.mean.z;

	// The ETA vectors as matrixes:
	CMatrixD	ETA1(1,3); ETA1 = ~(L1 * MU1);
	CMatrixD	ETA2(1,3); ETA2 = ~(L2 * MU2);
	CMatrixD	ETA(1,3); ETA = ETA1 + ETA2;

	// The solution:
	// x = ((~ETA1) * C1 * (ETA1))(0,0) + ((~ETA2)*C2*(ETA2))(0,0) - ((~ETA)*C*(ETA))(0,0);
	x = ETA1.multiplyByMatrixAndByTransposeScalar(C1) +
	    ETA2.multiplyByMatrixAndByTransposeScalar(C2) +
	     ETA.multiplyByMatrixAndByTransposeScalar(C);

	s = sqrt( L1.det() * L2.det() / L.det() ) / M_2PI;
	s*= exp( -0.5*x  );

	return s;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					productIntegralNormalizedWith
 ---------------------------------------------------------------*/
double  CPointPDFGaussian::productIntegralNormalizedWith( CPointPDFGaussian *p)
{
	MRPT_TRY_START;

	double	s,x;

	// ---------------------------------------------
	// Derivation in notebook, date 13-JUN-2006
	//  Jose Luis Blanco Claraco
	// ---------------------------------------------
	/** \todo Pass to "multiplyByMatrixAndByTransposeScalar"
	  */

	// Inverses of covariance matrixs:
	CMatrixD	C1(cov);
	CMatrixD	C2(p->cov);
	CMatrixD	L1(!C1);
	CMatrixD	L2(!C2);
	CMatrixD	C( !(L1+L2) );

	// The mean values as matrixes:
	CMatrixD	MU1(3,1); MU1(0,0)=mean.x; MU1(1,0)=mean.y; MU1(2,0)=mean.z;
	CMatrixD	MU2(3,1); MU2(0,0)=p->mean.x; MU2(1,0)=p->mean.y; MU2(2,0)=p->mean.z;

	// The ETA vectors as matrixes:

	// TODO: Optimizar sin matrices!! *************
	CMatrixD	ETA1(3,1); ETA1 = L1 * MU1;
	CMatrixD	ETA2(3,1); ETA2 = L2 * MU2;
	CMatrixD	ETA(3,1); ETA = ETA1 + ETA2;

	// The solution:
	x = ((~ETA1) * C1 * (ETA1))(0,0) + ((~ETA2)*C2*(ETA2))(0,0) - ((~ETA)*C*(ETA))(0,0);
	s = exp( -0.5*x  );

	/* DEBUG ********* /
	std::cout << "MU1:" << MU1 << "\n";
	std::cout << "MU2:" << MU2 << "\n";
	std::cout << "C1:" << p->cov << "\n";
	std::cout << "C2:" << cov << "\n";
	std::cout << "C':" << C << "\n";
	std::cout << "CORR:" << s << "\n";
	// ************** */


	// Loosy precision... :-(
	if (s>1) s=1;
	else if (s<0) s=0;

	return s;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					productIntegralNormalizedWith
 ---------------------------------------------------------------*/
double  CPointPDFGaussian::productIntegralNormalizedWith2D( CPointPDFGaussian *p)
{
	MRPT_TRY_START;

	double	s,x;

	// ---------------------------------------------
	// Derivation in notebook, date 13-JUN-2006
	//  Jose Luis Blanco Claraco
	// ---------------------------------------------

	// The covariances:
	double	C1_11 = cov(0,0);
	double	C1_12 = cov(0,1);
	double	C1_22 = cov(1,1);

	double	C2_11 = p->cov(0,0);
	double	C2_12 = p->cov(0,1);
	double	C2_22 = p->cov(1,1);


	// Inverses of covariance matrixes:
	double	invDetC1 = 1 / ( C1_11*C1_22 - C1_12*C1_12 );
	double	invDetC2 = 1 / ( C2_11*C2_22 - C2_12*C2_12 );

	double	invC1_11 = cov(1,1) * invDetC1;
	double	invC1_22 = cov(0,0) * invDetC1;
	double	invC1_12 = -cov(0,1) * invDetC1;

	double	invC2_11 = p->cov(1,1) * invDetC2;
	double	invC2_22 = p->cov(0,0) * invDetC2;
	double	invC2_12 = -p->cov(0,1) * invDetC2;

	// C^-1 = C1^-1 + C2^-1
	double	invC_11 = invC1_11 + invC2_11;
	double	invC_22 = invC1_22 + invC2_22;
	double	invC_12 = invC1_12 + invC2_12;

	double	invDetC = 1 / ( invC_11*invC_22 - square(invC_12) );

	double	C_11 = invC_22 * invDetC;
	double	C_22 = invC_11 * invDetC;
	double	C_12 = -invC_12 * invDetC;

	// The ETA vectors :
	double	ETA1_1 = invC1_11*mean.x + invC1_12*mean.y;
	double	ETA1_2 = invC1_12*mean.x + invC1_22*mean.y;

	double	ETA2_1 = invC2_11*p->mean.x + invC2_12*p->mean.y;
	double	ETA2_2 = invC2_12*p->mean.x + invC2_22*p->mean.y;

	double	ETA_1 = ETA1_1 + ETA2_1;
	double	ETA_2 = ETA1_2 + ETA2_2;

	// x = ((~ETA1) * C1 * (ETA1))(0,0) + ((~ETA2)*C2*(ETA2))(0,0) - ((~ETA)*C*(ETA))(0,0);

	x  = ETA1_1*ETA1_1*C1_11 + 2*ETA1_1*ETA1_2*C1_12+ETA1_2*ETA1_2*C1_22;
	x += ETA2_1*ETA2_1*C2_11 + 2*ETA2_1*ETA2_2*C2_12+ETA2_2*ETA2_2*C2_22;
	x -= ETA_1*ETA_1*C_11 + 2*ETA_1*ETA_2*C_12+ETA_2*ETA_2*C_22;
	s = exp( -0.5*x  );

	// Loosy precision... :-(
	if (s>1) s=1;
	else if (s<0) s=0;

	return s;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					drawSingleSample
 ---------------------------------------------------------------*/
void CPointPDFGaussian::drawSingleSample(CPoint3D &outSample) const
{
	MRPT_TRY_START

	vector_double vec;
	mrpt::random::randomNormalMultiDimensional(cov,vec);

	ASSERT_(vec.size()==3);
	outSample.x = mean.x + vec[0];
	outSample.y = mean.y + vec[1];
	outSample.z = mean.z + vec[2];

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					bayesianFusion
 ---------------------------------------------------------------*/
void  CPointPDFGaussian::bayesianFusion( CPointPDF &p1_, CPointPDF &p2_,const double &minMahalanobisDistToDrop )
{
	MRPT_TRY_START;

	// p1: CPointPDFGaussian, p2: CPosePDFGaussian:
	ASSERT_( p1_.GetRuntimeClass() == CLASS_ID(CPointPDFGaussian) );
	ASSERT_( p2_.GetRuntimeClass() == CLASS_ID(CPointPDFGaussian) );

	THROW_EXCEPTION("TODO!!!");

	MRPT_TRY_END
}
