/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CStereoServerBumblebee_H
#define CStereoServerBumblebee_H

#include <mrpt/slam/CObservationVisualLandmarks.h>
#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/vision/CStereoGrabber_Bumblebee.h>
#include <mrpt/vision/utils.h>
#include <mrpt/vision/CFeature.h>
#include <mrpt/synch.h>

#include <mrpt/config.h>

#if MRPT_HAS_BUMBLEBEE

#include <digiclops.h>
#include <triclops.h>

#endif

namespace mrpt
{
	namespace vision
	{
		class MRPTDLLIMPEXP  CStereoServerBumblebee
		{
		private:
			// ****************** GENERAL MEMBERS *********************
			vision::CStereoGrabber_Bumblebee		*grabber;				// Auxiliar grabber for the class

			mrpt::slam::CObservationStereoImages		*preImgs;				// Previous images of the camera
			mrpt::slam::CObservationStereoImages		*curImgs;				// Current images of the camera

			vector_float						pvX, pvY, pvZ;			// Vectors for the coordinates of the 3D cloud.
			vector_float						cvX, cvY, cvZ;			// Vectors for the coordinates of the 3D cloud.

			mrpt::slam::CObservationVisualLandmarks	preGrid;				// Previous map 3D computed by Bumblebee software
			mrpt::slam::CObservationVisualLandmarks	curGrid;				// Current map 3D computed by Bumblebee software

			mrpt::slam::CObservationVisualLandmarks	tmpGrid;				// Temporal copy of the current map 3D computed by Bumblebee software

			// ****************** TRACKING *********************
			vision::CFeatureList					KLTListL, KLTListR;		// Tracked features
			unsigned int						nStereoImages;			// Indicates if tracking is possible or not (there are two pairs of stereo images available)
			unsigned int						nIter;					// Number of iterations for tracking
			mrpt::system::TTimeStamp					lastTS;					// Timestamp of the current observation

			// ****************** OBSTACLE AVOIDANCE *********************
			vision::TROI							safetyZone;				// Safety Zone of the robot
			bool								alert;					// Safety Zone violation
			mrpt::system::TTimeStamp					lastTSGrid;				// Timestamp of the grid observation (for obstacle avoidance)
			float								safetyRange;			// Minimum distance allowed

			// ****************** CRITICAL SECTIONS *********************
			synch::CCriticalSection		csImgs;					// Images
			synch::CCriticalSection		csGrid;					// Grid
			synch::CCriticalSection		csList;					// List of tracked features
			synch::CCriticalSection		csGrabber;				// Stereo Grabber
			synch::CCriticalSection		csVec;					// Vectors

		public:
			// ** Methods **
			// Default constructor
			CStereoServerBumblebee( int cameraIndex = 0, unsigned int resolutionX = 320, unsigned int resolutionY = 240 );
			// Destructor
			virtual ~CStereoServerBumblebee();

			CStereoServerBumblebee(const CStereoServerBumblebee & )  :
				grabber(NULL),
				preImgs(NULL),
				curImgs(NULL),
				pvX(), pvY(), pvZ(),
				cvX(), cvY(), cvZ(),
				preGrid(),curGrid(),tmpGrid(),KLTListL(), KLTListR(),
				nStereoImages(0),nIter(0),lastTS(),safetyZone(),
				alert(false),lastTSGrid(),safetyRange(),
				csImgs(),csGrid(),csList(),csGrabber(),csVec()
			{
				THROW_EXCEPTION("This class cannot be copied.");
			}

			CStereoServerBumblebee & operator =(const CStereoServerBumblebee&) { THROW_EXCEPTION("This class cannot be copied."); }

			// -- Images --
			// These are BABEL monitors
			// Capture
			void  monitorCaptureImgs();
			void  monitorCaptureGrid();
			void  monitorCaptureImgsAndGrid();

			// Processing
			//void  monitorProcessTracking();
			void  monitorObstacleAvoidance();

			void  captureImgsAndObs();

			// Get a pair of images
			void  getImgs( mrpt::slam::CObservationStereoImages &outImgs );
			void  getObs( mrpt::slam::CObservationVisualLandmarks &outObs );

			void  getImgsAndObs( mrpt::slam::CObservationStereoImages &outImgs, mrpt::slam::CObservationVisualLandmarks &outGrid );

			//// -- Grid --
			//// Get grid in all the 3D space
			//void  getGrid3D( mrpt::slam::CObservationVisualLandmarks &outObs );
			//// Get grid with ROI
			//void  getGrid3D( TROI ROI, mrpt::slam::CObservationVisualLandmarks &outObs );

			//// -- Both --
			//// Get both Stereo Images and Grid in all the 3D space
			//void  getBothImagesAndGrid( mrpt::slam::CObservationStereoImages &outImgs, mrpt::slam::CObservationVisualLandmarks &outObs );
			//// Get both Stereo Images and Grid with ROI
			//void  getBothImagesAndGrid( TROI ROI, mrpt::slam::CObservationStereoImages &outImgs, mrpt::slam::CObservationVisualLandmarks &outObs );

		}; // end of class
	} // end of namespace
} // end of namespace

#endif
