/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CFeature_H
#define CFeature_H

#include <mrpt/utils/CMRPTImage.h>

namespace mrpt
{
	namespace vision
	{
		using namespace mrpt::utils;

		class MRPTDLLIMPEXP CFeatureList;
		class MRPTDLLIMPEXP CMatchedFeatureList;

		/** Definition of a feature ID
		*/
		typedef uint64_t TFeatureID;
		/** Types of features
		*/
		enum TFeatureType
		{
			/** Scale Invariant Feature Transform [LOWE'04]
			  */
			featSIFT,
			/** Kanade-Lucas-Tomasi feature [SHI'94]
			  */
			featKLT,
			/** Harris border and corner detector [HARRIS]
			  */
			featHarris,
			/** Speeded Up Robust Feature [BAY'06]
			  */
			featSURF,
			/** Non-defined feature
			  */
			featNotDefined
		};

		enum TKLTFeatureStatus
		{
			/** Inactive
			*/
			statusKLT_IDLE,
			/** Out Of Bounds
			*/
			statusKLT_OOB,
			/** Determinant of the matrix too small
			*/
			statusKLT_SMALL_DET,
			/** Error too big
			*/
			statusKLT_LARGE_RESIDUE,
			/**
			*/
			statusKLT_MAX_RESIDUE,
			/** Feature correctly tracked
			*/
			statusKLT_TRACKED,
			/** Iteration maximum reached
			*/
			statusKLT_MAX_ITERATIONS
		};

		/****************************************************
						Class CFEATURE
		*****************************************************/
		DEFINE_SERIALIZABLE_PRE( CFeature )
		/** A generic 2D feature from an image
		*/
		class MRPTDLLIMPEXP CFeature : public mrpt::utils::CSerializable
		{
			friend class CFeatureList;
			friend class CMatchedFeatureList;

			DEFINE_SERIALIZABLE( CFeature )

		public:
			/** Coordinates in the image
			*/
			float			x,y;
			/** ID of the feature
			*/
			TFeatureID		ID;

			/** A patch of the image surrounding the feature
			*/
			CMRPTImage		patch;

			/** Size of the patch (patchSize x patchSize) (it must be an odd number)
			*/
			unsigned int	patchSize;

			/** Type of the feature: featSIFT, featKLT,	featHarris, featSURF, featNotDefined
			*/	
			TFeatureType	type;

			/** Status of the feature tracking process
			*/
			TKLTFeatureStatus	KLT_status;

			/** Value of the goodness of the feature
			*/
			float				KLT_val;

			/** Main orientation of the feature
			*/
			float				orientation;

			/** Feature scale into the scale space
			*/
			float				scale;

			/** Feature descriptor
			*/
			std::vector<unsigned char>	descriptorSIFT;

			/** Feature descriptor
			*/
			std::vector<unsigned char>	descriptorSURF;

			/** Test if this feature has a descriptor
			*/
			bool hasDescriptorSIFT;

			/** Test if this feature has a descriptor
			*/
			bool hasDescriptorSURF;

			/** Computes de Euclidean Distance between "this" and the "other" descriptors
			*/
			float descriptorSIFTDistanceTo( const CFeature &oFeature ) const;

			/** Computes de Euclidean Distance between "this" and the "other" descriptors
			*/
			float descriptorSURFDistanceTo( const CFeature &oFeature ) const;

			/** Get the type of the feature
			*/
			TFeatureType get_type() const { return type; }

			/** Constructor
			*/
			CFeature();

			/** Virtual destructor */
			virtual ~CFeature() {}


		protected:
			//unsigned int numRefs;

		}; // end of class

		/****************************************************
						Class CFEATURELIST
		*****************************************************/
		/** A list of features
		*/
		class MRPTDLLIMPEXP CFeatureList : public std::deque<CFeaturePtr>
		{
		public:

			/** The type of feature which contains
			*/
			TFeatureType get_type() const {
				if( size() > 0 )
					return (*begin())->get_type();
				else
					return featNotDefined;
			}

			/** Save feature list to a text file
			*/
			void saveToTextFile( const std::string &fileName, bool APPEND = false );

			/** Save feature list to a text file
			*/
			TFeatureID getMaxID();

			/** Get a reference to a Feature from its ID
			*/
			CFeaturePtr getByID( TFeatureID ID ) const;

			/** Insert a feature into the list (use this function instead of push_back)
			*/
			void insert( const CFeaturePtr &feat );

			/** Remove a feature from the list (use this function instead of erase)
			*/
			CFeatureList::iterator remove( CFeatureList::iterator itFeat );

			/** Constructor
			*/
			CFeatureList();

			/** Copy Constructor
			*/
			CFeatureList(const CFeatureList &o);

			/** Copy Operator
			*/
			CFeatureList& operator=(const CFeatureList &o);

			/** Virtual destructor */
			virtual ~CFeatureList();

		}; // end of class

		/****************************************************
					Class CMATCHEDFEATURELIST
		*****************************************************/
		/** A list of features
		*/
		class MRPTDLLIMPEXP CMatchedFeatureList : public std::deque< std::pair<CFeaturePtr,CFeaturePtr> >
		{
		public:
			/** The type of feature which contains
			*/
			TFeatureType get_type() const {
				if( size() > 0 )
					return (begin()->first)->get_type();
				else
					return featNotDefined;
			}

			/** Save list of matched features to a text file
			*/
			void saveToTextFile(const std::string &fileName);

			/** Insert a pair of matched features into the list (use this function instead of push_back)
			*/
			void insert( std::pair<CFeaturePtr,CFeaturePtr> mFeats );

			/** Remove a pair of matched features from the list (use this function instead of erase)
			*/
			CMatchedFeatureList::iterator remove( CMatchedFeatureList::iterator pMatchedFeats );

			/** Constructor
			*/
			CMatchedFeatureList();

			/** Copy Constructor
			*/
			CMatchedFeatureList(const CMatchedFeatureList &o);

			/** Copy Operator
			*/
			CMatchedFeatureList& operator=(const CMatchedFeatureList &o);

			/** Virtual destructor */
			virtual ~CMatchedFeatureList();
		}; // end of class

	} // end of namespace
} // end of namespace

#endif

