/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CTexturedPlane_H
#define opengl_CTexturedPlane_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/utils/CMRPTImage.h>

namespace mrpt
{
	namespace opengl
	{
		class MRPTDLLIMPEXP CTexturedPlane;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CTexturedPlane, CRenderizable )

		/** A 2D plane in the XY plane with a texture image.
		  *  \sa opengl::COpenGLScene
		  */
		class MRPTDLLIMPEXP CTexturedPlane : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CTexturedPlane )
		protected:
			unsigned int		m_glTextureName;
			bool				m_init;
			mrpt::utils::CMRPTImage			m_textureImage;
			mrpt::utils::CMRPTImage			m_textureImageAlpha;
			bool				m_enableTransparency;

			int					r_width,r_height;

		public:
			float				m_tex_x_min,m_tex_x_max;
			float				m_tex_y_min,m_tex_y_max;

			float			m_xMin, m_xMax;
			float			m_yMin, m_yMax;

			/** Assigns a texture and a transparency image, and enables transparency (If the images are not 2^N x 2^M, they will be internally filled to its dimensions to be powers of two)
			  */
			void  assignImage(
				const mrpt::utils::CMRPTImage&	img,
				const mrpt::utils::CMRPTImage&	imgAlpha );

			/** Assigns a texture image, and disable transparency.
			  */
			void  assignImage(
				const mrpt::utils::CMRPTImage&	img );

			/** Similar to assignImage, but the passed images will be returned as empty: it avoids making a copy of the whole image, just copies a pointer.
			  */
			void  assignImage_fast(
				mrpt::utils::CMRPTImage&	img,
				mrpt::utils::CMRPTImage&	imgAlpha );

			/** Similar to assignImage, but the passed images will be returned as empty: it avoids making a copy of the whole image, just copies a pointer.
			  */
			void  assignImage_fast(
				mrpt::utils::CMRPTImage&	img );

			/** Render
			  */
			void  render();

			/** VERY IMPORTANT: If you use a multi-thread application, you MUST call this from the same thread that will later destruct the object in order to the OpenGL texture memory to be correctly deleted.
			  *  Calling this method more than once has no effects. If you user one thread, this method will be automatically called when rendering, so there is no need to explicitly call it.
			  */
			void 		loadTextureInOpenGL();


			/** Class factory  */
			static CTexturedPlanePtr Create(
				float				x_min = -1,
				float				x_max = 1,
				float				y_min = -1,
				float				y_max = 1 )
			{
				return CTexturedPlanePtr( new CTexturedPlane(x_min, x_max, y_min, y_max) );
			}

		private:
			/** Constructor
			  */
			CTexturedPlane(
				float				x_min = -1,
				float				x_max = 1,
				float				y_min = -1,
				float				y_max = 1
				);

			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CTexturedPlane();
		};

	} // end namespace

} // End of namespace


#endif
