/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CEllipsoid_H
#define opengl_CEllipsoid_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/math/CMatrixD.h>

namespace mrpt
{
	namespace opengl
	{
		class MRPTDLLIMPEXP CEllipsoid;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CEllipsoid, CRenderizable )

		/** A 2D ellipse or 3D ellipsoid, depending on the size of the m_cov matrix (2x2 or 3x3).
		  *  The center of the ellipsoid is the "m_x,m_y,m_z" object's coordinates. In the case of
		  *   a 2D ellipse it will be drawn in the XY plane, for z=0.
		  *  The color is determined by the RGBA fields in the class "CRenderizable". Note that a
		  *   transparent ellipsoid can be drawn for "0<alpha<1" values.
		  *  \sa opengl::COpenGLScene
		  */
		class MRPTDLLIMPEXP CEllipsoid : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CEllipsoid )

		private:
			/** Used to store computed values the first time this is rendered, and to avoid recomputing them again.
			 */
			math::CMatrixD		m_eigVal,m_eigVec,m_prevComputedCov;

		public:
			/** The 2x2 or 3x3 covariance matrix that will determine the aspect of the ellipsoid.
			  */
			math::CMatrixD		m_cov;
			/** If set to true (default), a whole ellipsoid surface will be drawn, or if set to "false" it will be drawn as a "wireframe".
			  */
			bool			m_drawSolid3D;
			/** The number of "sigmas" for drawing the ellipse/ellipsoid (default=3)
			  */
			float			m_quantiles;
			/** The number of segments of a 2D ellipse (default=20)
			  */
			unsigned int	m_2D_segments;
			/** The number of segments of a 3D ellipse (in both "axis") (default=20)
			  */
			unsigned int	m_3D_segments;
			/** The line width for 2D ellipses or 3D wireframe ellipsoids (default=1)
			  */
			float			m_lineWidth;

			/** Class factory  */
			static CEllipsoidPtr Create()
			{
				return CEllipsoidPtr( new CEllipsoid() );
			}

			/** Render
			  */
			void  render();

		private:
			/** Constructor
			  */
			CEllipsoid() : m_eigVal(),m_eigVec(),m_prevComputedCov(),
				m_cov(2,2),
				m_drawSolid3D(true),
				m_quantiles(3),
				m_2D_segments(20),
				m_3D_segments(20),
				m_lineWidth(1.0)
			{
			}
			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CEllipsoid() { }
		};

	} // end namespace

} // End of namespace


#endif
