/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  mrpt_math_vector_ops_H
#define  mrpt_math_vector_ops_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>

#include <iterator>

namespace mrpt
{
	namespace utils { class CFileStream; }

	namespace math
	{
		/** A template function for printing out the contents of a std::vector variable.
			*/
		template <class T>
		std::ostream& operator << (std::ostream& out, const std::vector<T> &d)
		{
			out << "[";
			copy(d.begin(),d.end(), std::ostream_iterator<T>(out," "));
			out << "]";
			return out;
		}

		/** A template function for printing out the contents of a std::vector variable.
			*/
		template <class T>
		std::ostream& operator << (std::ostream& out, std::vector<T> *d)
		{
			out << "[";
			copy(d->begin(),d->end(), std::ostream_iterator<T>(out," "));
			out << "]";
			return out;
		}


		/** A template function for the operator + between a scalar and a std::vector.
			*/
		template <class T>
		std::vector<T>  operator + (const std::vector<T> &a, T b)
		{
			typename std::vector<T>	res(a.size());
			typename std::vector<T>::const_iterator	it_a;
			typename std::vector<T>::iterator		it_res;

			for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
				*it_res = *it_a + b;

			return res;
		}

		/** A template function for the operator - between a scalar and a std::vector.
			*/
		template <class T>
		std::vector<T>  operator - (const std::vector<T> &a, T b)
		{
			typename std::vector<T>	res(a.size());
			typename std::vector<T>::const_iterator	it_a;
			typename std::vector<T>::iterator		it_res;

			for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
				*it_res = *it_a - b;

			return res;
		}

		/** A template function for the operator * between a scalar and a std::vector.
			*/
		template <class T>
		std::vector<T>  operator * (const std::vector<T> &a, T b)
		{
			typename std::vector<T>	res(a.size());
			typename std::vector<T>::const_iterator	it_a;
			typename std::vector<T>::iterator		it_res;

			for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
				*it_res = *it_a * b;

			return res;
		}

		/** A template function for dividing a std::vector by a scalar number.
			*/
		template <class T>
		std::vector<T>  operator / (const std::vector<T> &a, T b)
		{
			typename std::vector<T>	res(a.size());
			typename std::vector<T>::const_iterator	it_a;
			typename std::vector<T>::iterator		it_res;

			if (b==0)
				THROW_EXCEPTION("Division by zero: execution aborted!");

			for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
				*it_res = *it_a / b;

			return res;
		}

		/** A template function for adding std::vector's having the same number of elements.
			*/
		template <class T>
		std::vector<T>  operator + (const std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());

			typename std::vector<T>	res(a.size());
			typename std::vector<T>::const_iterator	it_a,it_b;
			typename std::vector<T>::iterator		it_res;

			for (it_a=a.begin(), it_b=b.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_b++, it_res++)
				*it_res = *it_a + *it_b;

			return res;
		}

		/** A template function for substracting std::vector's having the same number of elements.
			*/
		template <class T>
		std::vector<T>  operator - (const std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());

			typename std::vector<T>	res(a.size());
			typename std::vector<T>::const_iterator	it_a,it_b;
			typename std::vector<T>::iterator		it_res;

			for (it_a=a.begin(), it_b=b.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_b++, it_res++)
				*it_res = *it_a - *it_b;

			return res;
		}

		/** A template function for scalar product (element by element, like .* in MATLAB) of std::vector's having the same number of elements.
			*/
		template <class T>
		std::vector<T>  operator * (const std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());
			typename std::vector<T>	res(a.size());
			std::transform(a.begin(), a.end(), b.begin(), res.begin(), std::multiplies<T>() );
			return res;
		}

		/** A template function for scalar division (element by element, like ./ in MATLAB) of std::vector's having the same number of elements.
			*/
		template <class T>
		std::vector<T>  operator / (const std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());
			typename std::vector<T>	res(a.size());
			std::transform(a.begin(), a.end(), b.begin(), res.begin(), std::divides<T>() );
			return res;
		}

		/** A template for the operator += applied to std::vector's having the same number of elements.
			*/
		template <class T>
		void  operator += (std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());

			typename std::vector<T>::iterator			it_a;
			typename std::vector<T>::const_iterator		it_b;

			for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
				*it_a += *it_b;
		}

		/** A template for the operator -= applied to std::vector's having the same number of elements.
			*/
		template <class T>
		void  operator -= (std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());

			typename std::vector<T>::iterator			it_a;
			typename std::vector<T>::const_iterator		it_b;

			for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
				*it_a -= *it_b;
		}

		/** A template for the operator *= applied to std::vector's having the same number of elements.
			*/
		template <class T>
		void  operator *= (std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());

			typename std::vector<T>::iterator			it_a;
			typename std::vector<T>::const_iterator		it_b;

			for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
				*it_a *= *it_b;
		}

		/** A template for the operator /= applied to std::vector's having the same number of elements.
			*/
		template <class T>
		void  operator /= (std::vector<T> &a, const std::vector<T> &b)
		{
			ASSERT_(a.size()==b.size());

			typename std::vector<T>::iterator			it_a;
			typename std::vector<T>::const_iterator		it_b;

			for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
			{
				if (*it_b==0)
					THROW_EXCEPTION("Division by zero: execution aborted!")
				*it_a /= *it_b;
			}
		}

		/** A template for the operator += applied to a std::vector and a scalar.
			*/
		template <class T>
		void  operator += (std::vector<T> &a, T b)
		{
			for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it += b;
		}

		/** A template for the operator -= applied to a std::vector and a scalar.
			*/
		template <class T>
		void  operator -= (std::vector<T> &a, T b)
		{
			for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it -= b;
		}

		/** A template for the operator *= applied to a std::vector and a scalar.
			*/
		template <class T>
		void  operator *= (std::vector<T> &a, T b)
		{
			for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it *= b;
		}

		/** A template for the operator /= applied to a std::vector and a scalar.
			*/
		template <class T>
		void  operator /= (std::vector<T> &a, T b)
		{
			if (b==0)	THROW_EXCEPTION("Division by zero: execution aborted!")
			for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it /= b;
		}

	} // End of math namespace

} // End of mrpt namespace


#endif
