/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CMatrixTemplate_H
#define CMatrixTemplate_H

#include <mrpt/utils/utils_defs.h>
#include <iostream>
#include <iomanip>


namespace mrpt
{
namespace math
{
/**  This template class provides the basic functionality for a general 2D any-size, resizable container of numerical or non-numerical elements.
 * NOTES:
 *		- This class is not serializable since it is a template. Use or declare derived classes for using serialization.
 *		- First row or column index is "0".
 *		- This class includes range checks with ASSERT_() if compiling with "_DEBUG" or "MRPT_ALWAYS_CHECKS_DEBUG_MATRICES=1".
 *		- Please DO NOT use as template class type any other class. It can be safely used the following types:
 *			- Elemental types (int,char,float,doble,...)
 *			- Data struct (Not classes!)
 *			- Any kind of pointers (user is responsible for allocating and freeing the memory addressed by pointers).
 *
 * \sa CMatrixTemplateNumeric
 */
template <class T>
class CMatrixTemplate
{
protected:
	T				**m_Val;
	size_t			m_Rows, m_Cols;

	/** Internal use only: It realloc the memory for the 2D matrix, maintaining the previous contents if posible.
	  */
    void realloc(size_t row, size_t col, bool newElementsToZero = false)
	{
		if (row!=m_Rows || col!=m_Cols || m_Val==NULL)
		{
			size_t	r;
			bool    doZeroColumns   = newElementsToZero && (col>m_Cols);
			size_t	sizeZeroColumns = sizeof(T)*(col-m_Cols);

			// If we are reducing rows, free that memory:
            for (r=row;r<m_Rows;r++)
			{
				free( m_Val[r] );
			}

			// Realloc the vector of pointers:
			if (!row)
					{ ::free(m_Val); m_Val=NULL; }
			else	m_Val = static_cast<T**> (::realloc(m_Val, sizeof(T*) * row ) );

			// How many new rows/cols?
			size_t	row_size = col * sizeof(T);

			// Alloc new ROW pointers & resize previously existing rows, as required:
			for (r=0;r<row;r++)
			{
				if (r<m_Rows)
				{
					// This was an existing row: Resize the memory:
					m_Val[r] = static_cast<T*> (::realloc( m_Val[r], row_size ));

					if (doZeroColumns)
					{
						// Fill with zeros:
						::memset(&m_Val[r][m_Cols],0,sizeZeroColumns);
					}
				}
				else
				{
					// This is a new row, alloc the memory for the first time:
					m_Val[r] = static_cast<T*> (::calloc( row_size, 1 ));
				}
			}

			// Done!
			m_Rows	= row;
			m_Cols	= col;
		}
	}

public:
	/** Returns a given column to a vector (without modifying the matrix)
		* \exception std::exception On index out of bounds
		*/
	void  extractCol(size_t nCol, std::vector<T> &out, int startingRow = 0)
	{
		size_t		i,n;
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG)
		if (nCol>=m_Cols)
			THROW_EXCEPTION("extractCol: Column index out of bounds");
#endif

		n = m_Rows - startingRow;
		out.resize( n );

		for (i=0;i<n;i++)
			out[i] = m_Val[i+startingRow][nCol];
	}

	/** Gets a given column to a vector (without modifying the matrix)
		* \exception std::exception On index out of bounds
		*/
	void  extractCol(size_t nCol, CMatrixTemplate<T> &out, int startingRow = 0)
	{
		size_t		i,n;
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		if (nCol>=m_Cols)
			THROW_EXCEPTION("extractCol: Column index out of bounds");
#endif

		n = m_Rows - startingRow;
		out.setSize(n,1);

		for (i=0;i<n;i++)
			out(i,0) = m_Val[i+startingRow][nCol];
	}

	/** Gets a given row to a vector (without modifying the matrix)
		* \exception std::exception On index out of bounds
		*/
	template <class F>
	void  extractRow(size_t nRow, std::vector<F> &out, size_t startingCol = 0)
	{
		size_t		i,n;
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		if (nRow>=m_Rows)
			THROW_EXCEPTION("extractRow: Row index out of bounds");
#endif
		n = m_Cols - startingCol ;
		out.resize( n );

		for (i=0;i<n;i++)
			out[i] = static_cast<F> ( m_Val[nRow][i+startingCol] );
	}

	/** Gets a given row to a vector (without modifying the matrix)
		* \exception std::exception On index out of bounds
		*/
	template <class F>
	void  extractRow(size_t nRow, CMatrixTemplate<F> &out, size_t startingCol = 0)
	{
		size_t		i,n;
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		if (nRow>=m_Rows)
			THROW_EXCEPTION("extractRow: Row index out of bounds");
#endif
		n = m_Cols - startingCol ;
		out.setSize(1,n);

		for (i=0;i<n;i++)
			out(0,i) = static_cast<F>( m_Val[nRow][i+startingCol] );
	}

	/** Constructors
	*/
	CMatrixTemplate (const CMatrixTemplate& m) : m_Val(NULL),m_Rows(0),m_Cols(0)
	{
		(*this) = m;
	}

	CMatrixTemplate (size_t row = 3, size_t col = 3) :  m_Val(NULL),m_Rows(0),m_Cols(0)
	{
		realloc(row,col);
	}

	/** Constructor from a given size and a C array. The array length must match cols x row.
	  * \code
	  *  const double numbers[] = {
	  *    1,2,3,
	  *    4,5,6 };
	  *	 CMatrixDouble   M(3,2, numbers);
	  * \endcode
	  */
	template <typename V, size_t N>
	CMatrixTemplate (size_t row, size_t col, V (&theArray)[N] ) :  m_Val(NULL),m_Rows(0),m_Cols(0)
	{
		MRPT_COMPILE_TIME_ASSERT(N!=0)
		realloc(row,col);

		if (m_Rows*m_Cols != N)
		{
			THROW_EXCEPTION(format("Mismatch between matrix size %"PRIuPTR"x%"PRIuPTR" and array of length %"PRIuPTR,m_Rows,m_Cols,N))
		}
		size_t	idx=0;
		for (size_t i=0; i < m_Rows; i++)
			for (size_t j=0; j < m_Cols; j++)
				m_Val[i][j] = static_cast<T>(theArray[idx++]);

	}

	/** Destructor
	*/
	virtual ~CMatrixTemplate()
	{
		realloc(0,0);
	}

	/** Assignment operator from another matrix
	  */
	CMatrixTemplate& operator = (const CMatrixTemplate& m)
	{
		realloc( m.m_Rows, m.m_Cols );

		for (size_t i=0; i < m_Rows; i++)
			for (size_t j=0; j < m_Cols; j++)
				m_Val[i][j] = m.m_Val[i][j];

		return *this;
	}

	/** Assignment operator for initializing from a C array (The matrix must be set to the correct size before invoking this asignament)
	  * \code
	  *	 CMatrixDouble   M(3,2);
	  *  const double numbers[] = {
	  *    1,2,3,
	  *    4,5,6 };
	  *  M = numbers;
	  * \endcode
	  *  Refer also to the constructor with initialization data CMatrixTemplate::CMatrixTemplate
	  */
	template <typename V, size_t N>
	CMatrixTemplate& operator = (V (&theArray)[N] )
	{
		MRPT_COMPILE_TIME_ASSERT(N!=0)

		if (m_Rows*m_Cols != N)
		{
			THROW_EXCEPTION(format("Mismatch between matrix size %"PRIuPTR"x%"PRIuPTR" and array of length %"PRIuPTR,m_Rows,m_Cols,N))
		}
		size_t	idx=0;
		for (size_t i=0; i < m_Rows; i++)
			for (size_t j=0; j < m_Cols; j++)
				m_Val[i][j] = static_cast<T>(theArray[idx++]);

		return *this;
	}


	/** Number of rows in the matrix
	  * \sa getRowCount, getColCount, nr, nc
	  */
	inline size_t getRowCount() const
	{
		return m_Rows;
	}

	/** Number of columns in the matrix
	  * \sa getRowCount, getColCount, nr, nc
	 */
	inline size_t getColCount() const
	{
		return m_Cols;
	}

	/** Number of rows in the matrix
	  * \sa getRowCount, getColCount, nr, nc
	  */
	inline size_t nr() const
	{
		return m_Rows;
	}

	/** Number of columns in the matrix
	  * \sa getRowCount, getColCount, nr, nc
	 */
	inline size_t nc() const
	{
		return m_Cols;
	}


	/** Changes the size of matrix, maintaining the previous contents.
	*/
	void setSize(size_t row, size_t col)
	{
		realloc(row,col);
	}

	/** Check for type of matrix
	*/
	bool IsSquare ()
	{
		return ( m_Rows == m_Cols );
	}

	/** Subscript operator to get/set individual elements
		*/
	inline T& operator () (size_t row, size_t col)
	{
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		if (row >= m_Rows || col >= m_Cols)
			THROW_EXCEPTION( format("Indexes (%lu,%lu) out of range!",static_cast<unsigned long>(row),static_cast<unsigned long>(col)) );
#endif
		return m_Val[row][col];
	}

	/** Subscript operator to get/set individual elements
		*/
	inline T operator () (size_t row, size_t col) const
	{
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		if (row >= m_Rows || col >= m_Cols)
			THROW_EXCEPTION( format("Indexes (%lu,%lu) out of range!",static_cast<unsigned long>(row),static_cast<unsigned long>(col)) );
#endif
		return m_Val[row][col];
	}

	/** Subscript operator to get/set an individual element from a row or column matrix.
	  * \exception std::exception If the object is not a column or row matrix.
	  */
	inline T& operator () (size_t ith)
	{
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		ASSERT_(m_Rows==1 || m_Cols==1);
#endif
		if (m_Rows==1)
		{
			// A row matrix:
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
			if (ith >= m_Cols)
				THROW_EXCEPTION_CUSTOM_MSG1( "Index %u out of range!",static_cast<unsigned>(ith) );
#endif
			return m_Val[0][ith];
		}
		else
		{
			// A columns matrix:
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
			if (ith >= m_Rows)
				THROW_EXCEPTION_CUSTOM_MSG1( "Index %u out of range!",static_cast<unsigned>(ith) );
#endif
			return m_Val[ith][0];
		}
	}

	/** Subscript operator to get/set an individual element from a row or column matrix.
	  * \exception std::exception If the object is not a column or row matrix.
	  */
	inline T operator () (size_t ith) const
	{
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
		ASSERT_(m_Rows==1 || m_Cols==1);
#endif
		if (m_Rows==1)
		{
			// A row matrix:
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
			if (ith >= m_Cols)
				THROW_EXCEPTION_CUSTOM_MSG1( "Index %u out of range!",static_cast<unsigned>(ith) );
#endif
			return m_Val[0][ith];
		}
		else
		{
			// A columns matrix:
#if defined(_DEBUG) || (MRPT_ALWAYS_CHECKS_DEBUG_MATRICES)
			if (ith >= m_Rows)
				THROW_EXCEPTION_CUSTOM_MSG1( "Index %u out of range!",static_cast<unsigned>(ith) );
#endif
			return m_Val[ith][0];
		}
	}

	/** Fast but unsafe method to write a value in the matrix
		*/
	inline void set_unsafe(size_t row, size_t col,const T &v)
	{
		m_Val[row][col] = v;
	}

	/** Fast but unsafe method to read a value from the matrix
		*/
	inline T get_unsafe(size_t row, size_t col) const
	{
		return m_Val[row][col];
	}

	/** Fast but unsafe method to obtain a pointer to a given row of the matrix (Use only in time critical applications)
		*/
	inline T* get_unsafe_row(size_t row) const
	{
		return m_Val[row];
	}

	/** Inserts a row from a vector, replacing the current contents of that row.
		* \exception std::exception On index out of bounds
		* \sa extractRow
		*/
	void  insertRow(size_t nRow, const std::vector<T> &in)
	{
		if (nRow>=m_Rows) THROW_EXCEPTION("insertRow: Row index out of bounds");

		size_t n = in.size();
		ASSERT_(m_Cols>=in.size());

		for (size_t i=0;i<n;i++)
			m_Val[nRow][i] = in[i];
	}

	/** Appends a new row to the MxN matrix from a 1xN vector.
		* \exception std::exception On incorrect vector length.
		* \sa extractRow
		*/
	void  appendRow(const std::vector<T> &in)
	{
		size_t		i,n, row;

		n = m_Cols;
		row = m_Rows;
		ASSERT_(in.size()==m_Cols);

		realloc( row+1,n );

		for (i=0;i<n;i++)
			m_Val[row][i] = in[i];
	}

	/** Inserts a column from a vector, replacing the current contents of that column.
		* \exception std::exception On index out of bounds
		* \sa extractCol
		*/
	void  insertCol(size_t nCol, const std::vector<T> &in)
	{
		if (nCol>=m_Cols) THROW_EXCEPTION("insertCol: Row index out of bounds");

		size_t n = in.size();
		ASSERT_( m_Rows >= in.size() );

		for (size_t i=0;i<n;i++)
			m_Val[i][nCol] = in[i];
	}

	/** Inserts a matrix into this matrix
	    *  Notice that the matrix must "fit" into the existing size of this matrix.
	    * \param in The submatrix to insert into 'this' matrix.
	    * \param nRow The row in 'this' matrix where the submatrix will be inserted (0:first).
	    * \param nCol The column in 'this' matrix where the submatrix will be inserted (0:first).
		* \exception std::exception On index out of bounds
		* \sa extractCol
		*/
	template <class R>
	void  insertMatrix(size_t nRow, size_t nCol, const CMatrixTemplate<R> &in)
	{
		size_t	i,j,ncols,nrows;

		nrows = in.m_Rows;
		ncols = in.m_Cols;
		if ( (nRow+nrows > m_Rows) || (nCol+ncols >m_Cols) )
			THROW_EXCEPTION("insertMatrix: Row or Col index out of bounds");

		for (i=nRow;i<nRow+nrows;i++)
			for(j=nCol;j<nCol+ncols;j++)
				set_unsafe(i,j,  static_cast<T> (in.get_unsafe(i-nRow,j-nCol) ) );
	}

	/** Inserts the transpose of a given matrix into this matrix
	    *  Notice that the matrix must "fit" into the existing size of this matrix.
	    * \param in The submatrix to insert into 'this' matrix.
	    * \param nRow The row in 'this' matrix where the submatrix will be inserted (0:first).
	    * \param nCol The column in 'this' matrix where the submatrix will be inserted (0:first).
		* \exception std::exception On index out of bounds
		* \sa extractCol
		*/
	void  insertMatrixTranspose(size_t nRow, size_t nCol, const CMatrixTemplate<T> &in)
	{
		size_t	i,j,ncols,nrows;

		ncols = in.m_Rows;
		nrows = in.m_Cols;
		if ( (nRow+nrows > m_Rows) || (nCol+ncols >m_Cols) )
			THROW_EXCEPTION("insertMatrix: Row or Col index out of bounds");

		for (i=nRow;i<nRow+nrows;i++)
			for(j=nCol;j<nCol+ncols;j++)
				set_unsafe(i,j,  in.get_unsafe(j-nCol,i-nRow) );
	}


	/** Inserts a matrix line (vector) into this matrix
	    *  Notice that the matrix must "fit" into the existing size of this matrix.
		* \exception std::exception On index out of bounds
		* \sa extractCol
		* By AJOGD @ MAR-2007
		*/
	void  insertMatrix(size_t nRow, size_t nCol, const std::vector<T> &in)
	{
		size_t	j,ncols;

		ncols = in.size();
		if ( (nRow+1 > m_Rows) || (nCol+ncols >m_Cols) )
			THROW_EXCEPTION("insertMatrix: Row or Col index out of bounds");

		for(j = nCol ; j < nCol + ncols ; j++)
			set_unsafe(nRow,j, in[j-nCol] );
	}

	/** Inserts 4 matrixes corresponding to the "four corners" into this matrix.
		* \exception std::exception On index out of bounds
		* \sa insertMatrix
		*/
	void  joinMatrix(const CMatrixTemplate<T> &left_up,		const CMatrixTemplate<T> &right_up,
								const CMatrixTemplate<T> &left_down,	const CMatrixTemplate<T> &right_down)
	{
		if ((left_up.getRowCount()!= right_up.getRowCount())||(left_up.getColCount()!=left_down.getColCount())||
			(left_down.getRowCount()!=right_down.getRowCount())||(right_up.getColCount()!=right_down.getColCount()))
			THROW_EXCEPTION("join_Matrix: Row or Col index out of bounds");
		setSize(left_up.getRowCount()+left_down.getRowCount(),left_up.getColCount()+right_up.getColCount());
		insertMatrix(0,0,left_up);
        insertMatrix(0,left_up.getColCount(),right_up);
        insertMatrix(left_up.getRowCount(),0,left_down);
        insertMatrix(left_up.getRowCount(),left_up.getColCount(),right_down);
	}

	/** Fill the matrix with a given value:
	  */
	void  fill( const T &val)
	{
		for (size_t r=0;r<m_Rows;r++)
			for (size_t c=0;c<m_Cols;c++)
				m_Val[r][c]= val;
	}



}; // end of class definition

/** Textual output stream function.
  *    Use only for text output, for example:  "std::cout << mat;"
  */
template <class T>
std::ostream& operator << (std::ostream& ostrm, const CMatrixTemplate<T>& m)
{
	ostrm << std::setprecision(4);

	for (size_t i=0; i < m.getRowCount(); i++)
	{
		for (size_t j=0; j < m.getColCount(); j++)
		{
			ostrm << std::setw(13) << m(i,j);
		}
		ostrm << std::endl;
	}
	return ostrm;
}

	/** Returns the size of the matrix in the i'th dimension: 1=rows, 2=columns (MATLAB-compatible function) */
	template <class T>
	size_t size( const CMatrixTemplate<T>& m, int dim )
	{
		if (dim==1)
			return m.getRowCount();
		else if (dim==2)
			return m.getColCount();
		else THROW_EXCEPTION_CUSTOM_MSG1("size: Matrix dimensions are 1 & 2 only (called with i=%i)",dim);
	}


	} // End of namespace
} // End of namespace

#endif
