/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */


#include <mrpt/core.h>
#define BOOST_ALL_DYN_LINK
#include <boost/program_options.hpp>


using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace mrpt::slam;
using namespace std;

namespace po = boost::program_options;

void do_simulation();
void simulOdometry(const CPose2D & real_pose, CPose2D &last_pose, CPose2D &Apose, const CActionRobotMovement2D::TMotionModelOptions	&odo_opts );

int 	LASER_N_RANGES;
double 	LASER_STD_ERROR;
double 	LASER_BEARING_STD_ERROR;
string	grid_file;
string  gt_file;
string 	out_rawlog_file, in_rawlog_file;

int main(int argc, char ** argv)
{
    try
    {
		// Declare the supported options.
		po::options_description desc("Allowed options");
		desc.add_options()
			("help", "produce help message")
			("grid", po::value<string>(&grid_file), "grid map file (*.gridmap or *.gridmap.gz)")
			("poses", po::value<string>(&gt_file), "poses text file, one 'time x y phi' line per pose")
			("out-rawlog", po::value<string>(&out_rawlog_file), "the output rawlog to generate  from which to take noisy odometry")
			("in-rawlog", po::value<string>(&in_rawlog_file), "(optional) the rawlog from which to take noisy odometry")
			("ranges", po::value<int>(&LASER_N_RANGES)->default_value(361), "number of laser ranges per scan (default=361)")
			("span", po::value<double>()->default_value(180.0), "span of the laser scans (default=180 degrees)")
			("std_r", po::value<double>(&LASER_STD_ERROR)->default_value(0.01), "range noise sigma (default=0.01 meters)")
			("std_b", po::value<double>(&LASER_BEARING_STD_ERROR)->default_value(0.05), "bearing noise sigma (default=0.05 degrees)")
			("nologo", "skip the logo at startup")
		;

		// Parse arguments:
		po::variables_map vm;
		po::store(po::parse_command_line(argc, argv, desc), vm);
		po::notify(vm);

		if (1!=vm.count("nologo"))
		{
			printf(" simul-gridmap version 0.1 - Part of the MRPT\n");
			printf(" MRPT C++ Library: %s - BUILD DATE %s\n", MRPT_getVersion().c_str(), MRPT_getCompilationDate().c_str());
		}

		if (vm.count("help"))
		{
			cout << desc << endl;
			return 1;
		}

		if (1!=vm.count("out-rawlog"))
		{
			cout << "out-rawlog file not especified." << endl;
			cout << desc << endl;
			return 1;
		}

		if (1!=vm.count("grid"))
		{
			cout << "grid file not especified." << endl;
			cout << "Use --help to see usage." << endl;
			return 1;
		}

		if (1!=vm.count("poses"))
		{
			cout << "poses file not especified." << endl;
			cout << "Use --help to see usage." << endl;
			return 1;
		}

		/*if (1!=vm.count("in-rawlog"))
		{
			cout << "Sorry, in-rawlog must be given, is not optional yet!" << endl;
			cout << "Use --help to see usage." << endl;
			return 1;
		}*/


		// pass deg 2 rads:
		LASER_BEARING_STD_ERROR = DEG2RAD(LASER_BEARING_STD_ERROR);

		// Invoke method:
		do_simulation();

		return 0;
	}
	catch(std::exception &e)
	{
		cerr << e.what() << endl;
		return -1;
	}	
	catch(...)
	{
		cerr << "Untyped exception." << endl;
		return -1;
	}	
}



void do_simulation()
{
	ASSERT_( mrpt::system::fileExists(gt_file) );

	bool have_in_rawlog;
	if (!in_rawlog_file.empty())  
	{
		ASSERT_( mrpt::system::fileExists(in_rawlog_file) );
		have_in_rawlog = true;
	}
	else have_in_rawlog = false;

	ASSERT_( mrpt::system::fileExists(grid_file) );

	// Load the grid:
	COccupancyGridMap2D the_grid;
	CFileGZInputStream(grid_file) >> the_grid;

	// GT file rows are:
	//  time x y z
	CMatrixDouble	GT;
	GT.loadFromTextFile(gt_file);

	// Rawlog:
	CRawlog  rawlog;
	if (have_in_rawlog)
	{
		rawlog.loadFromRawLogFile(in_rawlog_file);
	    ASSERT_( rawlog.size()>0 );
	    ASSERT_( rawlog.getType(0) == CRawlog::etActionCollection );
		ASSERT_( rawlog.size()/2 == GT.getRowCount() );
	}

	// # of simulation steps:
	size_t  N = GT.getRowCount();

	// Assert sizes:
    ASSERT_( GT.getColCount() >= 4 );

    // ------------------------------------------
    // Resimulate scans:
    // ------------------------------------------
	if (have_in_rawlog)
	{
		// Modify old rawlog in-place:
		for (size_t i=1;i<rawlog.size();i+=2)
		{
    		ASSERT_( rawlog.getType(i) == CRawlog::etSensoryFrame );

    		CSensoryFramePtr sf = rawlog.getAsObservations(i);
    		CPose2D  gt_pose( GT(i/2,1),GT(i/2,2),GT(i/2,3) );

			CObservation2DRangeScanPtr the_scan = sf->getObservationByClass<CObservation2DRangeScan>();
			the_grid.laserScanSimulator( *the_scan, gt_pose, 0.5f, LASER_N_RANGES, LASER_STD_ERROR, 1, LASER_BEARING_STD_ERROR );
		}
	}
	else
	{
		rawlog.clear();
		
		CActionCollection	acts;
		CActionRobotMovement2D	act;
		CActionRobotMovement2D::TMotionModelOptions	odo_opts;

		odo_opts.modelSelection = CActionRobotMovement2D::mmGaussian;
		//odo_opts.gausianModel.

		CPose2D	last_pose( GT(0,1),GT(0,2),GT(0,3) );
		CPose2D	real_pose = last_pose;
		CPose2D Apose;

		simulOdometry(real_pose,last_pose,Apose, odo_opts);
		act.computeFromOdometry( CPose2D(0,0,0), odo_opts );
		act.timestamp = mrpt::system::now();
		acts.insert(act);
		rawlog.addActions(acts);

		// Create a rawlog from scratch:
		for (size_t i=1;i<N;i++)
		{
			// simulate scan:
    		real_pose = CPose2D( GT(i,1),GT(i,2),GT(i,3) );

			CSensoryFramePtr sf = CSensoryFramePtr( new CSensoryFrame());

			CObservation2DRangeScanPtr the_scan = CObservation2DRangeScanPtr( new CObservation2DRangeScan() );
			the_scan->aperture = M_PI;
			the_scan->timestamp = mrpt::system::now();
			the_grid.laserScanSimulator( *the_scan, real_pose, 0.5f, LASER_N_RANGES, LASER_STD_ERROR, 1, LASER_BEARING_STD_ERROR );
			sf->insert(the_scan);
			rawlog.addObservationsMemoryReference(sf);

			// Robot moves:
			simulOdometry(real_pose,last_pose,Apose, odo_opts);
			act.computeFromOdometry( CPose2D(0,0,0), odo_opts );
			act.timestamp = mrpt::system::now();
			acts.clear();
			acts.insert(act);
			rawlog.addActions(acts);

		}
	}

    // Save the new rawlog:
    rawlog.saveToRawLogFile(out_rawlog_file);
}


void simulOdometry(
	const CPose2D	&real_pose, 
	CPose2D			&last_pose, 
	CPose2D			&Apose, 
	const CActionRobotMovement2D::TMotionModelOptions	&odo_opts )
{
	Apose = real_pose-last_pose; 
	last_pose = real_pose;
	
	// Noise:
	Apose.x += mrpt::random::normalizedGaussian() * odo_opts.gausianModel.minStdXY;
	Apose.y += mrpt::random::normalizedGaussian() * odo_opts.gausianModel.minStdXY;
	Apose.phi += mrpt::random::normalizedGaussian() * odo_opts.gausianModel.minStdPHI;
	Apose.normalizePhi();
}


