/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */


/*-----------------------------------------------------------------------------
	APPLICATION: RawLogGrabber
	FILE: rawloggrabber_main.cpp
	AUTHOR: Jose Luis Blanco Claraco <jlblanco@ctima.uma.es>

	For instructions and details, see:
	 http://babel.isa.uma.es/mrpt/index.php/Application:RawLogGrabber
  -----------------------------------------------------------------------------*/

#include <mrpt/core.h>
#include <mrpt/hwdrivers.h>

using namespace mrpt;
using namespace mrpt::system;
using namespace mrpt::hwdrivers;
using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace std;


#define GLOBAL_SECTION_NAME			"global"

// Forward declarations:
void SensorThread(void *ptr);

struct TThreadParams
{
	CConfigFile		*cfgFile;
	string			sensor_label;
};


CGenericSensor::TListObservations		global_list_obs;
synch::CCriticalSection					cs_global_list_obs;

bool									allThreadsMustExit = false;

string 		rawlog_ext_imgs_dir;		// Directory where to save externally stored images, only for CCameraSensor's.

// Register existing sensors:
void doRegistrations()
{
	CSickLaserUSB::doRegister();
	CHokuyoURG::doRegister();
	CGPSInterface::doRegister();
	CBoardDLMS::doRegister();
	CIMUXSens::doRegister();
	CCameraSensor::doRegister();
	CActivMediaRobotBase::doRegister();
}

// ------------------------------------------------------
//					MAIN THREAD
// ------------------------------------------------------
int main(int argc, char **argv)
{
	try
	{
		printf(" rawlog-grabber version 0.2 - Part of the MRPT\n");
		printf(" MRPT C++ Library: %s - BUILD DATE %s\n", MRPT_getVersion().c_str(), MRPT_getCompilationDate().c_str());
		printf("-------------------------------------------------------------------\n");

		// Process arguments:
		if (argc<2)
		{
			printf("Usage: %s <config_file.ini>\n\n",argv[0]);
			mrpt::system::pause();
			return -1;
		}

		string INI_FILENAME( argv[1] );
		ASSERT_(fileExists(INI_FILENAME));

		CConfigFile		iniFile( INI_FILENAME );

		// ------------------------------------------
		//			Load config from file:
		// ------------------------------------------
		string			rawlog_prefix = "dataset";
		int				time_between_launches = 300;
		double			SF_max_time_span = 0.25;			// Seconds
		bool			use_sensoryframes = false;
		int				GRABBER_PERIOD_MS = 1000;

		MRPT_LOAD_CONFIG_VAR( rawlog_prefix, string, iniFile, GLOBAL_SECTION_NAME );
		MRPT_LOAD_CONFIG_VAR( time_between_launches, int, iniFile, GLOBAL_SECTION_NAME );
		MRPT_LOAD_CONFIG_VAR( SF_max_time_span, float,		iniFile, GLOBAL_SECTION_NAME );
		MRPT_LOAD_CONFIG_VAR( use_sensoryframes, bool,		iniFile, GLOBAL_SECTION_NAME );
		MRPT_LOAD_CONFIG_VAR( GRABBER_PERIOD_MS, int, iniFile, GLOBAL_SECTION_NAME );

		// Build full rawlog file name:
		string	rawlog_postfix = "_";
		rawlog_postfix += dateTimeToString( now() );

		// Only set this if we want externally stored images:
		rawlog_ext_imgs_dir = rawlog_prefix+fileNameStripInvalidChars( rawlog_postfix+string("_Images") );

		rawlog_postfix += string(".rawlog");
		rawlog_postfix = fileNameStripInvalidChars( rawlog_postfix );

		string			rawlog_filename = rawlog_prefix + rawlog_postfix;

		cout << endl ;
		cout << "Output rawlog filename: " << rawlog_filename << endl;
		cout << "External image storage: " << rawlog_ext_imgs_dir << endl << endl;

		// Sets directory for external image grabbing:
		CMRPTImage::IMAGES_PATH_BASE = rawlog_ext_imgs_dir;

		// ----------------------------------------------
		// Launch threads:
		// ----------------------------------------------
		doRegistrations();

		vector_string	sections;
		iniFile.getAllSections( sections );

		vector<TThreadHandle>		lstThreads;

		for (vector_string::iterator it=sections.begin();it!=sections.end();it++)
		{
			if (*it==GLOBAL_SECTION_NAME || it->empty()) continue;	// This is not a sensor:

			//cout << "Launching thread for sensor '" << *it << "'" << endl;

			TThreadParams	*threParms = new TThreadParams[1];
			threParms->cfgFile		= &iniFile;
			threParms->sensor_label	= *it;

			TThreadHandle	thre = createThread(SensorThread, (void*)threParms);

			lstThreads.push_back(thre);
			sleep(time_between_launches);
		}

		// ----------------------------------------------
		// Run:
		// ----------------------------------------------
		CFileGZOutputStream	out_file( rawlog_filename );

		CSensoryFrame						curSF;
		CGenericSensor::TListObservations	copy_of_global_list_obs;

		cout << endl << "Press any key to exit program" << endl;
		while (!os::kbhit() && !allThreadsMustExit)
		{
			// See if we have observations and process them:
			{
				synch::CCriticalSectionLocker	lock (&cs_global_list_obs);
				copy_of_global_list_obs.clear();

				if (!global_list_obs.empty())
				{
					CGenericSensor::TListObservations::iterator itEnd = global_list_obs.begin();
					std::advance( itEnd, global_list_obs.size() / 2 );
					copy_of_global_list_obs.insert(global_list_obs.begin(),itEnd );
					global_list_obs.erase(global_list_obs.begin(), itEnd);
				}
			}	// End of cs lock

			if (use_sensoryframes)
			{
				// -----------------------
				// USE SENSORY-FRAMES
				// -----------------------
				for (CGenericSensor::TListObservations::iterator it=copy_of_global_list_obs.begin();it!=copy_of_global_list_obs.end();it++)
				{
					// If we have an action, save the SF and start a new one:
					if (it->second->GetRuntimeClass()->derivedFrom( CLASS_ID(CAction) ) )
					{
						CActionPtr act = CActionPtr( it->second);

						out_file << curSF;
						cout << "[" << dateTimeToString(now()) << "] Saved SF with " << curSF.size() << " objects." << endl;
						curSF.clear();

						CActionCollection	acts;
						acts.insert(*act);
						act.clear_unique();

						out_file << acts;
					}
					else
					if (it->second->GetRuntimeClass()->derivedFrom( CLASS_ID(CObservation) ) )
					{
						CObservationPtr obs = CObservationPtr(it->second);
						curSF.insert( obs );

						if ( timeDifference( curSF.getObservationByIndex(0)->timestamp, obs->timestamp ) > SF_max_time_span )
						{
							// Show GPS mode:
							CObservationGPSPtr gps;
							size_t idx=0;
							do
							{
								gps = curSF.getObservationByClass<CObservationGPS>(idx++ );
								if (gps)
								{
									cout << "  GPS mode: " << (int)gps->GGA_datum.fix_quality << " label: " << gps->sensorLabel << endl;
								}
							} while (gps);

							// Show IMU angles:
							CObservationIMUPtr imu = curSF.getObservationByClass<CObservationIMU>();
							if (imu)
							{
								cout << format("   IMU angles (degrees): (yaw,pitch,roll)=(%.06f, %.06f, %.06f)",
									RAD2DEG( imu->rawMeasurements[IMU_YAW] ),
									RAD2DEG( imu->rawMeasurements[IMU_PITCH] ),
									RAD2DEG( imu->rawMeasurements[IMU_ROLL] ) ) << endl;
							}

							// Save and start a new one:
							out_file << curSF;
							cout << "[" << dateTimeToString(now()) << "] Saved SF with " << curSF.size() << " objects." << endl;
							curSF.clear();
						}
					}
					else THROW_EXCEPTION("*** ERROR *** Class is not an action or an observation");
				}
			}
			else
			{
				// ---------------------------
				//  DO NOT USE SENSORY-FRAMES
				// ---------------------------
				CObservationIMUPtr imu; // Default:NULL

				for (CGenericSensor::TListObservations::iterator it=copy_of_global_list_obs.begin();it!=copy_of_global_list_obs.end();it++)
				{
					out_file << *(it->second);

					// Show GPS mode:
					if ( (it->second)->GetRuntimeClass() == CLASS_ID(CObservationGPS) )
					{
						CObservationGPSPtr gps = CObservationGPSPtr( it->second );
						cout << "  GPS mode: " << (int)gps->GGA_datum.fix_quality << " label: " << gps->sensorLabel << endl;
					}
					else if ( (it->second)->GetRuntimeClass() == CLASS_ID(CObservationIMU) )
					{
						imu = CObservationIMUPtr( it->second );
					}
				}

				// Show IMU angles:
				if (imu)
				{
					cout << format("   IMU angles (degrees): (yaw,pitch,roll)=(%.06f, %.06f, %.06f)",
						RAD2DEG( imu->rawMeasurements[IMU_YAW] ),
						RAD2DEG( imu->rawMeasurements[IMU_PITCH] ),
						RAD2DEG( imu->rawMeasurements[IMU_ROLL] ) ) << endl;
				}


				if (!copy_of_global_list_obs.empty())
				{
					cout << "[" << dateTimeToString(now()) << "] Saved " << copy_of_global_list_obs.size() << " objects." << endl;
				}
			}

			sleep(GRABBER_PERIOD_MS);
		}

		if (allThreadsMustExit)
		{
			cerr << "[main thread] Ended due to other thread signal to exit application." << endl;
		}

		// Wait all threads:
		// ----------------------------
		allThreadsMustExit = true;
		mrpt::system::sleep(300);
		cout << endl << "Waiting for all threads to close..." << endl;
		for (vector<TThreadHandle>::iterator th=lstThreads.begin();th!=lstThreads.end();th++)
			joinThread( *th );

		return 0;
	} catch (std::exception &e)
	{
		std::cerr << e.what() << std::endl << "Program finished for an exception!!" << std::endl;
		mrpt::system::pause();
		return -1;
	}
	catch (...)
	{
		std::cerr << "Untyped exception!!" << std::endl;
		mrpt::system::pause();
		return -1;
	}
}


// ------------------------------------------------------
//				SensorThread
// ------------------------------------------------------
void SensorThread(void *ptr)
{
    TThreadParams   *params_ptr = static_cast<TThreadParams*>(ptr);
	TThreadParams	params = *params_ptr;
	delete[] params_ptr; params_ptr=NULL;

	try
	{
		string driver_name = params.cfgFile->read_string(params.sensor_label,"driver","",true);

		CGenericSensorPtr	sensor = CGenericSensorPtr( CGenericSensor::createSensor(driver_name ) );
		if (!sensor)
		{
			cerr << endl << "***ERROR***: Class name not recognized: " << driver_name << endl;
			allThreadsMustExit = true;
		}

		// Load common parameters:
		float	process_rate=1;
		int		max_queue_len = 100;

		MRPT_LOAD_CONFIG_VAR_NO_DEFAULT( process_rate, float,	(*params.cfgFile), params.sensor_label );
		MRPT_LOAD_CONFIG_VAR(max_queue_len, int,				(*params.cfgFile), params.sensor_label );

		cout << format("[thread_%s] Starting...",params.sensor_label.c_str()) << " at " << process_rate <<  " Hz" << endl;

		ASSERT_(process_rate>0);
		int		process_period_ms = round( 1000.0 / process_rate );

		// Load sensor config params:
		sensor->loadConfig( *params.cfgFile, params.sensor_label );

		// For imaging sensors, set external storage directory:

		if ( !rawlog_ext_imgs_dir.empty() &&  SENSOR_IS_CLASS(sensor, CCameraSensor ) )
		{
			CCameraSensor* camSens = dynamic_cast<CCameraSensor*>( sensor.pointer() );
			camSens->setPathForExternalImages( rawlog_ext_imgs_dir );
		}

		// Init device:
		sensor->initialize();


		while (! allThreadsMustExit )
		{
			TTimeStamp t0= now();

			// Process
			sensor->doProcess();

			// Get new observations
			CGenericSensor::TListObservations	lstObjs;
			sensor->getObservations( lstObjs );

			{
				synch::CCriticalSectionLocker	lock (&cs_global_list_obs);
				global_list_obs.insert( lstObjs.begin(), lstObjs.end() );
			}

			lstObjs.clear();

			// wait until the process period:
			TTimeStamp t1= now();
			double	At = timeDifference(t0,t1);
			int At_rem_ms = process_period_ms - At*1000;
			if (At_rem_ms>0)
				sleep(At_rem_ms);
		}

		sensor.clear();
		cout << format("[thread_%s] Closing...",params.sensor_label.c_str()) << endl;
	}
	catch (std::exception &e)
	{
		cerr << e.what() << endl;
		allThreadsMustExit = true;
	}
	catch (...)
	{
		cerr << "Untyped exception!!" << endl;
		allThreadsMustExit = true;
	}
}
