/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#pragma once

#include <highgui.h>

typedef enum _CalibState
{
    CalibState_Initial            = 0,
    CalibState_NotCalibrated      = 1,
    CalibState_Calibrated         = 2,
    CalibState_CalibrationProcess = 3
}
CalibState;

typedef enum _CalibEtalonType
{
    CalibEtalon_ChessBoard = 0
}
CalibEtalonType;

typedef struct _CvCameraParams
{
    float   focalLength[2];
    float   distortion[4];
    //float   principalPoint[2];
    CvPoint2D32f  principalPoint;
    float   matrix[9];
    float   rotMatr[9];
    float   transVect[3];
}
CvCameraParams;

typedef struct _CalibFilterParams
{
    CalibEtalonType  etalon_type;
    long        etalon_params_count;
    float       etalon_params[7];
    long        etalon_points;
    long        show_feature_points_flag;
    long        frame_interval;
    long        frames_to_collect;
    long        frames_collected;
    long        frames_passed;
	double      last_frame_time;
    CalibState  calib_state;
    long        enable_undistortion;
    long        show_3d_window;
}
CalibFilterParams;

class CCalibFilter
{
public:
    CCalibFilter();
    ~CCalibFilter();

    long getEtalonParams(CalibEtalonType* etalon_type, float* etalon_params, long* etalon_param_count);
    long getFrameInterval(long* count);
    long getFramesToCollect(long* frames);
    long getEnableUndistortion(long* enable);
    long getTrackEtalon(long* enable);

    long setEtalonParams(CalibEtalonType etalon_type, float* etalon_params, long etalon_param_count);
    long setFrameInterval(long count);
    long setFramesToCollect(long frames);
    long setEnableUndistortion(long enable);
    long setTrackEtalon(long enable);

    long getCameraParams(CvCameraParams *camera);
    long getState(CalibState *calib_state, long *frames_collected, long *frames_passed, double *last_frame_time);

////////////////////////////////
    void reset()
    {
	m_params.calib_state = CalibState_NotCalibrated;
    }

////////////////////////////////

    long StartCalibrate();
    long StopCalibrate();

////////////////////////////////

    long SaveCameraParams(const char *fileName);
    long LoadCameraParams(const char *fileName);

////////////////////////////////

	long ProcessFrame(IplImage* frame);

private:

	long Transform();
    void CheckReallocBuffers( IplImage* rgb_img );
    void ProcessFrame( IplImage* rgb_img, double frame_time );
    void CalculateCameraParams( CvSize img_size );
    void DrawEtalon( IplImage* rgb_img, CvPoint2D32f* corners,
                       int corner_count, CvSize etalon_size,
                       int draw_ordered );
    void FillEtalonObjPoints( CvPoint3D32f* obj_points,
                                CvSize etalon_size,
                                float square_size );

    CvSize GetEtalonSize();

    CalibFilterParams  m_params, m_initial_params;

    /* calibration buffers */
    int                m_max_points;
    CvPoint2D32f*      m_imagePoints;
    CvPoint3D32f*      m_objectPoints;
    CvPoint3D32f*      m_transVects;
    float*             m_rotMatrs;
    int*               m_numsPoints;

    /* temporary images, used in processing */
    IplImage*         m_gray_img;
    IplImage*         m_thresh_img;
    IplImage*         m_undist_img;

    /* processed frame */
    IplImage*         m_rgb_img;

    /* camera parameters */
    CvCameraParams   m_camera;

    /* undistortion data */
    CvCameraParams   m_undistort_params;
    IplImage*        m_undistort_data;

}; /* CCalibFilter */

/* End of file. */

