#ifndef DICTIONARY_H_
#define DICTIONARY_H_

#include <map>
#include <vector>
#include <string>

#include <wx/defs.h>

namespace jcs {
  
  const wxUint32 UNDEFINED = 0xFFFFFFFF; // Undefined value length
  const std::string fnDicom = "dicom.txt";
  const std::string fnExcite = "excite.txt";
  const std::string fnNumaris = "numaris.txt";
  const std::string fnPhilips = "philips_mr.txt";

  struct DicomTag {
    
    DicomTag() {}
    DicomTag(wxUint16 g, wxUint16 e) : group(g), element(e) {}
    
    wxUint16 group;
    wxUint16 element;
    
    bool operator== (const DicomTag &rhs) const;
    bool operator!= (const DicomTag &rhs) const;
    bool operator<  (const DicomTag &rhs) const;
    bool operator>  (const DicomTag &rhs) const;
  };

  std::ostream &operator<<( std::ostream &out, const DicomTag &rhs );

  struct DicomElement {
  
    DicomElement() {}
    DicomElement(const DicomTag& tag,
		 const char* desc = "", 
		 const char* vr = "UN") :
      tag(tag), description(desc), vr(vr) {}
    
    DicomTag tag;
    std::string vr;
    wxUint32 value_length;
    std::string value;
    std::string description;
  };
  
  class Dictionary {

  public:
    const DicomElement Lookup(const DicomTag& tag) const;
    const DicomElement Lookup(const std::string& desc) const;

    std::vector<std::string> TagList() const;

  protected:
    Dictionary(const std::string& filename);
    Dictionary(const Dictionary&);
    Dictionary& operator= (const Dictionary&);

  private:
    typedef std::map<DicomTag, DicomElement> TagMap;
    typedef std::map<std::string, DicomElement> StringMap;

    std::string mFilename;
    TagMap mTagMap;
    StringMap mStringMap;

    DicomElement NULL_ENTRY;
    void Init();
    void mAddEntry(const DicomElement& e);

  };

  class Dicom_Dictionary : public Dictionary
  {
  public:
    static Dictionary* Instance();

  protected:
    Dicom_Dictionary():Dictionary(fnDicom) {}

  };

  class Numaris_Dictionary : public Dictionary
  {
  public:
    static Dictionary* Instance();

  protected:
    Numaris_Dictionary():Dictionary(fnNumaris) {}

  };

  class Excite_Dictionary : public Dictionary
  {
  public:
    static Dictionary* Instance();

  protected:
    Excite_Dictionary():Dictionary(fnExcite) {}

  };

  class PhilipsMr_Dictionary : public Dictionary
  {
  public:
    static Dictionary* Instance();

  protected:
    PhilipsMr_Dictionary():Dictionary(fnPhilips) {}

  };


  DicomTag Tag(const std::string& s, 
	       const Dictionary* dict = Dicom_Dictionary::Instance());

}

#endif
