/***
	* Datastore class
	* 
	* handles persistent data and lookup tables
	*/
var su_Datastore = function (enabledb, host)
{
	this._components = Components;
	
	this.globals = new Object();
	this.globals.messageLog = "";
	this.globals.messageCount = 0;
	
	// The following in-memory hashtables help minimize the data that
	// must be stored persistently.  They could be replaced when we 
	// eventually eliminate prefs tables in favor of db tables.
	//   sls: array of slstats objects that are used to build $slstats;
	//        some stats get updated incrementally when we do sequential
	//        10-url hits to links.php
	// sluqh: hash of url->slq that indicates to the content click 
	//        handler that a page contains tracked links
	// sltih: hash of slt->sli, where slt is a query-specific id of
	//        tracked link targets and where sli is a record of tracking
	//        details
	this.globals.sls = new Array();
	this.globals.sluqh = new Object();
	this.globals.sltih = new Object();
	
	this.prefRetries = 1000;
	
	this._SCRIPTABLE_INPUT_STREAM_C = new Components.Constructor(
				"@mozilla.org/scriptableinputstream;1",
				"nsIScriptableInputStream");

	this._NSIFILE_PATH_C = new Components.Constructor(
				"@mozilla.org/file/local;1",
				"nsILocalFile",
				"initWithPath");
	
	this._prefAccessInitialized = false;
	this._prefService = null;
	this._prefBranch = null;
	this._initPrefAccess();
	this.prefsDirty = false;
	
	try {
		// When we start using this, we need to initialize similarly to
		// prefs, logging errors and retrying as necessary.  -- JW
		var localeService =
					Components.classes["@mozilla.org/intl/nslocaleservice;1"]
					.getService(Components.interfaces.nsILocaleService);
		var stringBundleService =
					Components.classes["@mozilla.org/intl/stringbundle;1"]
					.getService(Components.interfaces.nsIStringBundleService);
		this._stringBundle = stringBundleService.createBundle(
					"chrome://stumbleupon/locale/stumbleupon.properties",
					localeService.getApplicationLocale());
	} catch (e) {}
	
	this._dicts = new Object();
	this._initConstDictionaries();
	this._host = host;
	this._enabledb = enabledb;
	
	this.userid = this.getValue("@current_user");
	this._userdb = null;
	
	if (this._enabledb)
		this._initUserDB();
	
//	this._queuedFileWriteSpecs = new Array();

	this.eventListenerListsByEventId = new Object();

	this._batchedErrorLog = "";
	this._batchedPrefErrorLog = "";
	this._startupTimeMs = (new Date()).getTime();
}

// static properties
su_Datastore._instance = null;

// static methods
su_Datastore.getService = function ()
{
	// the datastore is a singleton across all browsers

	if (! window.su_Datastore._instance)
	{
		var enumerator =
				Components.classes["@mozilla.org/appshell/window-mediator;1"]
				.getService(Components.interfaces.nsIWindowMediator)
				.getEnumerator("navigator:browser");
		
		var win = window;
		while (enumerator.hasMoreElements())
		{
			var winTmp = enumerator.getNext();
			if (winTmp != window)
				win = winTmp;
		}
		if (win == window)
			window.su_Datastore._instance = new su_Datastore(window.su_enable_db, window.su_host);
		else
			window.su_Datastore._instance = win.su_Datastore._instance;
	}

	return window.su_Datastore._instance;
}
su_Datastore.prototype =
{ // BEGIN prototype

_initUserDB: function ()
{
	if (this.userid == "")
	{
		this._userdb = null;
		return;
	}
	
	this._userdb = new su_DatabaseConnection(this.userid, this._host);
	
	var dbfile = this._userdb.getDBFile();
	
	if (! dbfile.exists())
	{
		var nsiuri = this._createHostInstance(
					"@mozilla.org/network/standard-url;1",
					"nsIURI");
		nsiuri.spec = "chrome://stumbleupon/content/userdb.sql";
		var sql = this.readURI(nsiuri);
		this._userdb.beginTransaction();
		this._userdb.query("PRAGMA auto_vacuum = 1");
		this._userdb.query(sql);
		this._userdb.commitTransaction();
	}
	this._userdb.startDummyStatement();
},

_initPrefAccess: function ()
{
	try {
		this._prefService =
					Components.classes["@mozilla.org/preferences-service;1"]
					.getService(Components.interfaces.nsIPrefService);
	
		this._prefBranch = this._prefService.getBranch("");
		
		this._prefAccessInitialized = true;
	} catch (e) { this._logPrefError("PREFS INIT", e); }
},

_createHostInstance: function (nsclass, nsinterface)
{
	try {
		return this._components.classes[nsclass]
					.createInstance(this._components.interfaces[nsinterface]);
	} catch (e) {
		return null;
	}
},

_getHostService: function (nsclass, nsinterface)
{
	try {
		return this._components.classes[nsclass]
					.getService(this._components.interfaces[nsinterface]);
	} catch (e) {
		return null;
	}
},

// creates an object from a json string
deserialize: function (str)
{
	// This method is derived from the json.org implementation:
	// http://www.json.org/js.html
	// -- JW

	try {
		if (/^("(\\.|[^"\\\n\r])*?"|[,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t])+?$/.
					test(str))
			return eval('(' + str + ')');
	} catch (e) {}
	return null;
},

// creates a json string from an object
serialize: function (obj, include_linefeeds)
{
	return this._JSONRecurse(obj, include_linefeeds).join('');
},

_JSONRecurse: function (arg, lf, out)
{
	// [IP:] [kudos:] This method is derived from the TrimPath
	// implementation:
	// http://trimpath.com/project/wiki/JsonLibrary
	// -- JW
/*
Copyright (c) 2002 JSON.org

Permission is hereby granted, free of charge, to any person obtaining a copy 
of this software and associated documentation files (the "Software"), to deal 
in the Software without restriction, including without limitation the rights 
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
copies of the Software, and to permit persons to whom the Software is 
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all 
copies or substantial portions of the Software.

The Software shall be used for Good, not Evil.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
SOFTWARE.
*/
	out = out || new Array();
	var u; // undefined
	
	switch (typeof arg) {
		case 'object':
			if (! arg)
			{
				out.push('"');
			}
			else if (arg.constructor == Array)
			{
				out.push('[');
				var i;
				for (i = 0; i < arg.length; ++i)
				{
					if (i > 0)
					{
						if (lf)
							out.push(',\n');
						else
							out.push(',');
					}
					this._JSONRecurse(arg[i], lf, out);
				}
				out.push(']');
			}
			else if (typeof arg.toString != 'undefined')
			{
				out.push('{');
				var first = true;
				var p;
				for (p in arg)
				{
					if ((typeof (arg[p])) == "function") continue;
					var curr = out.length; // Record position to allow undo when arg[p] is undefined.
					if (! first)
						if (lf)
							out.push(',\n');
						else
							out.push(',');
					this._JSONRecurse(p, lf, out);
					out.push(':');                    
					this._JSONRecurse(arg[p], lf, out);
					if (out[out.length - 1] == u)
						out.splice(curr, out.length - curr);
					else
						first = false;
				}
				out.push('}');
			}
			break;
		case 'unknown':
		case 'undefined':
		case 'function':
			out.push(u);
			break;
		case 'string':
			out.push('"');
			out.push(arg.replace(/(["\\])/g, '\\$1').replace(/\r/g, '').replace(/\n/g, '\\n'));
			out.push('"');
			break;
		default:
			out.push(String(arg));
			break;
	}
	return out;
},

_validateFilepathComponent: function (str)
{
	// Disallow characters that have potential to be abused in the case
	// of dns poisoning. -- JW
	var strtmp = str.replace(/[\\\/%\$]/g, "");
	var strtmp = strtmp.replace("..", ".");
	return (str == strtmp);
},

_expandKey: function (key)
{
	if (key.charAt(0) == "$")
		return "stumble." + this.userid + "." + key.substr(1);
	else if (key.charAt(0) == "@")
		return "stumble." + key.substr(1);
	else
		return key;
},

_getPrefTypeFromValue: function (val)
{
	var type;
	switch (typeof val)
	{
		case "string":  type = "Char"; break;
		case "boolean": type = "Bool"; break;
		case "number":  type = "Int";  break;
		case "object":  type = "JSON"; break;
		default:        type = null;   break;
	}
	return type;
},

_verifyPrefAccess: function ()
{
	if (this._prefAccessInitialized)
		return;
	
	this._initPrefAccess();
},

getGlobalValue: function (id, optionalOverrideDefault)
{
	if (typeof(this.globals[id.substr(1)]) == "undefined")
	{
		if (typeof(optionalOverrideDefault) == "undefined")
			return this.getDefaultValue(id);
		else
			return optionalOverrideDefault;
	}
	else
	{
		return this.globals[id.substr(1)];
	}
},

// Use getValue unless you need to override the default.
getPrefValue: function (id, optionalOverrideDefault)
{
	var type;
	if ((typeof optionalOverrideDefault) == "undefined")
		type = this.getPrefType(id);
	else
		type = this._getPrefTypeFromValue(optionalOverrideDefault);
	
	var jsonFlag = false;
	if (type == "JSON")
	{
		jsonFlag = true;
		type = "Char";
	}
	else if (type == null)
	{
		var e = new Object();
		e.message = "getPrefValue : key [" + id + "] is not defined";
		e.fileName = "chrome://stumbleupon/content/datastore.js";
		e.toString = function (){ return e.fileName + " " + e.message; }
		throw(e);
	}
	
	var value;
	if (this.isPrefDefined(id))
	{
		// get
		var key = this._expandKey(id);
		
		var attemptCount = 0;
		var success = false;
		var error = new Object();
		while ((! success) && (attemptCount < this.prefRetries))
		{
			attemptCount++;
			
			this._verifyPrefAccess();
		
			try {
				eval("value = this._prefBranch.get" + type + "Pref(key)");
				success = true;
			}
			catch (e) {
//				error = e;
			}
		}
		
		if (success)
		{
			if (jsonFlag)
				value = this.deserialize(value);
		}
		else
		{
//			this._logError(true, "PREFS GET", error, id);
			if ((typeof optionalOverrideDefault) == "undefined")
				value = this.getDefaultValue(id);
			else
				value = optionalOverrideDefault;
		}
	}
	else
	{
		// get or set
		if ((typeof optionalOverrideDefault) == "undefined")
			value = this.getDefaultValue(id);
		else
			value = optionalOverrideDefault;
		
		this.setValue(id, value);
	}
	
	return value;
},

getPrefType: function (id)
{
	if ((id.charAt(0) == "@") || (id.charAt(0) == "$"))
	{
		if ((typeof this._DEFAULTS_BY_ID[id]) != "undefined")
			return this._getPrefTypeFromValue(this._DEFAULTS_BY_ID[id]);
		else
			return null;
	}
	else
	{
		var attemptCount = 0;
		var success = false;
		var error = new Object();
		var type = null;
		while ((! success) && (attemptCount < this.prefRetries))
		{
			attemptCount++;
			
			this._verifyPrefAccess();
		
			try {
				type = this._prefBranch.getPrefType(id)
				success = true;
			}
			catch (e) {
				error = e;
			}
		}
		
		if (success)
		{
			switch (type)
			{
				case this._prefBranch.PREF_STRING: return "Char";
				case this._prefBranch.PREF_INT:    return "Int";
				case this._prefBranch.PREF_BOOL:   return "Bool";
				default:                           return null;
			}
		}
		else
		{
			this._logError(true, "PREF TYPE", error, id)
			return null;
		}
	}
},

// Use this to get preference and locale values.  The first character 
// of param id has the following special meanings:
// $           -> current user value      (ex: "$nick")
// @           -> client value            (ex: "@current_user")
// #           -> global value            (ex: "#checked_dyn_channels")
// %           -> locale string           (ex: "%menu.anytopic")
// [otherwise] -> arbitrary preference
getValue: function (id)
{
	if (id == "$contacts")
		return this._getTableRows("contact");
	
	else if (id == "$dyn_channels")
		return this._getTableRows("dyn_channel");
	
	else if (id == "$slclicks")
		return this._getTableRows("slclick");
	
	else if ((id.charAt(0) == "$") || (id.charAt(0) == "@"))
		return this.getPrefValue(id);

	else if (id.charAt(0) == "#")
		return this.getGlobalValue(id);

	else if (id.charAt(0) == "%")
		return this._stringBundle.GetStringFromName(id.substr(1));
	
	else
		return this.getPrefValue(id);
},

incrementValue: function (id)
{
	var val = this.getValue(id);
	val++;
	this.setValue(id, val);
	return val;
},

setPrefValueForAllUsers: function (name, value)
{
	var ids = this.getValue("@id_list").split(":");
	var i;
	for (i = 0; i < ids.length; i++)
	{
		if (ids[i] == "") continue;
		
		this.setValue("stumble." + ids[i] + "." + name, value);
	}
},

_getTableIdFromName: function (tableName)
{
	switch (tableName)
	{
		case "contact":     return "c";
		case "dyn_channel": return "d";
		case "slclick":     return "s";
	}
	return null;
},

_getTableRows: function (tableName)
{
	var tableId = this._getTableIdFromName(tableName);
	var names = this.getPrefNames("stumble." + this.userid + "." + 
				tableId + ".");
	
	if (! names)
		return new Array();
	
	var rows = new Array();
	var row;
	var i;
	for (i = 0; i < names.length; i++)
	{
		var row_str = this.getPrefValue(names[i], "");
		if (row_str == "")
			continue;
		row = this.deserialize(row_str);
		try {
			row._t = tableId; // fixes a migration problem for alpha testers
			rows.push(row);
		} catch (e) {}
	}
	return rows;
},

insertRow: function (tableName, spec)
{
	spec._t = this._getTableIdFromName(tableName);
	var autoincName = "stumble." + this.userid + "." + spec._t + "_ai";
	spec._r = this.getPrefValue(autoincName, 0);
	this.setValue(autoincName, (spec._r + 1));
	var name = "stumble." + this.userid + "." + spec._t + "." + spec._r;
	this.setValue(name, this.serialize(spec));
	return spec._r;
},

deleteAllRows: function (tableName)
{
	var names = this.getPrefNames("stumble." + this.userid + "." + 
				this._getTableIdFromName(tableName) + ".");
	
	if (! names)
		return;
	
	var i;
	for (i = 0; i < names.length; i++)
		this.clearPref(names[i]);
},

selectRow: function (tableName, colName, value)
{
	var tableId = this._getTableIdFromName(tableName);
	var names;
	if (colName == "_r")
	{
		var row_str = this.getPrefValue("stumble." + this.userid + 
					"." + tableId + "." + value, "");
		if (row_str == "")
			return null;
		else
			return this.deserialize(row_str);
	}
	else
	{
		names = this.getPrefNames("stumble." + this.userid + "." + 
				tableId + ".");
	}
	
	if (! names)
		return null;
	
	var row;
	var i;
	for (i = 0; i < names.length; i++)
	{
		var row_str = this.getPrefValue(names[i], "");
		if (row_str == "")
			return null;
		row = this.deserialize(row_str);
		row._t = tableId; // fixes a migration problem for alpha testers
		if (row[colName] && (row[colName] == value))
			return row;
	}
	return null;
},

updateRow: function (spec)
{
	var name = "stumble." + this.userid + "." + spec._t + "." + spec._r;
	this.setValue(name, this.serialize(spec));
},

deleteRow: function (spec)
{
	var name = "stumble." + this.userid + "." + spec._t + "." + spec._r;
	this.clearPref(name);
},

define: function (fromClass, toClass, key, value)
{
	var dictName = fromClass + ":" + toClass;
	
	if ((typeof (this._dicts[dictName])) == "undefined")
		this._dicts[dictName] = new Object();
	this._dicts[dictName][key] = value;
},

lookup: function (fromClass, toClass, key)
{
	var dictName = fromClass + ":" + toClass;
	
	if (this._dicts[dictName] && 
				((typeof (this._dicts[dictName][key])) != "undefined"))
		return this._dicts[dictName][key];

	else
		return null;
},

getDictionary: function (fromClass, toClass)
{
	var dictName = fromClass + ":" + toClass;
	
	if ((typeof (this._dicts[dictName])) == "undefined")
		this._dicts[dictName] = new Object();
	
	return this._dicts[dictName];
},

setValue: function (id, value)
{
	if (id.charAt(0) == "#")
		this.globals[id.substr(1)] = value;

	else
		this.setPrefValue(id, value);
},

setPrefValue: function (id, value)
{
	this.prefsDirty = true;
	
	var type = this.getPrefType(id);
	
	if (type == null)
	{
//		if (id != "privacy.popups.disable_from_plugins")
//			this._logError(false, "UNDECLARED PREF SET WARNING", new Object(), id, value);
		type = this._getPrefTypeFromValue(value);
	}

	if (type == "JSON")
	{
		value = this.serialize(value, false);
		type = "Char";
	}
	
	var key = this._expandKey(id);
	if (key == "stumble.current_user")
	{
		this.userid = value;
		if (this._enabledb)
			this._initUserDB();
	}
	
	var attemptCount = 0;
	var success = false;
	var error = new Object();
	while ((! success) && (attemptCount < this.prefRetries))
	{
		attemptCount++;
		
		this._verifyPrefAccess();
	
		try {
			eval("this._prefBranch.set" + type + "Pref(key, value)");
			success = true;
		}
		catch (e) {
			error = e;
		}
	}
	
	if (! success)
		this._logError(true, "PREF SET", error, id);
},

getDefaultValue: function (id)
{
	if ((typeof this._DEFAULTS_BY_ID[id]) != "undefined")
		return this._DEFAULTS_BY_ID[id];
	else
		throw("stumbleupon: Datastore: cannot get default for key [" + id + "]");
},

isPrefDefined: function (id)
{
	var key = this._expandKey(id);
	
	var attemptCount = 0;
	var success = false;
	var error = new Object();
	var defined = null;
	while ((! success) && (attemptCount < this.prefRetries))
	{
		attemptCount++;
		
		this._verifyPrefAccess();
	
		try {
			defined = (this._prefBranch.getPrefType(key) != 0);
			success = true;
		}
		catch (e) {
			error = e;
		}
	}
	if (success)
	{
		return defined;
	}
	else
	{
		this._logError(true, "PREF DEFINED", error, id);
		return true;
	}
},

clearPref: function (id)
{
	var key = this._expandKey(id);

	if (! this.isPrefDefined(id))
		return;
	
	this.prefsDirty = true;
	
	var attemptCount = 0;
	while ((attemptCount < this.prefRetries))
	{
		attemptCount++;
		
		this._verifyPrefAccess();
	
		try {
			this._prefBranch.clearUserPref(key);
		} catch (e) {}
	}
},

getPrefNames: function (prefix)
{
	var attemptCount = 0;
	var success = false;
	var error = new Object();
	var list = new Array();
	while ((! success) && (attemptCount < this.prefRetries))
	{
		attemptCount++;
		
		this._verifyPrefAccess();
	
		try {
			list = this._prefBranch.getChildList(prefix, {});
			success = true;
		}
		catch (e) {
			error = e;
		}
	}
	if (! success)
	{
		list = new Array();
		this._logError(true, "PREF NAMES", error, prefix);
	}
	return list;
},

flushPrefs: function (force)
{
	if ((! force) && (! this.prefsDirty))
		return;
	this.prefsDirty = false;
	var attemptCount = 0;
	var success = false;
	var error = new Object();
	while ((! success) && (attemptCount < 1000))
	{
		attemptCount++;
		
		this._verifyPrefAccess();
	
		try {
			this._prefService.savePrefFile(null);
			success = true;
		}
		catch (e) {
			error = e;
		}
	}
	if (! success)
		this._logError(true, "PREF WRITE", error);
},

// ioflags
// RDONLY       0x01
// WRONLY       0x02
// RDWR         0x04
// CREATE_FILE  0x08
// APPEND       0x10
// TRUNCATE     0x20
// SYNC         0x40
// EXCL         0x80

readFile: function (nsifile)
{
	if (! nsifile.exists())
		return "";
	
	var data = "";
	
	try {
		var nsiuri = this._getHostService(
					"@mozilla.org/network/io-service;1",
					"nsIIOService")
					.newFileURI(nsifile);
		data = this.readURI(nsiuri);
	}
	catch (e) {
		if (((typeof nsifile.path) == "string") && 
					(nsifile.path.toLowerCase().indexOf("network") != -1))
			this._logError(false, "READ FILE", e, "network");
		else
			this._logError(false, "READ FILE", e, "local");
		return "";
	}
	return data;
},

readURI: function (nsiuri)
{
	var data = "";
	
	var channel = this._getHostService(
				"@mozilla.org/network/io-service;1",
				"nsIIOService")
				.newChannelFromURI(nsiuri);
	var input = channel.open();
	var stream = this._getHostService(
				"@mozilla.org/scriptableinputstream;1",
				"nsIScriptableInputStream");
	stream.init(input);
	var str_raw = stream.read(input.available());
	stream.close();
	input.close();
	
	try {
		var converter = this._createHostInstance(
					"@mozilla.org/intl/scriptableunicodeconverter",
					"nsIScriptableUnicodeConverter");
		converter.charset = "UTF-8";
		data = converter.ConvertToUnicode(str_raw);
	}
	catch( e ) {
		data = str_raw;
	}
	return data;
},

writeFile: function (nsifile, str, optAppendFlag)
{
	var fstream = null;
	var channel = null;
	var attemptCount = 0
	while ((attemptCount < this.prefRetries) && (! fstream))
	{
		attemptCount++;
		try {
			var ioservice = this._createHostInstance(
						"@mozilla.org/network/io-service;1",
						"nsIIOService");
			channel = ioservice.newChannelFromURI(
						ioservice.newFileURI(nsifile));
			fstream = this._createHostInstance(
						"@mozilla.org/network/file-output-stream;1",
						"nsIFileOutputStream");
		} catch (e) {}
	}
	
	try {
		// see comment block above for param documentation
		if (! nsifile.exists())
			nsifile.create(0x00, 0644);
		if (optAppendFlag)
			fstream.init(nsifile, 0x04 | 0x10, 0004, null);
		else
			fstream.init(nsifile, 0x02 | 0x20, 0004, null);
		fstream.write(str, str.length);
		fstream.close();
	}
	catch (e) {
		if (((typeof nsifile.path) == "string") && 
					(nsifile.path.toLowerCase().indexOf("network") != -1))
			this._logError(false, "WRITE FILE", e, true);
		else
			this._logError(false, "WRITE FILE", e, false);
	}
},

getFileFromPath: function (path)
{
	return new this._NSIFILE_PATH_C(path);
},

deleteFile: function (nsifile)
{
	if (! nsifile.exists()) return;

	try {
		nsifile.remove(false);
	}
	catch (e) {
//		this._logError(false, "DELETE FILE", e, nsifile.path);
	}
},

deleteDirectory: function (nsifile)
{
	if (! nsifile.exists()) return;

	try {
		nsifile.remove(true);
	} catch (e) {
		this._logError(false, "DELETE DIRECTORY", e, nsifile.path);
	}
},

getResourceNSIFile: function (subdir, filename)
{
	if (subdir != null)
	{
		subdir = subdir.toString();
		if (! this._validateFilepathComponent(subdir))
			return null;
	}
	
	filename = filename.toString();
	if (! this._validateFilepathComponent(filename))
		return null;
	
	try {
		var file = this._getHostService(
					"@mozilla.org/file/directory_service;1",
					"nsIProperties")
					.get("ProfD", this._components.interfaces.nsIFile);

		file.append("StumbleUpon");
		if (! file.exists())
			file.create(file.DIRECTORY_TYPE, 0700);
		
		if (subdir != null)
		{
			file.append(subdir);
			if (! file.exists())
				file.create(file.DIRECTORY_TYPE, 0700);
		}
		
		file.append(filename);
	} catch (e) {
		this._logError(false, "CREATE RESOURCE", e, subdir, filename);
		return null;
	}
	return file;
},

_saveURLResourceToFile: function (url, nsifile)
{
	try {
		var persist = this._createHostInstance(
					"@mozilla.org/embedding/browser/nsWebBrowserPersist;1",
					"nsIWebBrowserPersist");
		
		var nsiuri = this._createHostInstance(
					"@mozilla.org/network/standard-url;1",
					"nsIURI");
		
		nsiuri.spec = url;
		persist.progressListener = this;
		persist.saveURI(nsiuri, null, null, null, null, nsifile);
	}
	catch (e) {
		this._logError(false, "SAVE URI", e, url, nsifile.path);
		return false;
	}
	return true;
},

installResource: function (type, filename, sourceURL)
{
	var file = this.getResourceNSIFile(type, filename);
	if (! file) return false;

	this.define(
				"resource_source_url",
				"resource_id",
				sourceURL,
				type + "/" + filename);
	
	return this._saveURLResourceToFile(sourceURL, file);
},

isResourceInstalled: function (type, filename)
{
	var file = this.getResourceNSIFile(type, filename);
	
	return file && file.exists();
},

deleteResource: function (type, filename)
{
	this.deleteFile(this.getResourceNSIFile(type, filename));
},

getResourceURLFromSourceURL: function (sourceURL)
{
	var id = this.lookup(
				"resource_source_url",
				"resource_id",
				sourceURL);
	
	if ((id == null) || (id == ""))
		return null;
	
	return this.getResourceURLFromName(
				id.split("/")[0],
				id.split("/")[1]);
},

getResourceURLFromName: function (type, name)
{
	var file = this.getResourceNSIFile(type, name);
	var protocol = this._createHostInstance(
				"@mozilla.org/network/protocol;1?name=file",
				"nsIFileProtocolHandler");

	if (file)
		return protocol.getURLSpecFromFile(file);
	else
		return null;
},

hasFeature: function (name, optBits)
{
	var bits;
	if (optBits)
		bits = optBits;
	else if (name.charAt(0) == "$")
		bits = this.getValue("$form");
	else if (name.charAt(0) == "@")
		bits = this.getValue("@client_form");
	var mask = this.getFeatureMask(name);
	return ((bits & mask) == mask);
},

getFeatureMask: function (name)
{
	var idx = (this.lookup("featureid", "bit_num", name) - 1);
	return (1 << idx);
},

enableFeature: function (name)
{
	var prefName;
	if (name.charAt(0) == "$")
		prefName = "$form";
	else if (name.charAt(0) == "@")
		prefName = "@client_form";

	var mask = this.getFeatureMask(name);
	var bits = this.getValue(prefName);
	this.setValue(prefName, (bits | mask));
},

disableFeature: function (name)
{
	var prefName;
	if (name.charAt(0) == "$")
		prefName = "$form";
	else if (name.charAt(0) == "@")
		prefName = "@client_form";

	var mask = this.getFeatureMask(name);
	var bits = this.getValue(prefName);
	this.setValue(prefName, (bits & (~ mask)));
},

getSessionTimeMs: function ()
{
	return (new Date()).getTime() - this._startupTimeMs;
},

// Required by a routine in migrate.js
migrateToContacts: function ()
{
	// Move data from $friends, $emails and $sendto_stats into 
	// $contacts.

	var friendsStr = this.getValue("$friends");
	if ((friendsStr != "") && (friendsStr != "FRIEND"))
		this.updateLegacyFriend(friendsStr);
	
	var emails;
	if (this.isPrefDefined("$emails"))
		emails = this.getValue("$emails").split("\t");
	else
		emails = new Array();
	
	var i;
	
	var contact;
	for (i = emails.length - 1; i >= 0; i--)
	{
		if (emails[i] == "") continue;
		
		contact = this.selectRow("contact", "email", emails[i]);
		if (! contact)
		{
			contact = new Object();
			contact.email = emails[i];
			this.insertRow("contact", contact);
		}
	}
	
	
	var sends;
	if (this.isPrefDefined("$sendto_stats"))
		sends = this.getValue("$sendto_stats").split("\n");
	else if (this.isPrefDefined("$sendtos"))
		sends = this.getValue("$sendtos").split("\n");
	else
		sends = new Array();
	
	var artificialTimestamp = 1;
	for (i = sends.length - 1; i >= 0; i--)
	{
		if (sends[i] == "")
			continue;
		
		var fields = sends[i].split("\t");
		if (fields.length == 2)
		{
			// Migrate pre-2.7 data to include the 'type' field. -- JW
			fields.unshift("friend");
		}
		if (fields[0] == "friend")
		{
			contact = this.selectRow("contact", "nickname", fields[1]);
			if (! contact)
			{
				contact = new Object();
				contact.nickname = fields[i];
				this.insertRow("contact", contact);
			}
		}
		else if (fields[0] == "email")
		{
			contact = this.selectRow("contact", "email", fields[1]);
			if (! contact)
			{
				contact = new Object();
				contact.email = fields[i];
				contact.hidden = true;
				this.insertRow("contact", contact);
			}
		}
		contact.referral_count = parseInt(fields[2]);
		contact.referral_timestamp = artificialTimestamp;
		artificialTimestamp++;
	}
},

// Required by this.migrateToContacts() 
updateLegacyFriend: function (commandStr)
{
	var nicks = commandStr.split(" ");
	nicks.shift();
	var mutuals = new Object();
	var i;
	var contact;
	for (i = 0; i < nicks.length; i++)
	{
		mutuals[nicks[i]] = true;
		contact = this.selectRow("contact", "nickname", nicks[i]);
		if (! contact)
		{
			contact = new Object();
			contact.nickname = nicks[i];
			this.insertRow("contact", contact);
		}
	}
	
	var contacts = this.getValue("$contacts");
	for (i = 0; i < contacts.length; i++)
	{
		if ((typeof (contacts[i])) == "undefined")
		{
			this._logError(false, "LEGACY FRIEND");
			contacts.splice(i, 1);
			i--;
			continue;
		}

		if ((typeof (contacts[i].nickname)) != "undefined")
		{
			if (mutuals[contacts[i].nickname])
				contacts[i].mutual = 1;
			else
				contacts[i].mutual = 0;
			
			this.updateRow(contacts[i]);
		}
	}
},

_dispatchEvent: function (eventType, optionalEvent)
{
	var listeners = this.eventListenerListsByEventId[eventType];
	if (! listeners) return;
	
	var i;
	for (i = 0; i < listeners.length; i++)
	{
		var event;
		if (optionalEvent)
			event = this.deserialize(this.serialize(optionalEvent, false));
		
		event.target = this;
		event.eventName = eventType;
		listeners[i](event);
	}
},

addEventListener: function (eventType, listener)
{
	var listeners = this.eventListenerListsByEventId[eventType];
	if (! listeners)
	{
		listeners = new Array();
		this.eventListenerListsByEventId[eventType] = listeners;
	}
	else
	{
		this.removeEventListener(eventType, listener);
	}
	listeners.push(listener);
},

removeEventListener: function (eventType, listener)
{
	var listeners = this.eventListenerListsByEventId[eventType];
	var i;
	for (i = 0; i < listeners.length; i++)
	{
		if (listener == listeners[i])
		{
			listeners = listeners.splice(i, 1);
			break;
		}
	}
},

_getErrorObjectDump: function (o)
{
	if (! o)
		return "\n" + (typeof o);
	
	var str = "\n===== dump ===\n"; 
	var p;
	for (p in o)
	{
		if (p.match(/.*_ERR$/))
			continue;
		
		try {
			str += "[" + p + "]\n" + o[p] + "\n";
		}
		catch (e) {
			str += "[" + p + "] ERROR\n" + e + "\n";
		}
	}
	str += "========";
	return str;
},

_logError: function ()
{
	var i;
	var str = "";
	for (i = 1; i < arguments.length; i++)
	{
		var type = typeof(arguments[i]);
		if ((i == 2) && (type != "string") && (type != "number"))
			str += this._getErrorObjectDump(arguments[i]);
		
		else
			str += "\n" + arguments[i];
	}
	
	if (arguments[0])
		this._batchedPrefErrorLog += (this._batchedPrefErrorLog == "") ? str : ("\n" + str);
	else
		this._batchedErrorLog += (this._batchedErrorLog == "") ? str : ("\n" + str);
},

QueryInterface: function (iid)
{
	if (!iid.equals(this._components.interfaces.nsIWebProgressListener) &&
		!iid.equals(this._components.interfaces.nsISupportsWeakReference) &&
		!iid.equals(this._components.interfaces.nsISupports))
	{
		throw this._components.errors.NS_ERROR_NO_INTERFACE;
	}

	return this;
},

onLocationChange: function(aWebProgress, aRequest, aLocation)
{},

onProgressChange: function (aWebProgress, aRequest, aCurSelfProgress, aMaxSelfProgress, aCurTotalProgress, aMaxTotalProgress)
{},

onStateChange: function(aWebProgress, aRequest, aStateFlags, aStatus)
{
	const nsiwpl = this._components.interfaces.nsIWebProgressListener;

	if (aRequest && (aStateFlags & nsiwpl.STATE_STOP))
	{
		var event = new Object();
		event.URL = aRequest.name;
		this._dispatchEvent("resourceinstalled", event);
	}
},

onStatusChange: function(aWebProgress, aRequest, aStatus, aMessage)
{},

onSecurityChange: function(aWebProgress, aRequest, aState)
{},

_initConstDictionaries: function ()
{
	
	this._dicts["lang_code:language"] = {
"AR":"Arabic",
"EU":"Basque",
"BN":"Bengali",
"BG":"Bulgarian",
"ZH":"Chinese",
"CA":"Catalan",
"HR":"Croatian", 
"CS":"Czech",
"NL":"Dutch",
"EN":"English",
"EO":"Esperanto",
"ET":"Estonian",
"FA":"Farsi",
"FI":"Finnish",
"FR":"French",
"DA":"Danish",
"DE":"German",
"EL":"Greek",
"HE":"Hebrew",
"HI":"Hindi",
"HU":"Hungarian",
"IS":"Icelandic",
"ID":"Indonesian",
"IT":"Italian",
"JA":"Japanese",
"KO":"Korean",
"LT":"Lithuanian",
"LV":"Latvian",
"MK":"Macedonian",
"NO":"Norwegian",
"PL":"Polish",
"PT":"Portuguese",
"RO":"Romanian",
"RU":"Russian",
"SR":"Serbian",
"SK":"Slovak",
"SL":"Slovenian",
"ES":"Spanish",
"SV":"Swedish",
"TH":"Thai",
"TR":"Turkish",
"VI":"Vietnamese"};

	this._dicts["nickname:bad_nick_flag"] = {
"www":1,
"mail":1,
"www1":1,
"www2":1,
"irc":1,
"www3":1,
"tips":1,
"pop":1,
"smtp":1,
"pop3":1,
"ssl":1,
"secure":1, 
"server1":1,
"ftp":1,
"bug":1,
"bugs":1,
"bugzilla":1,
"group":1,
"groups":1,
"search":1,
"match":1,
"matchmaker":1,
"people":1,
"stumblers":1,
"stumbler":1,
"stumbles":1,
"su":1,
"matches":1,
"ww":1,
"wwww":1,
"w":1,
"wwww":1,
"business":1,
"community":1,
"stumble":1,
"labs":1,
"dating":1,
"friends":1,
"lab":1,
"toolbar":1,
"relationship":1,
"relationships":1,
"about":1,
"squirt":1,
"buzz":1,
"reviews":1,
"rss":1,
"nickname":1,
"backup":1,
"v":1,
"vid":1,
"vids":1,
"video":1,
"videos":1,
"m":1,
"mov":1,
"movies":1,
"movie":1,
"img":1,
"image":1,
"images":1,
"pic":1,
"wii":1,
"pics":1,
"forum":1,
"blog":1,
"facebook":1};

	this._dicts["userid:uc_logger_flag"] = {
"1":1,
"2":1,
"3":1,
"27":1,
"74208":1,
"728238":1,
"1329486":1,
"1427242":1,
"1461473":1,
"1477154":1,
"1693643":1, 
"1764575":1,
"1940576":1,
"2326041":1,
"2441706":1,
"2461565":1,
"2562486":1,
"2894556":1,
"3166398":1,
"3432487":1,
"3661073":1,
"4176101":1,
"4504773":1};
		
	this._dicts["toolbarid:bad_target_flag"] = {
"FindToolbar":1,
"linktoolbar":1,
"main-menubar":1,
"fbToolbar":1,
"anontoolbar":1,
"MyWebSearch":1,
"toronto_blue_jays":1,
"webpedia":1};

	this._dicts["featureid:bit_num"] = {
"$sociallinks":              1,
"$freereporting":            2,
"$mediareporting":           3,
"$slbuttonprompt":           4,
"$slbuttonpermaprompt":      5,
"$reportoption":             6,
"$unbatchedlinks":           7,
"$limitedlinks":             8,
"$newsclicktracking":        9,
"$titleclicktracking":      10,
"$unlimitedslpromptclicks": 11};

	// We don't support using lookup() to acquire default values.
	// Clients who need a default value should use getDefaultValue().
	this._DEFAULTS_BY_ID = {

// globals
"#checked_dyn_channels":       false,
"#checked_facebook":           false,
"#checking_facebook":          false,
"#facebook_userid":            0,
"#find_friends_facebook_count": 0,
"#find_friends_optin":         false,
"#find_friends_pre":           "",
"#shown_prompt2":              false,
"#sldetail":                   null,
"#slprocessed":                false,
"#visited_find_friends":       false,
"#visited_signup":             false,
"#message_count":              0,

// client preferences
"@clear_favicons":             false,
"@client_form":                0,
"@client_migration_state":     0,  // overridden
"@client_version":             "", // overridden
"@current_user":               "",
"@dd_display_message":         false,
"@dd_links_m":                 "",
"@dd_uc":                      false,
"@dist_id_list":               "",
"@dist_reg":                   "0",
"@dist_regid":                 "",
"@enable_prompt1":             true,
"@enable_prompt2":             true,
"@enable_secure_store_auth":   false,
"@enable_slstats":             true,
"@favicon_update_time_spec":   {},
"@facebook_client_invite_count": 0,
"@facebook_user":              false,
"@fbcontacts":                 {}, // stored in prefs table
"@id_list":                    "",
"@installed":                  "0",
"@json_db":                    {}, // obsolete v3.05
"@latch-to-sidebar":           false,
"@log_prefetch_progress":      false,
"@position-group":             "first",
"@position_history":           "stumbleupon",
"@recommend_timeout_ms":       60000,
"@report_error_count":         0,
"@report_error_count_max":     3,  // overridden
"@right-justify-width":        0,  // overridden
"@search-width":               156,
"@shown_toolbar":              false,
"@stumble_action_timeout_ms":  15000,
"@time_spec":                  {client:0,server:0}, // obsolete v3.08
"@toolbar-position":           "stumbleupon",
"@toolbar_toggle_visible":     false,
"@toolbar-visible":            true,
"@update_avatar_on_receive":   true, // obsolete v3.08
"@update_avatar_on_send":      true, // obsolete v3.08

// user preferences
"$autocomplete_type":          "tag,query",
"$autologout":                 false,
"$bad_stumble":                false,
"$block_flash_popups":         true,
"$bookmark_folder_resource_id": "",
"$check-referral":             "",
"$comment_firstrating":        true,
"$contacts":                   {}, // stored in prefs table
"$dd_rec_label":               "[rec label]",
"$dd_rec_rating":              false,
"$dyn_channels":               {}, // stored in prefs table
"$emails":                     "", // obsolete v3.0
"$facebook_added":             false,
"$facebook_contacts_time_s":   0,
"$facebook_homeprompt_time_s": 0,
"$facebook_homeprompt_optout": false,
"$facebook_invite_count":      0,
"$facebook_linked":            false,
"$firstfriends":               "", // unused by client
"$form":                       1,
"$friends":                    "FRIEND", // obsolete v3.0
"$friends_synced":             "0",
"$great_stumble":              false,
"$has_avatars":                false,
"$has_favicons":               false,
"$icons":                      "text-icons",
"$interests":                  "",
"$imported_fbcontacts_time_s": 0,
"$imported_contacts_time_s":   0,
"$intro_count":                0,
"$last_incat":                 "0",
"$last_stumble":               "0",
"$last_uploaded":              "0",
"$migration_state":            0,  // overridden
"$newmessage":                 false,
"$nick":                       "",
"$password":                   "", // obsolete v3.0
"$prefetch":                   true,
"$prefetcher_fetch_depth_in_query": -1,
"$prefetcher_fetch_depth_in_topic": 3,
"$prefetcher_pass_1_timeout_ms": 10000,
"$prefetcher_pass_2_timeout_ms": 30000,
"$prefetcher_pass_3_timeout_ms": 120000,
"$prefetcher_pass_max":        3,
"$prefetcher_skip_resources":  false,
"$process_rarely_timestamp":   "0",
"$query_history_depth":        100,
"$rate_new_window":            false,
"$recent_sendtos_menu_depth":  15,
"$referral_count":             "0", // legacy incorrect type
"$review_new_window":          false,
"$search_clear_queries":       false,
"$search_new_window":          false,
"$searchlink_logos":           true, // obsolete v3.08
"$sender_click_platform":      false,
"$sendtos":                    "", // obsolete v2.7
"$sendtos_menu_depth":         16,
"$sendto_stats":               "", // obsolete v3.0
"$shortcut_reviews":           "", // platform-specific
"$shortcut_stumble":           "", // platform-specific
"$shortcut_tag":               "", // Alt+VK_SLASH
"$shortcut_thumbdown":         "", // platform-specific
"$shortcut_thumbup":           "", // platform-specific
"$shortcut_toolbar":           "", // platform-specific
"$shortcut-reviews":           "", // obsolete v2.7
"$shortcut-stumble":           "", // obsolete v2.7
"$shortcut-thumbdown":         "", // obsolete v2.7
"$shortcut-thumbup":           "", // obsolete v2.7
"$shortcut-toolbar":           "", // obsolete v2.7
"$shortcuts_enabled":          false,
"$show_aboutme":               false,
"$show_editinfo":              false, // obsolete v3.0
"$show_field":                 false,
"$show_firstrater_label_always": false,
"$show_flag":                  false,
"$show_forums":                false, // obsolete v3.07
"$show_friends":               true,
"$show_groups":                false,
"$show_home":                  true,
"$show_info":                  true,
"$show_legacy_forums":         false,
"$show_legacy_network":        false,
"$show_matches":               false,
"$show_messages":              false,
"$show_mode":                  true,
"$show_mode_all":              true,
"$show_mode_friends":          true,
"$show_mode_more":             true,
"$show_mode_news":             true,
"$show_mode_photo":            true,
"$show_mode_search":           false,
"$show_mode_stumbler":         true,
"$show_mode_stumblers":        false,
"$show_mode_video":            true,
"$show_mode_wiki":             false,
"$show_myinfo":                false, // undocumented feature
"$show_mystumblers":           false, // obsolete v3.07
"$show_referral":              true,
"$show_search":                false, // obsolete v3.0
"$show_searchlinks":           false, // obsolete v3.08
"$show_searchlinks_comment_icon": true,
"$show_searchlinks_friends":   false,
"$show_searchlinks_logo":      true,
"$show_searchlinks_tooltip":   true,  // undocumented feature
"$show_searchlinks_topic":     false,
"$show_searchlinks_score":     false,
"$show_separators":            true,
"$show_tag":                   false,
"$show_topics":                true,
"$shown_find_friends":         false,
"$shown_find_friends_clicks":  0,
"$shown_find_friends_time_s":  0,
"$shown_find_friends_optout":  false,
"$shown_searchlinks_dialog":   false,
"$shown_searchlinks":          false,
"$shown_tag":                  false,
"$slclicks":                   {},    // stored in prefs table
"$slidfstats":                 "0:0:0:0:0:0:0:0:0:0",
"$slistats":                   "0:0:0:0:0:0:0:0:0:0",
"$slstats":                    "",
"$sponsor":                    false, // unused by client
"$stumble_topics":             false,
"$stumble_upon_change":        true,
"$stumblereferrals":           "",
"$stumblestats":               "",
"$stumbletimes":               "",
"$stumbletypes":               "",
"$sync_clientid":              0,
"$sync_recent_taskid":         0,
"$tag_history_depth":          100,
"$tagged_discovery_count":     0,
"$token":                      "",
"$version":                    ""}; // overridden
}



} // END prototype


