//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Gui
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QtGui/QPushButton>
#include <QtGui/QLabel>
#include <QtGui/QCheckBox>
#include <QtGui/QLineEdit>
#include <QtGui/QComboBox>
#include <QtGui/QTableWidget>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QAction>
#include <QtGui/QSpinBox>
#include <QtGui/QGroupBox>
#include "../core/diskdevice.h"
#include "../core/diskcore.h"
#include "volumewidget.h"
#include "optionstable.h"
#include "infobutton.h"
#include "choosemountpointwidget.h"
#include "const.h"

VolumeWidget::VolumeWidget(DiskCore *core,DiskDevice *dev,PopupWindow *popup) {
	device = dev;
	
	changeMountPointAction = new QAction(this);
	changeMountPointAction->setText(tr("Change mount point"));

	if (device->isSupportDvd()) {
		udfOrIsoComboBox = new QComboBox;
		udfOrIsoComboBox->addItem(tr("DVD and CD together"),"udf,iso9660");
		udfOrIsoComboBox->addItem(tr("Only DVD"),"udf");
		udfOrIsoComboBox->addItem(tr("Only CD"),"iso9660");
		connect(udfOrIsoComboBox,SIGNAL(currentIndexChanged(int)),this,SLOT(udfOrIsoChangedSlot(int)));
		udfOrIsoLabel = new QLabel(tr("What kind of disks do you use?"));
	} else {
		udfOrIsoComboBox = 0;
		udfOrIsoLabel = 0;
	}

	showDefaultOptions = new QCheckBox(tr("Show default options"));

	mountPointLine = new ChooseMountPointWidget;
	mountPointLine->setMountPoint(device->currentMountPoint());
	if (!mountPointLine->isValid())
		mountPointLine->setMountPoint(device->fstabMountPoint());

	dump = new QCheckBox(tr("Use the program \"Dump\" for system restoration"));
	fsck = new QSpinBox;
	fsck->setMaximum(2);
	fsck->setMinimum(0);

	dump->setChecked(device->dump());
	fsck->setValue(device->fsck());
	
	optionsLine = new QLineEdit;
	optionsLine->setReadOnly(true);
	optionsLineInfoButton = new InfoButton(popup);
	optionsLineInfoButton->setPopupText(tr("This line will be written to configuration file. When the Linux system mount the partition, it will mount with these options."));
	optionsLineInfoButton->setPopupHeaderText(tr("Options line"));
	
	filterLine = new QLineEdit;
	clearFilterLineButton = new QPushButton;
	clearFilterLineButton->setIcon(QIcon(ICONS_PATH"clear_left.png"));
	clearFilterLineButton->setFlat(true);
	clearFilterLineButton->setEnabled(false);

	optionTypes = new QComboBox;
	optionsTable = new OptionsTable(device,popup);
	optionsTable->addOptions(core->fileSystemOptions("common"),"common");

	optionTypes->addItem(tr("All"),"all");
	optionTypes->addItem(tr("Common"),"common");

	if (device->type() == "cdrom") {
		optionsTable->addOptions(core->fileSystemOptions("iso9660"),"iso9660");
		optionTypes->addItem("Iso9660","iso9660");
		if (device->isSupportDvd()) {
			optionsTable->addOptions(core->fileSystemOptions("udf"),"udf");
			optionTypes->addItem("Udf","udf");
		}
	} else {
		optionsTable->addOptions(core->fileSystemOptions(device->fileSystem()),device->fileSystem());
		optionTypes->addItem(device->fileSystem(),device->fileSystem());
	}

	if (device->fileSystem() == "ntfs") {
		useNtfs3gOptions = new QCheckBox(tr("Use special program for work with ntfs file system \"ntfs-3g\""));
		optionsTable->addOptions(core->fileSystemOptions("ntfs-3g"),"ntfs-3g");

		useNtfs3gOptions->setChecked(device->fstabFileSystem() == "ntfs-3g");
		useNtfs3g(device->fstabFileSystem() == "ntfs-3g");
	} else
		useNtfs3gOptions = 0;
	
	QVBoxLayout *mainLayout = new QVBoxLayout;
	
	QHBoxLayout *optionsLayout = new QHBoxLayout;
	optionsLayout->addWidget(new QLabel(tr("Options") + ":"));
	optionsLayout->addWidget(optionsLine,1);
	optionsLayout->addWidget(optionsLineInfoButton);
		
	QHBoxLayout *filterAndTypeLayout = new QHBoxLayout;
	filterAndTypeLayout->addWidget(new QLabel(tr("Filter") + ":"));
	filterAndTypeLayout->addWidget(filterLine);
	filterAndTypeLayout->addWidget(clearFilterLineButton);
	filterAndTypeLayout->addStretch();
	filterAndTypeLayout->addWidget(new QLabel(tr("Show options type") + ":"));
	filterAndTypeLayout->addWidget(optionTypes);

	if (udfOrIsoLabel) {
		QHBoxLayout *udfOrIsoLayout = new QHBoxLayout;
		udfOrIsoLayout->addWidget(udfOrIsoLabel);
		udfOrIsoLayout->addWidget(udfOrIsoComboBox);
		udfOrIsoLayout->addStretch();
		mainLayout->addLayout(udfOrIsoLayout);
	}

	if (udfOrIsoComboBox) {
		if (!device->fstabFileSystem().isEmpty()) {
			if (device->fstabFileSystem() == "udf,iso9660")
				udfOrIsoChangedSlot(0);
			else if (device->fstabFileSystem() == "udf")
				udfOrIsoComboBox->setCurrentIndex(1);
			else if (device->fstabFileSystem() == "iso9660")
				udfOrIsoComboBox->setCurrentIndex(2);
		} else
			udfOrIsoChangedSlot(0);
	}

	if (useNtfs3gOptions)
		mainLayout->addWidget(useNtfs3gOptions);

	QHBoxLayout *mountPointLayout = new QHBoxLayout;
	mountPointLayout->addWidget(new QLabel(tr("Mount point") + ":"));
	mountPointLayout->addWidget(mountPointLine);
	mountPointLayout->addStretch();

	QHBoxLayout *dumpLayout = new QHBoxLayout;
	dumpLayout->addWidget(dump);
	dumpLayout->addStretch();

	QHBoxLayout *fsckLayout = new QHBoxLayout;
	fsckLayout->addWidget(new QLabel(tr("Value of the program \"Fsck\"") + ":"));
	fsckLayout->addWidget(fsck);
	fsckLayout->addStretch();

	mainLayout->addLayout(mountPointLayout);
	mainLayout->addLayout(dumpLayout);
	mainLayout->addLayout(fsckLayout);

	optionsGroupBox = new QGroupBox;
	optionsGroupBox->setTitle(tr("Options"));
	
	QVBoxLayout *boxLayout = new QVBoxLayout;
	boxLayout->addWidget(showDefaultOptions);
	boxLayout->addLayout(optionsLayout);
	boxLayout->addLayout(filterAndTypeLayout);
	boxLayout->addWidget(optionsTable);
	optionsGroupBox->setLayout(boxLayout);
	
	mainLayout->addWidget(optionsGroupBox);
	setLayout(mainLayout);

	connect(filterLine,SIGNAL(textChanged(const QString &)),this,SLOT(setFilter(const QString&)));
	connect(clearFilterLineButton,SIGNAL(clicked()),filterLine,SLOT(clear()));
	connect(optionTypes,SIGNAL(currentIndexChanged(int)),this,SLOT(setTypeFilter(int)));
	connect(optionsTable,SIGNAL(optionChanged()),this,SLOT(updateOptionLine()));
	connect(optionsTable,SIGNAL(optionChanged()),this,SIGNAL(optionsChanged()));
	connect(showDefaultOptions,SIGNAL(clicked()),this,SLOT(updateOptionLine()));
	if (useNtfs3gOptions)
		connect(useNtfs3gOptions,SIGNAL(toggled(bool)),this,SLOT(useNtfs3g(bool)));
	connect(device,SIGNAL(currentMountPointChanged(const QString &)),this,SLOT(setMountPoint(const QString &)));
	
	updateOptionLine();
}

VolumeWidget::~VolumeWidget() {
	delete showDefaultOptions;
	delete changeMountPointAction;
	delete mountPointLine;
	delete dump;
	delete fsck;

	delete optionsLine;
	delete optionsLineInfoButton;

	delete filterLine;
	delete clearFilterLineButton;

	delete optionTypes;
	delete optionsTable;
	delete optionsGroupBox;
}

void VolumeWidget::setFilter(const QString& filterText) {
	clearFilterLineButton->setEnabled(!filterText.isEmpty());
	optionsTable->setFilter(filterText);
}

void VolumeWidget::setTypeFilter(int index) {
	optionsTable->showOptionsFromSource(optionTypes->itemData(index).toString());
}

void VolumeWidget::updateOptionLine() {
	QString optionLineText = optionsTable->options(showDefaultOptions->isChecked());
	
	// rw,suid,dev,exec,auto,nouser,async == defaults
	if (!showDefaultOptions->isChecked())
		replaceDefaultOptionsOnOneWord(optionLineText);
	// If first character of option line is comma remove it
	if (!optionLineText.isEmpty() && optionLineText[0] == ',')
		optionLineText.remove(0,1);
	// If last character of option line is comma remove it
	if (!optionLineText.isEmpty() && optionLineText[optionLineText.length()-1] == ',')
		optionLineText.remove(optionLineText.length()-1,1);
	// If option line is empty then all options are default
	if (optionLineText.isEmpty())
		optionLineText = tr("All options are default");
	optionsLine->setText(optionLineText);
}

void VolumeWidget::replaceDefaultOptionsOnOneWord(QString& options) {
	if (options.contains("rw") && options.contains("suid") && options.contains("dev") && options.contains("exec")
		&& options.contains("auto") && options.contains("nouser") && options.contains("async")) {
		options = options.remove("rw,");
		options = options.remove("suid,");
		options = options.remove("dev,");
		options = options.remove("exec,");
		options = options.remove("auto,");
		options = options.remove("nouser,");
		options = options.remove("async,");
		options += ",defaults";
	}
}

void VolumeWidget::udfOrIsoChangedSlot(int index) {
	bool thereIsUdf = false;
	bool thereIsIso = false;
	switch (index) {
		case 0:
			optionsTable->setFixedFilter("common");
			for (int i = optionTypes->count()-1; i >= 2 ; i--)
				optionTypes->removeItem(i);
			break;
		case 1:
			optionsTable->setFixedFilter(QStringList() << "udf" << "common");
			for (int i = optionTypes->count()-1; i >= 2 ; i--)
				if (optionTypes->itemData(i).toString() == "iso9660")	
					optionTypes->removeItem(i);
				else if (optionTypes->itemData(i).toString() == "udf")
					thereIsUdf = true;
			if (!thereIsUdf)
				optionTypes->addItem("udf","udf");
			break;
		case 2:
			optionsTable->setFixedFilter(QStringList() << "iso9660" << "common");
			for (int i = optionTypes->count()-1; i >= 2 ; i--)
				if (optionTypes->itemData(i).toString() == "udf")
					optionTypes->removeItem(i);
				else if (optionTypes->itemData(i).toString() == "iso9660")
					thereIsIso = true;
			if (!thereIsIso)
				optionTypes->addItem("iso9660","iso9660");
			break;
	}
	updateOptionLine();
}

void VolumeWidget::useNtfs3g(bool use) {
	if (use) {
		optionsTable->setFixedFilter("ntfs-3g");
		for (int i = optionTypes->count()-1; i >= 0; i--)
			optionTypes->removeItem(i);
		optionTypes->addItem("ntfs-3g","ntfs-3g");
	} else {
		optionsTable->setFixedFilter(QStringList() << "common" << "ntfs");
		for (int i = optionTypes->count()-1; i >= 0; i--)
			optionTypes->removeItem(i);
		optionTypes->addItem(tr("All"),"all");
		optionTypes->addItem("Common","common");
		optionTypes->addItem("ntfs","ntfs");
	}
	updateOptionLine();
	
}

const QString& VolumeWidget::mountPoint() const {
	return mountPointLine->mountPoint();
}

QString VolumeWidget::options() {
	return optionsTable->options(false).isEmpty() ? "defaults" : optionsTable->options(false);
}

void VolumeWidget::setMountPoint(const QString &mountPoint) {
	if (mountPoint.isEmpty())
		mountPointLine->setMountPoint(device->fstabMountPoint());
	else
		mountPointLine->setMountPoint(mountPoint);
}

QString VolumeWidget::fileSystem() {
	if (useNtfs3gOptions && useNtfs3gOptions->isChecked())
		return "ntfs-3g";
	else if (udfOrIsoComboBox)
		return udfOrIsoComboBox->itemData(udfOrIsoComboBox->currentIndex()).toString();
	return device->fileSystem();
}

QString VolumeWidget::fstabLine(bool warn,const QString& format) {
	QString mainLine;
	if (format == "name")
		mainLine += device->blockName() + '\t';
	else if (format == "uuid")
		if (device->uuid().isEmpty())
			mainLine += device->blockName() + '\t';
		else
			mainLine += "UUID=" + device->uuid() + '\t';
	else if (format == "label") {
		if (device->label().isEmpty()) {
			if (device->uuid().isEmpty())
				mainLine += device->blockName() + '\t';
			else
				mainLine += "UUID=" + device->uuid() + '\t';
		} else
			mainLine += "LABEL=" + device->label() + '\t';
	}
	if (!mountPointLine->isValid() && warn) {
		emit (showWarning(tr("Mount point for device %1 is empty - in configuration file this device will skip.").arg(device->blockName()),QList<QAction *>() << changeMountPointAction));
		return QString();
	} else
		mainLine += mountPointLine->mountPoint() + '\t';
	mainLine += fileSystem() + '\t';
	QString opts = options();
	mainLine += opts.isEmpty() ? "defaults" : opts;
	mainLine += '\t';
	mainLine += dump->isChecked() ? "1" : "0";
	mainLine += '\t';
	mainLine += QVariant(fsck->value()).toString();
	
	return mainLine;
}
