//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Gui
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QLineEdit>
#include <QAction>
#include <QGroupBox>
#include <QComboBox>
#include <QCheckBox>
#include <QHideEvent>
#include <QPushButton>
#include <QSplitter>
#include <QTableWidget>
#include <QGridLayout>
#include <QHeaderView>
#include <QFile>
#include <QTextStream>
#include <QLabel>
#include <QMessageBox>
#include <QFileDialog>
#include "choosemountpointwidget.h"
#include "usbmanager.h"
#include "infobutton.h"
#include "const.h"
#include "../core/diskdevice.h"
#include "volumewidget.h"
#include "specialcomboboxwidgets.h"
#include "usbmanagerfsoptionsdialog.h"
#include "usbmanagerfscellwidget.h"
#include "treewidget.h"

UsbManager::UsbManager(PopupWindow *popupWindow,QWidget *parent) : QDialog(parent) {

	QFile udevFile("/etc/udev/rules.d/01-mountmanager.rules");
	udevFile.open(QIODevice::ReadOnly);
	QTextStream udevStream(&udevFile);
	allRules = udevStream.readAll().split("#Dont_remove_this_row!#\n");
	
	isCreatingDeviceRuleNow = false;
	currentDevice = 0;
	
	splitter = new QSplitter;
	
	deviceDefinitionBox = new QGroupBox;
	deviceDefinitionBox->setTitle(tr("Devices definition"));
	
	deviceName = new QLineEdit;
	vendorName = new QLineEdit;
	modelName = new QLineEdit;
	busName = new QLineEdit;
	subsystemName = new QLineEdit;

	otherRuleSettingsBox = new QGroupBox;
	otherRuleSettingsBox->setTitle(tr("Other rule settings") + ":");

	groupName = new GIDWidget;
	ownerName = new UIDWidget;
	oneMoreDeviceName = new QLineEdit;
	
	autoMountSettingsBox = new QGroupBox;
	autoMountSettingsBox->setTitle(tr("Settings of auto-mounting"));

	optionsForFileSystems = new QTableWidget(0,1);
	optionsForFileSystems->horizontalHeader()->hide();
	optionsForFileSystems->verticalHeader()->hide();
	optionsForFileSystems->horizontalHeader()->setStretchLastSection(true);
	optionsForFileSystems->verticalHeader()->setResizeMode(QHeaderView::Fixed);
	optionsForFileSystems->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	optionsForFileSystems->setSelectionMode(QAbstractItemView::SingleSelection);
	optionsForFileSystems->setAlternatingRowColors(true);

	fileSystem = new QComboBox;
	fileSystem->addItem("Fat","vfat");
	fileSystem->addItem("Ntfs","ntfs");
	fileSystem->addItem("Ext2","ext2");
	fileSystem->addItem("Ext3","ext3");
	fileSystem->addItem("ReiserFs","reiserfs");
	
	mountPointWidget = new ChooseMountPointWidget;
	mountPointWidget->setReadOnly(false);
	createAndRemoveMountPoint = new QCheckBox(tr("Remove mount point directory after device removing"));
		
	treeWidget = new TreeWidget(true);
	treeWidget->setColumnCount(4);
	treeWidget->setHeaderLabels(QStringList() << tr("Device") << tr("Vendor") << tr("Model") << tr("Bus"));
	treeWidget->setContextMenuIcon(QIcon(PROGRAM_ICON_PATH));
	treeWidget->setContextMenuHeader(tr("All rules"));
	removeRuleAction = new QAction(treeWidget);
	removeRuleAction->setText(tr("Remove the rule"));
	removeRuleAction->setIcon(QIcon(ICONS_PATH"remove.png"));
	editRuleAction = new QAction(treeWidget);
	editRuleAction->setText(tr("Edit the rule"));
	editRuleAction->setIcon(QIcon(ICONS_PATH"edit.png"));
	treeWidget->addContextMenuAction(editRuleAction);
	treeWidget->addContextMenuAction(removeRuleAction);
	
	rulesFilter = new QLineEdit;

	editRuleButton = new QPushButton(tr("Edit rule"));
	editRuleButton->setIcon(QIcon(ICONS_PATH"edit.png"));
	editRuleButton->setVisible(false);
	
	addFileSystemButton = new QPushButton;
	addFileSystemButton->setIcon(QIcon(ICONS_PATH"add.png"));
	addFileSystemButton->setFlat(true);
	
	removeFileSystemButton = new QPushButton;
	removeFileSystemButton->setIcon(QIcon(ICONS_PATH"remove.png"));
	removeFileSystemButton->setFlat(true);
	removeFileSystemButton->setEnabled(false);
	
	closeCancelButton = new QPushButton(tr("Close"));
	
	addRuleButton = new QPushButton(tr("Add rule"));
	addRuleButton->setIcon(QIcon(ICONS_PATH"add.png"));
	
	saveAsRuleButton = new QPushButton(tr("Save rule as"));
	saveAsRuleButton->setIcon(QIcon(ICONS_PATH"save_as.png"));

	ownerInfoButton = new InfoButton(popupWindow);
	ownerInfoButton->setPopupText(tr("You can point owner of your device. It means that only this user has access to this device."));
	ownerInfoButton->setPopupHeaderText(tr("Who is owner"));
	
	setOneMoreDeviceNameInfoButton = new InfoButton(popupWindow);
	setOneMoreDeviceNameInfoButton->setPopupText(tr("When you connect any device, in \"/dev\" directory file with defined name will be created and you can specify one more name and in /dev/ directory after device connection file with this name will be created."));
	setOneMoreDeviceNameInfoButton->setPopupHeaderText(tr("One more device name"));
	
	optionsForFileSystemsInfoButton = new InfoButton(popupWindow);
	optionsForFileSystemsInfoButton->setPopupText(tr("File system of the USB devices can be different and you have to point for what device file system this rule is written. You can write one rule for any file system. To do it click on a plus button, choose the file system and set the options. For every file system mounting options are different, but there are options which influence on any file system. After the device is connected it will be mounted with this options."));
	optionsForFileSystemsInfoButton->setPopupHeaderText(tr("What is it"));
	
	groupInfoButton = new InfoButton(popupWindow);
	groupInfoButton->setPopupText(tr("You can point group for device. It means that all users of this group have access to this device."));
	groupInfoButton->setPopupHeaderText(tr("What is group"));
	
	subsystemInfoButton = new InfoButton(popupWindow);
	subsystemInfoButton->setPopupText(tr("When we connect an usb device, Linux kernel and udev connect a lot of inside devices with different subsystems; we always mount devices with \"block\" subsystem, but if you want to see another subsystems run \"udevmonitor --env\" and connect the device."));
	subsystemInfoButton->setPopupHeaderText(tr("What is subsystem"));
	
	busInfoButton = new InfoButton(popupWindow);
	busInfoButton->setPopupText(tr("Through this bus we can get information about device. There are a lot of types of bus and \"usb\" bus is for USB devices."));
	busInfoButton->setPopupHeaderText(tr("What is bus"));
	
	informationButton = new InfoButton(popupWindow);
	informationButton->setPopupText(tr("In majority of cases Udev and command \"mount\" are used for auto-mounting in Linux. Udev can create devices in /dev directory and it's possible to create rules for udev, which can act on one defined device or on the devices, which meet the conditions of the rule. USB devices manager will help you to create the rules. In these rules you can point what actions must be executed after device connection or disconnection. MountManager uses it to make setting of auto-mounting easy and fast."));
	informationButton->setPopupHeaderText(tr("More information"));
	informationButton->setText(tr("More information"));
	
	deviceNameInfoButton = new InfoButton(popupWindow);
	deviceNameInfoButton->setPopupText(tr("This field is usually used for creation of the rules, which influence on some devices. For example, you can enter sd* - it means that this rule is written for all devices, which have names starting with \"sd\". Other variants: sd? - started with \"sd\" and only one any character after it, sd[a-z][0-9] - started with \"sd\" and after it follows any character for \"a\" to \"z\" and any digit, for example: sda1,sdb1,sdc2 satisfy the conditions."));
	deviceNameInfoButton->setPopupHeaderText(tr("What is device name"));
	
	vendorInfoButton = new InfoButton(popupWindow);
	vendorInfoButton->setPopupText(tr("This thing is used to create the rule for only one device, because in majority of cases vendor is different from USB devices. To define what vendor from your device you have to connect this device and find it in descriptions of your devices from devices tree item of your device."));
	vendorInfoButton->setPopupHeaderText(tr("What is vendor"));
	
	modelInfoButton = new InfoButton(popupWindow);
	modelInfoButton->setPopupText(tr("This this is used to create the rule for only one device or devices of one model."));
	modelInfoButton->setPopupHeaderText(tr("What is model"));
	
	mountPointInfoButton = new InfoButton(popupWindow);
	mountPointInfoButton->setPopupText(tr("You can set fixed mount point, for example /media/usb or dinamic by using special characters: %k - name of the device, for example if you set mount point \"/media/usb-%k\" and device has name sda1, it will be mounted to \"/media/usb-sda1\", %n - number of the device, for example for device sda1 and mount point \"/media/myusb_%n\" the device will be mounted to \"/media/myusb_1\""));
	mountPointInfoButton->setPopupHeaderText(tr("What is mount point"));

	QVBoxLayout *fileSystemButtonsLayout = new QVBoxLayout;
	fileSystemButtonsLayout->addWidget(addFileSystemButton);
	fileSystemButtonsLayout->addWidget(removeFileSystemButton);
	fileSystemButtonsLayout->addWidget(optionsForFileSystemsInfoButton);
	fileSystemButtonsLayout->addStretch();

	QHBoxLayout *fileSystemsHorizontalLayout = new QHBoxLayout;
	fileSystemsHorizontalLayout->addWidget(optionsForFileSystems,1);
	fileSystemsHorizontalLayout->addLayout(fileSystemButtonsLayout);
	
	QGridLayout *autoMountBoxLayout = new QGridLayout;
	autoMountBoxLayout->addLayout(fileSystemsHorizontalLayout,0,0,1,3);
	autoMountBoxLayout->addWidget(mountPointInfoButton,1,0);
	autoMountBoxLayout->addWidget(new QLabel(tr("Mount point") + ":"),1,1);
	autoMountBoxLayout->addWidget(mountPointWidget,1,2);
	autoMountBoxLayout->addWidget(createAndRemoveMountPoint,2,0,1,3);
	autoMountBoxLayout->setColumnStretch(2,1);
	autoMountSettingsBox->setLayout(autoMountBoxLayout);
	
	QGridLayout *definitionBoxLayout = new QGridLayout;
	definitionBoxLayout->addWidget(deviceNameInfoButton,0,0);
	definitionBoxLayout->addWidget(new QLabel(tr("Device name") + ":"),0,1);
	definitionBoxLayout->addWidget(deviceName,0,2);
	definitionBoxLayout->addWidget(vendorInfoButton,1,0);
	definitionBoxLayout->addWidget(new QLabel(tr("Vendor") +":"),1,1);
	definitionBoxLayout->addWidget(vendorName,1,2);
	definitionBoxLayout->addWidget(modelInfoButton,2,0);
	definitionBoxLayout->addWidget(new QLabel(tr("Model") + ":"),2,1);
	definitionBoxLayout->addWidget(modelName,2,2);
	definitionBoxLayout->addWidget(busInfoButton,3,0);
	definitionBoxLayout->addWidget(new QLabel("Bus:"),3,1);
	definitionBoxLayout->addWidget(busName,3,2);
	definitionBoxLayout->addWidget(subsystemInfoButton,4,0);
	definitionBoxLayout->addWidget(new QLabel("Subsystem:"),4,1);
	definitionBoxLayout->addWidget(subsystemName,4,2);
	definitionBoxLayout->setColumnStretch(2,1);
	deviceDefinitionBox->setLayout(definitionBoxLayout);

	QGridLayout *otherRuleSettingsBoxLayout = new QGridLayout;
	otherRuleSettingsBoxLayout->addWidget(groupInfoButton,0,0);
	otherRuleSettingsBoxLayout->addWidget(new QLabel(tr("Group") + ":"),0,1);
	otherRuleSettingsBoxLayout->addWidget(groupName,0,2);
	otherRuleSettingsBoxLayout->addWidget(ownerInfoButton,1,0);
	otherRuleSettingsBoxLayout->addWidget(new QLabel(tr("Owner") + ":"),1,1);
	otherRuleSettingsBoxLayout->addWidget(ownerName,1,2);
	otherRuleSettingsBoxLayout->addWidget(setOneMoreDeviceNameInfoButton,2,0);
	otherRuleSettingsBoxLayout->addWidget(new QLabel(tr("One more name") + ":"),2,1);
	otherRuleSettingsBoxLayout->addWidget(oneMoreDeviceName,2,2);
	otherRuleSettingsBoxLayout->setColumnStretch(2,1);
	otherRuleSettingsBox->setLayout(otherRuleSettingsBoxLayout);
	
	QHBoxLayout *rulesActionsLayout = new QHBoxLayout;
	rulesActionsLayout->addWidget(new QLabel(tr("Filter") + ":"));
	rulesActionsLayout->addWidget(rulesFilter);
	rulesActionsLayout->addStretch();
	
	QVBoxLayout *leftLayout = new QVBoxLayout;
	leftLayout->addLayout(rulesActionsLayout);
	leftLayout->addWidget(treeWidget);
	
	QHBoxLayout *informationLayout = new QHBoxLayout;
	informationLayout->addStretch();
	informationLayout->addWidget(addRuleButton);
	informationLayout->addWidget(editRuleButton);
	informationLayout->addWidget(informationButton);
	
	QVBoxLayout *rightLayout = new QVBoxLayout;
	rightLayout->addWidget(deviceDefinitionBox);
	rightLayout->addWidget(autoMountSettingsBox);
	rightLayout->addWidget(otherRuleSettingsBox);
	rightLayout->addStretch();
	rightLayout->addLayout(informationLayout);

	QWidget *leftWidget = new QWidget;
	leftWidget->setLayout(leftLayout);

	QWidget *rightWidget = new QWidget;
	rightWidget->setLayout(rightLayout);
	
	splitter->addWidget(leftWidget);
	splitter->addWidget(rightWidget);
	splitter->setStretchFactor(1,1);
	splitter->setCollapsible(0,false);
	splitter->setCollapsible(1,false);
	
	QHBoxLayout *bottomLayout = new QHBoxLayout;
	bottomLayout->addStretch();
	bottomLayout->addWidget(saveAsRuleButton);
	bottomLayout->addWidget(closeCancelButton);
	
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(splitter);
	mainLayout->addLayout(bottomLayout);
	setLayout(mainLayout);

	// Connections
	connect(closeCancelButton,SIGNAL(clicked()),this,SLOT(close()));
	connect(addFileSystemButton,SIGNAL(clicked()),this,SLOT(addFileSystem()));
	connect(removeFileSystemButton,SIGNAL(clicked()),this,SLOT(removeFileSystem()));
	connect(optionsForFileSystems,SIGNAL(cellClicked(int,int)),this,SLOT(currentFileSystemChanged(int,int)));
	connect(addRuleButton,SIGNAL(clicked()),this,SLOT(addRule()));
	connect(this,SIGNAL(rulesChanged()),this,SLOT(rulesChangedSlot()));
	connect(editRuleAction,SIGNAL(triggered()),this,SLOT(fillRuleParts()));
	connect(removeRuleAction,SIGNAL(triggered()),this,SLOT(removeRule()));
	connect(editRuleButton,SIGNAL(clicked()),this,SLOT(editRule()));
	connect(rulesFilter,SIGNAL(textChanged(const QString &)),this,SLOT(setFilter(const QString &)));
	connect(saveAsRuleButton,SIGNAL(clicked()),this,SLOT(saveRuleAs()));
	
	setWindowTitle(tr("USB devices manager"));
	resize(800,350);
	createTreeWidgetItems();
}


UsbManager::~UsbManager() {
	foreach (UsbManagerFsOptionsDialog *dialog, dialogs)
		delete dialog;
	foreach (UsbManagerFsCellWidget *cell,cells)
		delete cell;
	
	delete informationButton;
	delete deviceNameInfoButton;
	delete vendorInfoButton;
	delete modelInfoButton;
	delete mountPointInfoButton;
	delete busInfoButton;
	delete subsystemInfoButton;
	delete groupInfoButton;
	delete optionsForFileSystemsInfoButton;
	delete setOneMoreDeviceNameInfoButton;
	delete ownerInfoButton;

	delete removeFileSystemButton;
	delete addFileSystemButton;
	
	delete deviceName;
	delete vendorName;
	delete modelName;
	delete busName;
	delete subsystemName;
	delete groupName;
	delete optionsForFileSystems;
	delete oneMoreDeviceName;
	delete ownerName;
		
	delete mountPointWidget;
	delete createAndRemoveMountPoint;
		
	delete treeWidget;
	delete rulesFilter;

	delete closeCancelButton;
	delete addRuleButton;
	delete saveAsRuleButton;
	delete editRuleButton;
	
	delete deviceDefinitionBox;
	delete autoMountSettingsBox;
	delete otherRuleSettingsBox;
	delete splitter;
}

void UsbManager::addFileSystem(const QString &fileSystem,const QString &opt) {

	UsbManagerFsOptionsDialog *newDialog = new UsbManagerFsOptionsDialog(this);
	
	if (fileSystem.isEmpty()) {
		newDialog->exec();
		if (newDialog->fileSystem().isEmpty() || newDialog->options().isEmpty()) {
			delete newDialog;
			return;
		}
	} else
		newDialog->setFileSystem(fileSystem);
	if (!opt.isEmpty())
		newDialog->setOptions(opt);
	
	optionsForFileSystems->insertRow(optionsForFileSystems->rowCount());

	UsbManagerFsCellWidget *cellWidget = new UsbManagerFsCellWidget(cells.count());
	cellWidget->setFileSystem(newDialog->fileSystem());
	cellWidget->setOptions(newDialog->options());
	if (isCreatingDeviceRuleNow)
		cellWidget->setIdentifier(currentDevice->udi());
	else
		cellWidget->setIdentifier("unknown");
	cellWidget->setRow(optionsForFileSystems->rowCount()-1);
	
	connect(newDialog,SIGNAL(fileSystemChanged(const QString&)),cellWidget,SLOT(setFileSystem(const QString&)));
	connect(newDialog,SIGNAL(optionsChanged(const QString&)),cellWidget,SLOT(setOptions(const QString&)));
	connect(cellWidget,SIGNAL(editFsOptionsSignal()),newDialog,SLOT(exec()));
	
	optionsForFileSystems->setCellWidget(optionsForFileSystems->rowCount()-1,0,cellWidget);
	dialogs.append(newDialog);
	cells.append(cellWidget);
}

void UsbManager::removeFileSystem() {
	optionsForFileSystems->hideRow(optionsForFileSystems->currentRow());
	if (optionsForFileSystems->currentRow() == -1)
		removeFileSystemButton->setEnabled(false);
}

void UsbManager::setDeviceAutomount(DiskDevice *device,VolumeWidget *volumeWidget) {
	init();
	isCreatingDeviceRuleNow = true;
	currentDevice = device;
	closeCancelButton->setText(tr("Cancel"));
	vendorName->setText(device->vendor());
	modelName->setText(device->model());
	busName->setText(device->bus());
	subsystemName->setText("block");
	bool cellWasFound = false;
	foreach (UsbManagerFsCellWidget *cell,cells) 
		if (cell->identifier() == device->udi()) {
			optionsForFileSystems->setRowHidden(cell->row(),false);
			if (cell->fileSystem() == device->fileSystem()) {
				cell->setOptions(volumeWidget->options());
				cellWasFound = true;
			}
			if (cell->fileSystem() != device->fileSystem() && !cellWasFound) {
				addFileSystem(device->fileSystem(),volumeWidget->options());
				cellWasFound = true;
			}
		}
	if (!cellWasFound)
		addFileSystem(device->fileSystem(),volumeWidget->options());
	mountPointWidget->setMountPoint(volumeWidget->mountPoint());
	createAndRemoveMountPoint->setChecked(true);
	exec();
}

void UsbManager::addDeviceAutomountRule(DiskDevice *device,VolumeWidget *volumeWidget) {
	init();
	vendorName->setText(device->vendor());
	modelName->setText(device->model());
	busName->setText(device->bus());
	subsystemName->setText("block");
	bool cellWasFound = false;
	foreach (UsbManagerFsCellWidget *cell,cells)
			if (cell->identifier() == device->udi()) {
				optionsForFileSystems->setRowHidden(cell->row(),false);
			if (cell->fileSystem() == device->fileSystem()) {
				cell->setOptions(volumeWidget->options());
				cellWasFound = true;
			}
			if (cell->fileSystem() != device->fileSystem() && !cellWasFound) {
				addFileSystem(device->fileSystem(),volumeWidget->options());
				cellWasFound = true;
			}
		}
	if (!cellWasFound)
		addFileSystem(device->fileSystem(),volumeWidget->options());
	mountPointWidget->setMountPoint(volumeWidget->mountPoint());
	createAndRemoveMountPoint->setChecked(true);
	addRule();
	init();
}

void UsbManager::currentFileSystemChanged(int /*row*/,int /*column*/) {
	removeFileSystemButton->setEnabled(true);
}

void UsbManager::closeEvent(QCloseEvent *event) {
	if (isCreatingDeviceRuleNow) {
		closeCancelButton->setText(tr("Close"));
		init();
		currentDevice = 0;
		isCreatingDeviceRuleNow = false;
		event->accept();
	}
	// If user edit some rule
	if (editRuleButton->isVisible()) {
		QMessageBox *messageBox = new QMessageBox(this);
		messageBox->setText(tr("Do you really want to abort editing of the rule?"));
		messageBox->setIcon(QMessageBox::Question);
		messageBox->setWindowTitle(tr("Attention"));
		messageBox->addButton(QMessageBox::Yes);
		QPushButton *noButton = messageBox->addButton(QMessageBox::No);
		messageBox->exec();
		if (messageBox->clickedButton() == noButton) {
			event->ignore();
			return;
		}
		delete messageBox;
		init();
		editRuleButton->setVisible(false);
		event->accept();
	}
}

void UsbManager::addRule() {
	QString ruleString = rule();
	if (ruleString.isEmpty())
		return;
	QTreeWidgetItem *item = new QTreeWidgetItem(treeWidget);
	item->setText(0,deviceName->text());
	item->setText(1,vendorName->text());
	item->setText(2,modelName->text());
	item->setText(3,busName->text());
	item->setData(4,4,allRules.count());
	allRules.append(ruleString);
	rulesItems.append(item);
	init();
	emit (rulesChanged());
}

QString UsbManager::rule() {
	QString ruleString;
	QList<UsbManagerFsCellWidget *> cellsList;
	foreach (UsbManagerFsCellWidget *cell,cells)
		if (!optionsForFileSystems->isRowHidden(cell->row()))
			cellsList.append(cell);
	if (cellsList.count() == 0) {
		QMessageBox::warning(this,tr("Error"),tr("There are no options for this device"));
		return QString();
	}
	if (mountPointWidget->mountPoint().isEmpty()) {
		QMessageBox::warning(this,tr("Error"),tr("There is no mount point for this device"));
		return QString();
	}
	if (createAndRemoveMountPoint->isChecked())
		ruleString += "#CreateDir#\n" + deviceDefinitionRuleString() + QString("ACTION==\"add\", RUN+=\"/bin/mkdir -p %1\"\n").arg(mountPointWidget->mountPoint());
	foreach (UsbManagerFsCellWidget *cell,cellsList) {
		ruleString += "#Mount#\n";
		ruleString += deviceDefinitionRuleString();
		ruleString += "ACTION==\"add\", ";
		if (groupName->currentIndex() != 0)
			ruleString += "GROUP=\"" + groupName->itemText(groupName->currentIndex()) + "\", ";
		if (ownerName->currentIndex() != 0)
			ruleString += "OWNER=\"" + ownerName->itemText(ownerName->currentIndex()) + "\", ";
		if (!oneMoreDeviceName->text().isEmpty())
			ruleString += "SYMLINK=\"" + oneMoreDeviceName->text() + "\", ";
		ruleString += QString("PROGRAM==\"/lib/udev/vol_id -t %N\", RESULT==\"%1\", ").arg(dialogs[cell->index()]->fileSystem());
		ruleString += QString("RUN+=\"/bin/mount -t %1 -o %2 /dev/%k %3\"\n")
				.arg(dialogs[cell->index()]->fileSystem())
				.arg(dialogs[cell->index()]->options())
				.arg(mountPointWidget->mountPoint());
			
	}
	if (createAndRemoveMountPoint->isChecked()) {
		ruleString += "#UmountDir#\n" + deviceDefinitionRuleString() +
				QString("ACTION==\"remove\", RUN+=\"/bin/umount -l %1\"\n").arg(mountPointWidget->mountPoint());
		ruleString += "#RemoveDir#\n" + deviceDefinitionRuleString() +
				QString("ACTION==\"remove\", RUN+=\"/bin/rmdir %1\"\n").arg(mountPointWidget->mountPoint());
	}
	return ruleString;
}

QString UsbManager::deviceDefinitionRuleString() {
	QString ruleString;
	if (!deviceName->text().isEmpty())
		ruleString += "KERNEL==\"" + deviceName->text() + "\", ";
	if (!vendorName->text().isEmpty())
		ruleString += "SYSFS{vendor}==\"" + vendorName->text() + "\", ";
	if (!modelName->text().isEmpty())
		ruleString += "SYSFS{model}==\"" + modelName->text() + "\", ";
	if (!subsystemName->text().isEmpty())
		ruleString += "SUBSYSTEM==\"" + subsystemName->text() + "\", ";
	if (!busName->text().isEmpty())
		ruleString += "BUS==\"" + busName->text() + "\", ";
	return ruleString;
}

void UsbManager::rulesChangedSlot() {
	QFile udevFile("/etc/udev/rules.d/01-mountmanager.rules");
	if (!udevFile.open(QIODevice::WriteOnly)) {
		udevFile.setFileName(QDir::homePath() + "/01-mountmanager.rules");
		udevFile.open(QIODevice::WriteOnly);
		QMessageBox::warning(this,tr("Error"),tr("Cannot save the rule in this directory, for there are no rights for it. The rule is saved in your home directory: %1. Copy it to /etc/udev/rules.d/ by yourself.").arg(QDir::homePath() + "/01-mountmanager.rules"));
	}
	QTextStream udevStream(&udevFile);
	QString currentUdevText;
	foreach (QString rule,allRules)
		if (!rule.isEmpty())
			udevStream << rule << "#Dont_remove_this_row!#\n";
}

void UsbManager::createTreeWidgetItems() {
	int index = 0;
	foreach (QString rule,allRules) {
		if (rule.isEmpty())
			continue;
		QStringList ruleStrings = rule.split("\n");
		QString deviceNameString;
		QString vendorNameString;
		QString modelNameString;
		QString busNameString;
		foreach (QString str,ruleStrings) {
			QStringList optionsList = str.split(",");
			foreach (QString option,optionsList) {
				if (option.contains("KERNEL") && deviceNameString.isEmpty())
					deviceNameString = option.replace("KERNEL==","").replace("\"","");
				else if (option.contains("SYSFS{vendor}") && vendorNameString.isEmpty())
					vendorNameString = option.replace("SYSFS{vendor}==","").replace("\"","");
				else if (option.contains("SYSFS{model}") && modelNameString.isEmpty())
					modelNameString = option.replace("SYSFS{model}==","").replace("\"","");
				else if (option.contains("BUS") && busNameString.isEmpty())
					busNameString = option.replace("BUS==","").replace("\"","");
			}
		}
		QTreeWidgetItem *item = new QTreeWidgetItem(treeWidget);
		item->setText(0,deviceNameString);
		item->setText(1,vendorNameString);
		item->setText(2,modelNameString);
		item->setText(3,busNameString);
		item->setData(4,4,index);
		treeWidget->resizeColumnToContents(0);
		treeWidget->resizeColumnToContents(1);
		treeWidget->resizeColumnToContents(2);
		treeWidget->resizeColumnToContents(3);
		index++;
		rulesItems.append(item);
	}
}

void UsbManager::removeRule() {
	QTreeWidgetItem *currentItem = treeWidget->currentItem();
	allRules.removeAt(currentItem->data(4,4).toInt());
	currentItem->setHidden(true);
	removedRulesItems.append(currentItem);
	emit (rulesChanged());
}

void UsbManager::fillRuleParts() {
	if (isCreatingDeviceRuleNow) {
		QMessageBox *messageBox = new QMessageBox(this);
		messageBox->setText(tr("Do you want to abort setting of device auto-mounting?"));
		messageBox->setIcon(QMessageBox::Question);
		messageBox->setWindowTitle(tr("Attention"));
		messageBox->addButton(QMessageBox::Yes);
		QPushButton *noButton = messageBox->addButton(QMessageBox::No);
		messageBox->exec();
		if (messageBox->clickedButton() == noButton)
			return;
		isCreatingDeviceRuleNow = false;
		delete messageBox;
	}
	QString currentRule = allRules[treeWidget->currentItem()->data(4,4).toInt()];
	init();
	bool isMountLineNow = false;
	bool allCompletedExceptFs = false;
	QString fs; // file system
	foreach (QString str,currentRule.split('\n')) {
		if (str.contains("CreateDir") || str.contains("UmountDir") || str.contains("RemoveDir"))
			createAndRemoveMountPoint->setChecked(true);
		if (str.contains("Mount")) {
			isMountLineNow = true;
			continue;
		}
		if (isMountLineNow) {
			foreach (QString option,str.split(',')) {
				if (!allCompletedExceptFs) {
					if (option.contains("KERNEL"))
						deviceName->setText(option.replace("KERNEL==","").replace("\"","").replace(" ",""));
					else if (option.contains("SYSFS{vendor}"))
						vendorName->setText(option.replace("SYSFS{vendor}==","").replace("\"","").replace(" ",""));
					else if (option.contains("SYSFS{model}"))
						modelName->setText(option.replace("SYSFS{model}==","").replace("\"","").replace(" ",""));
					else if (option.contains("BUS"))
						busName->setText(option.replace("BUS==","").replace("\"","").replace(" ",""));
					else if (option.contains("SUBSYSTEM"))
						subsystemName->setText(option.replace("SUBSYSTEM==","").replace("\"","").replace(" ",""));
					else if (option.contains("GROUP")) {
						QString group = option.replace("GROUP=","").replace("\"","").replace(" ","");
						for (int i = 0; i < groupName->count(); i++) {
							if (groupName->itemText(i) == group) {
								groupName->setCurrentIndex(i);
								break;
							}
						}
					} else if (option.contains("OWNER")) {
						QString owner = option.replace("OWNER=","").replace("\"","").replace(" ","");
						for (int i = 0; i < ownerName->count(); i++)
							if (ownerName->itemText(i) == owner) {
								ownerName->setCurrentIndex(i);
								break;
							}
					} else if (option.contains("SYMLINK"))
						oneMoreDeviceName->setText(option.replace("SYMLINK=","").replace("\"","").replace(" ",""));
				}
				if (option.contains("RESULT")) {
					allCompletedExceptFs = true;
					fs = option.replace("RESULT==","").replace("\"","").replace(" ","");
				} else if (option.contains("RUN")) {
					QStringList list = option.replace("RUN+=","").replace("\"","").split(" ");
					list.removeAll("");
					list.removeAll(" ");
					if (list.count() == 7) {
						mountPointWidget->setMountPoint(list[6]);
						addFileSystem(fs,list[4]);
					}
				}
			}
			isMountLineNow = false;
		}
	}
	editRuleButton->setVisible(true);
	addRuleButton->setVisible(false);
	closeCancelButton->setText(tr("Cancel"));
}

void UsbManager::init() {
	deviceName->clear();
	vendorName->clear();
	modelName->clear();
	busName->clear();
	subsystemName->clear();
	mountPointWidget->setMountPoint("");
	createAndRemoveMountPoint->setChecked(false);
	foreach (UsbManagerFsCellWidget *cell,cells)
		optionsForFileSystems->hideRow(cell->row());
	groupName->setCurrentIndex(0);
	ownerName->setCurrentIndex(0);
	addRuleButton->setVisible(true);
	isCreatingDeviceRuleNow = false;
}

void UsbManager::editRule() {
	QTreeWidgetItem *currentItem = treeWidget->currentItem();
	QString ruleString = rule();
	if (ruleString.isEmpty())
		return;
	allRules.replace(currentItem->data(4,4).toInt(),rule());
	currentItem->setText(0,deviceName->text());
	currentItem->setText(1,vendorName->text());
	currentItem->setText(2,modelName->text());
	currentItem->setText(3,busName->text());
	init();
	editRuleButton->setVisible(false);
	emit (rulesChanged());
}

void UsbManager::setFilter(const QString& filter) {
	foreach (QTreeWidgetItem *item,rulesItems) {
		if (removedRulesItems.contains(item))
			continue;
		if (filter.isEmpty()) {
			item->setHidden(false);
			continue;
		}
		bool toHide = true;
		for (int i = 0; i < 4; i++)
			if (item->text(i).contains(filter,Qt::CaseInsensitive)) {
				toHide = false;
				break;
			}
		item->setHidden(toHide);
	}
}

void UsbManager::saveRuleAs() {
	QString ruleString = rule();
	if (ruleString.isEmpty())
		return;
	QString filePath = QFileDialog::getSaveFileName(this,tr("Save rule as"),"/etc/udev/rules.d/");
	if (filePath.isEmpty())
		return;
	QFile file(filePath);
	QTextStream stream(&file);
	if (file.open(QIODevice::WriteOnly))
		stream << ruleString;
	else {
		file.setFileName(QDir::homePath() + "/02-mountmanager.rules");
		file.open(QIODevice::WriteOnly);
		stream << ruleString;
		QMessageBox::warning(this,tr("Error"),tr("Cannot save the rule in this directory, for there are no rights for it. The rule was saved in your home directory: %1. Copy it to /etc/udev/rules.d/ by yourself.").arg(QDir::homePath() + "/02-mountmanager.rules"));
	}
}
